/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkDirect3DRenderer.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkDirect3DRenderer
 * @brief   Direct3D renderer
 *
 * vtkDirect3DRenderer is a concrete implementation of the abstract class
 * vtkRenderer. vtkDirect3DRenderer interfaces to the Direct3D graphics library.
*/

#ifndef vtkDirect3DRenderer_h
#define vtkDirect3DRenderer_h

#include "vtkRenderingDirect3DModule.h" // For export macro
#include "vtkRenderer.h"
#include <vector>  // STL Header
#include <windows.h>  // used for ivars
#include <d3d11_1.h>  // used for ivars

class vtkRenderPass;
class vtkDirect3DTexture;
class vtkTextureObject;

class VTKRENDERINGDIRECT3D_EXPORT vtkDirect3DRenderer : public vtkRenderer
{
public:
  static vtkDirect3DRenderer *New();
  vtkTypeMacro(vtkDirect3DRenderer, vtkRenderer);
  void PrintSelf(ostream& os, vtkIndent indent);

  /**
   * Concrete open gl render method.
   */
  void DeviceRender(void);

  /**
   * Overridden to support hidden line removal.
   */
  virtual void DeviceRenderOpaqueGeometry();

  /**
   * Render translucent polygonal geometry. Default implementation just call
   * UpdateTranslucentPolygonalGeometry().
   * Subclasses of vtkRenderer that can deal with depth peeling must
   * override this method.
   */
  virtual void DeviceRenderTranslucentPolygonalGeometry();

  void Clear(void);

  /**
   * Ask lights to load themselves into graphics pipeline.
   */
  int UpdateLights(void);

  /**
   * Is rendering at translucent geometry stage using depth peeling and
   * rendering a layer other than the first one? (Boolean value)
   * If so, the uniform variables UseTexture and Texture can be set.
   * (Used by vtkDirect3DProperty or vtkDirect3DTexture)
   */
  int GetDepthPeelingHigherLayer();

protected:
  vtkDirect3DRenderer();
  ~vtkDirect3DRenderer();

  /**
   * Check the compilation status of some fragment shader source.
   */
  void CheckCompilation(unsigned int fragmentShader);

  // Internal method to release graphics resources in any derived renderers.
  virtual void ReleaseGraphicsResources(vtkWindow *w);

  /**
   * Ask all props to update and draw any opaque and translucent
   * geometry. This includes both vtkActors and vtkVolumes
   * Returns the number of props that rendered geometry.
   */
  virtual int UpdateGeometry();

  // Picking functions to be implemented by sub-classes
  virtual void DevicePickRender();
  virtual void StartPick(unsigned int pickFromSize);
  virtual void UpdatePickId();
  virtual void DonePick();
  virtual unsigned int GetPickedId();
  virtual unsigned int GetNumPickedIds();
  virtual int GetPickedIds(unsigned int atMost, unsigned int *callerBuffer);
  virtual double GetPickedZ();

  double PickedZ;

  friend class vtkDirect3DProperty;
  friend class vtkDirect3DTexture;
  friend class vtkDirect3DImageSliceMapper;
  friend class vtkDirect3DImageResliceMapper;

  // Is rendering at translucent geometry stage using depth peeling and
  // rendering a layer other than the first one? (Boolean value)
  // If so, the uniform variables UseTexture and Texture can be set.
  // (Used by vtkDirect3DProperty or vtkDirect3DTexture)
  int DepthPeelingHigherLayer;

  friend class vtkRenderPass;

  ID3D11DeviceContext*    ImmediateContext;
  ID3D11RenderTargetView* RenderTargetView;
  ID3D11DepthStencilView* DepthStencilView;

private:
  vtkDirect3DRenderer(const vtkDirect3DRenderer&) VTK_DELETE_FUNCTION;
  void operator=(const vtkDirect3DRenderer&) VTK_DELETE_FUNCTION;
};

#endif
