/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkDirect3DRenderWindow.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkDirect3DRenderWindow
 * @brief   Direct3D rendering window
 *
 * vtkDirect3DRenderWindow is a concrete implementation of the abstract class
 * vtkRenderWindow. vtkDirect3DRenderer interfaces to the Direct3D graphics
 * library. Application programmers should normally use vtkRenderWindow
 * instead of the Direct3D specific version.
*/

#ifndef vtkDirect3DRenderWindow_h
#define vtkDirect3DRenderWindow_h

#include "vtkRenderingDirect3DModule.h" // For export macro
#include "vtkRenderWindow.h"

#include <windows.h>  // used for ivars
#include <d3d11_1.h>  // used for ivars

#include <string> // for ivar
#include <map> // for ivar
#include <set> // for ivar
#include "vtkType.h" // for ivar

class vtkIdList;
class vtkDirect3DHardwareSupport;
class vtkDirect3DShaderCache;
class vtkDirect3DVertexArrayObject;
class vtkShaderProgram;
class vtkStdString;
class vtkTexture;
class vtkTextureObject;
class vtkTextureUnitManager;
class vtkGenericDirect3DResourceFreeCallback;

class VTKRENDERINGDIRECT3D_EXPORT vtkDirect3DRenderWindow : public vtkRenderWindow
{
public:
  static vtkDirect3DRenderWindow *New();
  vtkTypeMacro(vtkDirect3DRenderWindow, vtkRenderWindow);
  void PrintSelf(ostream& os, vtkIndent indent);

  /**
   * Begin the rendering process.
   */
  virtual void Start(void);

  /**
   * End the rendering process and display the image.
   */
  void Frame(void);

  /**
   * Initialize the rendering window.  This will setup all system-specific
   * resources.  This method and Finalize() must be symmetric and it
   * should be possible to call them multiple times, even changing WindowId
   * in-between.  This is what WindowRemap does.
   */
  virtual void Initialize(void);

  /**
   * Finalize the rendering window.  This will shutdown all system-specific
   * resources.  After having called this, it should be possible to destroy
   * a window that was used for a SetWindowId() call without any ill effects.
   */
  virtual void Finalize(void);

  /**
   * What rendering backend has the user requested
   */
  virtual const char *GetRenderingBackend();

  //@{
  /**
   * Set/Get the maximum number of multisamples
   */
  static void SetGlobalMaximumNumberOfMultiSamples(int val);
  static int  GetGlobalMaximumNumberOfMultiSamples();
  //@}

  /**
   * Update system if needed due to stereo rendering.
   */
  virtual void StereoUpdate();

  //@{
  /**
   * Set/Get the pixel data of an image, transmitted as RGBRGB...
   */
  virtual unsigned char *GetPixelData(int x,int y,int x2,int y2,int front);
  virtual int GetPixelData(int x,int y,int x2,int y2, int front,
                           vtkUnsignedCharArray *data);
  virtual int SetPixelData(int x,int y,int x2,int y2,unsigned char *data,
                           int front);
  virtual int SetPixelData(int x,int y,int x2,int y2,
                           vtkUnsignedCharArray *data, int front);
  //@}

  //@{
  /**
   * Set/Get the pixel data of an image, transmitted as RGBARGBA...
   */
  virtual float *GetRGBAPixelData(int x,int y,int x2,int y2,int front);
  virtual int GetRGBAPixelData(int x,int y,int x2,int y2, int front,
                               vtkFloatArray* data);
  virtual int SetRGBAPixelData(int x,int y,int x2,int y2, float *data,
                               int front, int blend=0);
  virtual int SetRGBAPixelData(int x,int y,int x2,int y2, vtkFloatArray *data,
                               int front, int blend=0);
  virtual void ReleaseRGBAPixelData(float *data);
  virtual unsigned char *GetRGBACharPixelData(int x,int y,int x2,int y2,
                                              int front);
  virtual int GetRGBACharPixelData(int x,int y,int x2,int y2, int front,
                                   vtkUnsignedCharArray *data);
  virtual int SetRGBACharPixelData(int x, int y, int x2, int y2,
                                   unsigned char *data, int front,
                                   int blend=0);
  virtual int SetRGBACharPixelData(int x,int y,int x2,int y2,
                                   vtkUnsignedCharArray *data, int front,
                                   int blend=0);
  //@}

  //@{
  /**
   * Set/Get the zbuffer data from an image
   */
  virtual float *GetZbufferData( int x1, int y1, int x2, int y2 );
  virtual int GetZbufferData( int x1, int y1, int x2, int y2, float* z );
  virtual int GetZbufferData( int x1, int y1, int x2, int y2,
                              vtkFloatArray* z );
  virtual int SetZbufferData( int x1, int y1, int x2, int y2, float *buffer );
  virtual int SetZbufferData( int x1, int y1, int x2, int y2,
                              vtkFloatArray *buffer );
  //@}


  /**
   * Get the size of the depth buffer.
   */
  int GetDepthBufferSize() { return 8;};

  /**
   * Get the size of the color buffer.
   * Returns 0 if not able to determine otherwise sets R G B and A into buffer.
   */
  int GetColorBufferSizes(int *rgba) {
    rgba[0] = 8; rgba[1] = 8; rgba[2] = 8; rgba[3] = 8; return 1;};

  //@{
  /**
   * Set the size of the window in screen coordinates in pixels.
   */
  virtual void SetSize(int a[2]);
  virtual void SetSize(int,int);
  //@}

  /**
   * Return the Direct3D name of the back left buffer.
   * It is GL_BACK_LEFT if GL is bound to the window-system-provided
   * framebuffer. It is vtkgl::COLOR_ATTACHMENT0_EXT if GL is bound to an
   * application-created framebuffer object (GPU-based offscreen rendering)
   * It is used by vtkDirect3DCamera.
   */
  unsigned int GetBackLeftBuffer();

  /**
   * Return the Direct3D name of the back right buffer.
   * It is GL_BACK_RIGHT if GL is bound to the window-system-provided
   * framebuffer. It is vtkgl::COLOR_ATTACHMENT0_EXT+1 if GL is bound to an
   * application-created framebuffer object (GPU-based offscreen rendering)
   * It is used by vtkDirect3DCamera.
   */
  unsigned int GetBackRightBuffer();

  /**
   * Return the Direct3D name of the front left buffer.
   * It is GL_FRONT_LEFT if GL is bound to the window-system-provided
   * framebuffer. It is vtkgl::COLOR_ATTACHMENT0_EXT if GL is bound to an
   * application-created framebuffer object (GPU-based offscreen rendering)
   * It is used by vtkDirect3DCamera.
   */
  unsigned int GetFrontLeftBuffer();

  /**
   * Return the Direct3D name of the front right buffer.
   * It is GL_FRONT_RIGHT if GL is bound to the window-system-provided
   * framebuffer. It is vtkgl::COLOR_ATTACHMENT0_EXT+1 if GL is bound to an
   * application-created framebuffer object (GPU-based offscreen rendering)
   * It is used by vtkDirect3DCamera.
   */
  unsigned int GetFrontRightBuffer();

  /**
   * Return the Direct3D name of the back left buffer.
   * It is GL_BACK if GL is bound to the window-system-provided
   * framebuffer. It is vtkgl::COLOR_ATTACHMENT0_EXT if GL is bound to an
   * application-created framebuffer object (GPU-based offscreen rendering)
   * It is used by vtkDirect3DCamera.
   */
  unsigned int GetBackBuffer();

  /**
   * Return the Direct3D name of the front left buffer.
   * It is GL_FRONT if GL is bound to the window-system-provided
   * framebuffer. It is vtkgl::COLOR_ATTACHMENT0_EXT if GL is bound to an
   * application-created framebuffer object (GPU-based offscreen rendering)
   * It is used by vtkDirect3DCamera.
   */
  unsigned int GetFrontBuffer();

  /**
   * Get the time when the Direct3D context was created.
   */
  virtual vtkMTimeType GetContextCreationTime();

  //@{
  /**
   * Returns an Shader Cache object
   */
  vtkGetObjectMacro(ShaderCache,vtkDirect3DShaderCache);
  //@}

  //@{
  /**
   * Returns the current default FBO (0 when OffScreenRendering is inactive).
   */
  vtkGetMacro(FrameBufferObject, unsigned int);
  //@}

  /**
   * Replacement for the old glDrawPixels function
   */
  virtual void DrawPixels(int x1, int y1, int x2, int y2,
              int numComponents, int dataType, void *data);

  /**
   * Replacement for the old glDrawPixels function, but it allows
   * for scaling the data and using only part of the texture
   */
  virtual void DrawPixels(
    int dstXmin, int dstYmin, int dstXmax, int dstYmax,
    int srcXmin, int srcYmin, int srcXmax, int srcYmax,
    int srcWidth, int srcHeight, int numComponents, int dataType, void *data);

  /**
   * Replacement for the old glDrawPixels function.  This simple version draws all
   * the data to the entire current viewport scaling as needed.
   */
  virtual void DrawPixels(
    int srcWidth, int srcHeight, int numComponents, int dataType, void *data);

  /**
   * Return the largest line width supported by the hardware
   */
  virtual float GetMaximumHardwareLineWidth() {
    return this->MaximumHardwareLineWidth; };

  ID3D11DeviceContext*    GetImmediateContext() {
    return this->ImmediateContext; }
  ID3D11RenderTargetView* GetRenderTargetView() {
    return this->RenderTargetView; }
  ID3D11Device* GetD3DDevice() {
    return this->D3dDevice; }
  ID3D11DepthStencilView* GetDepthStencilView() {
    return this->DepthStencilView; }


  // implement required virtual functions
  void SetWindowInfo(char *) {};
  void SetNextWindowInfo(char *) {};
  void SetParentInfo(char *) {};
  virtual void *GetGenericDisplayId() {return (void *)this->D3dDevice;};
  virtual void *GetGenericWindowId()  {return (void *)this->WindowId;};
  virtual void *GetGenericParentId()  {return (void *)NULL;};
  virtual void *GetGenericContext()   {return (void *)this->ImmediateContext;};
  virtual void *GetGenericDrawable()  {return (void *)this->WindowId;};
  virtual void SetDisplayId(void *) {};
  void  SetWindowId(void *) {};
  void  SetParentId(void *) {};
  void HideCursor() {};
  void ShowCursor() {};
  virtual void SetFullScreen(int) {};
  virtual void WindowRemap(void) {};
  virtual void SetNextWindowId(void *) {};

  /**
   * Get the current size of the screen in pixels.
   */
  virtual int *GetScreenSize();

  void MakeCurrent() {};
  void WaitForCompletion() {};
  int GetEventPending();
  bool IsCurrent() { return true;}

  /**
   * Get the window id.
   */
  virtual HWND  GetWindowId() {
    return this->WindowId; }

protected:
  vtkDirect3DRenderWindow();
  ~vtkDirect3DRenderWindow();

  vtkDirect3DShaderCache *ShaderCache;

  // message handler
  virtual LRESULT MessageProc(HWND hWnd, UINT message,
                              WPARAM wParam, LPARAM lParam);

  static LRESULT APIENTRY WndProc(HWND hWnd, UINT message,
    WPARAM wParam, LPARAM lParam);

  // used in testing for Direct3D support
  // in the SupportsDirect3D() method
  bool Direct3DSupportTested;
  int Direct3DSupportResult;
  std::string Direct3DSupportMessage;

  int TextureInternalFormats[VTK_UNICODE_STRING][3][5];
  void InitializeTextureInternalFormats();

  std::map<const vtkTextureObject *, int> TextureResourceIds;

  virtual int GetPixelData(int x, int y, int x2, int y2, int front, unsigned char* data);
  int GetRGBAPixelData(int x, int y, int x2, int y2, int front, float* data);
  int GetRGBACharPixelData(int x, int y, int x2, int y2, int front,
                           unsigned char* data);

  /**
   * Create an offScreen window based on Direct3D framebuffer extension.
   * Return if the creation was successful or not.
   * \pre positive_width: width>0
   * \pre positive_height: height>0
   * \pre not_initialized: !OffScreenUseFrameBuffer
   * \post valid_result: (result==0 || result==1)
   * && (result implies OffScreenUseFrameBuffer)
   */
  int CreateHardwareOffScreenWindow(int width, int height);

  int CreateHardwareOffScreenBuffers(int width, int height, bool bind = false);
  void BindHardwareOffScreenBuffers();

  /**
   * Destroy an offscreen window based on Direct3D framebuffer extension.
   * \pre initialized: OffScreenUseFrameBuffer
   * \post destroyed: !OffScreenUseFrameBuffer
   */
  void DestroyHardwareOffScreenWindow();

  void UnbindHardwareOffScreenBuffers();
  void DestroyHardwareOffScreenBuffers();

  /**
   * Flag telling if a framebuffer-based offscreen is currently in use.
   */
  int OffScreenUseFrameBuffer;

  //@{
  /**
   * Variables used by the framebuffer-based offscreen method.
   */
  int NumberOfFrameBuffers;
  unsigned int TextureObjects[4]; // really GLuint
  unsigned int FrameBufferObject; // really GLuint
  unsigned int DepthRenderBufferObject; // really GLuint
  int HardwareBufferSize[2];
  bool HardwareOffScreenBuffersBind;
  //@}

  /**
   * Create a not-off-screen window.
   */
  virtual void CreateAWindow();

  /**
   * Destroy a not-off-screen window.
   */
  virtual void DestroyWindow();

  /**
   * Free up any graphics resources associated with this window
   * a value of NULL means the context may already be destroyed
   */
  virtual void ReleaseGraphicsResources(vtkRenderWindow *);

  /**
   * Set the texture unit manager.
   */
  void SetTextureUnitManager(vtkTextureUnitManager *textureUnitManager);


  /**
   * Query and save Direct3D state
   */
  void SaveGLState();

  /**
   * Restore Direct3D state at end of the rendering
   */
  void RestoreGLState();

  std::map<std::string, int> GLStateIntegers;

  unsigned int BackLeftBuffer;
  unsigned int BackRightBuffer;
  unsigned int FrontLeftBuffer;
  unsigned int FrontRightBuffer;
  unsigned int FrontBuffer;
  unsigned int BackBuffer;

  #ifndef VTK_LEGACY_REMOVE
  /**
   * @deprecated Replaced by
   * vtkDirect3DCheckErrorMacro
   */
  unsigned int LastGraphicError;
  #endif

  /**
   * Flag telling if the context has been created here or was inherited.
   */
  int OwnContext;

  vtkTimeStamp ContextCreationTime;

  vtkTextureUnitManager *TextureUnitManager;

  vtkTextureObject *DrawPixelsTextureObject;

  bool Initialized; // ensure glewinit has been called

  float MaximumHardwareLineWidth;
  HINSTANCE ApplicationInstance;
  HWND      WindowId;
  D3D_DRIVER_TYPE DriverType;
  D3D_FEATURE_LEVEL       FeatureLevel = D3D_FEATURE_LEVEL_11_0;
  ID3D11Device*           D3dDevice;
  ID3D11Device1*          D3dDevice1;
  ID3D11DeviceContext*    ImmediateContext;
  ID3D11DeviceContext1*   ImmediateContext1;
  IDXGISwapChain*         SwapChain;
  IDXGISwapChain1*        SwapChain1;
  ID3D11RenderTargetView* RenderTargetView;
  ID3D11RasterizerState* RasterizerState;
  ID3D11Texture2D*        DepthStencil;
  ID3D11DepthStencilView* DepthStencilView;

  HRESULT InitDevice();

private:
  vtkDirect3DRenderWindow(const vtkDirect3DRenderWindow&) VTK_DELETE_FUNCTION;
  void operator=(const vtkDirect3DRenderWindow&) VTK_DELETE_FUNCTION;
};

#endif
