/*=========================================================================

  Program:   Visualization Toolkit

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#ifndef vtkDirect3DIndexBufferObject_h
#define vtkDirect3DIndexBufferObject_h

#include "vtkRenderingDirect3DModule.h" // for export macro
#include "vtkDirect3DBufferObject.h"


/**
 * @brief Direct3D vertex buffer object
 *
 * Direct3D buffer object to store geometry and/or attribute data on the
 * GPU.
 */

class VTKRENDERINGDIRECT3D_EXPORT vtkDirect3DIndexBufferObject :
  public vtkDirect3DBufferObject
{
public:
  static vtkDirect3DIndexBufferObject *New();
  vtkTypeMacro(vtkDirect3DIndexBufferObject, vtkDirect3DBufferObject);
  void PrintSelf(ostream& os, vtkIndent indent);

  /**
   * Bind the buffer object ready for rendering.
   * @note Only one ARRAY_BUFFER and one ELEMENT_ARRAY_BUFFER may be bound at
   * any time.
   */
  virtual bool Bind();

  // Sizes/offsets are all in bytes as Direct3D API expects them.
  size_t IndexCount; // Number of indices in the VBO

  // Description:
  // used to create an IBO for triangle primatives
  size_t CreateTriangleIndexBuffer(vtkCellArray *cells,
     vtkPoints *points);

  // Description:
  // used to create an IBO for triangle primatives
  static void AppendTriangleIndexBuffer(
    std::vector<unsigned int> &indexArray,
    vtkCellArray *cells,
    vtkPoints *points,
    vtkIdType vertexOffset);

  // Description:
  // create a IBO for wireframe polys/tris
  size_t CreateTriangleLineIndexBuffer(vtkCellArray *cells);

  // Description:
  // used to create an IBO for line primatives
  static void AppendLineIndexBuffer(
    std::vector<unsigned int> &indexArray,
    vtkCellArray *cells,
    vtkIdType vertexOffset);

  // Description:
  // create a IBO for wireframe polys/tris
  size_t CreateLineIndexBuffer(vtkCellArray *cells);

  // Description:
  // create a IBO for wireframe polys/tris
  static void AppendTriangleLineIndexBuffer(
    std::vector<unsigned int> &indexArray,
    vtkCellArray *cells,
    vtkIdType vertexOffset);

  // Description:
  // used to create an IBO for primatives as points
  size_t CreatePointIndexBuffer(vtkCellArray *cells);

  // Description:
  // used to create an IBO for primatives as points
  static void AppendPointIndexBuffer(
    std::vector<unsigned int> &indexArray,
    vtkCellArray *cells,
    vtkIdType vertexOffset);

  // Description:
  // used to create an IBO for line strips and triangle strips
  size_t CreateStripIndexBuffer(
    vtkCellArray *cells, bool wireframeTriStrips);

  static void AppendStripIndexBuffer(
    std::vector<unsigned int> &indexArray,
    vtkCellArray *cells,
    vtkIdType vertexOffset,  bool wireframeTriStrips);

  // Description:
  // special index buffer for polys wireframe with edge visibilityflags
  static void AppendEdgeFlagIndexBuffer(
    std::vector<unsigned int> &indexArray,
    vtkCellArray *cells,
    vtkIdType vertexOffset,  vtkDataArray *edgeflags);

  size_t CreateEdgeFlagIndexBuffer(
    vtkCellArray *cells, vtkDataArray *edgeflags);

  // Create supporting arays that are needed when rendering cell data
  // Some VTK cells have to be broken into smaller cells for Direct3D
  // When we have cell data we have to map cell attributes from the VTK
  // cell number to the actual Direct3D cell
  // The following code fills in
  //
  //   cellCellMap which maps a Direct3D cell id to the VTK cell it came from
  //
  static void CreateCellSupportArrays(
    vtkCellArray *[4],
    std::vector<unsigned int> &cellCellMap,
    int representation,
    vtkPoints *points);

protected:
  vtkDirect3DIndexBufferObject();
  ~vtkDirect3DIndexBufferObject();

  virtual bool UploadInternal(const void *buffer, size_t size);

private:
  vtkDirect3DIndexBufferObject(const vtkDirect3DIndexBufferObject&) VTK_DELETE_FUNCTION;
  void operator=(const vtkDirect3DIndexBufferObject&) VTK_DELETE_FUNCTION;
};

#endif
