/*=========================================================================

  Program:   Visualization Toolkit

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#ifndef vtkDirect3DConstantBufferObject_h
#define vtkDirect3DConstantBufferObject_h

#include "vtkRenderingDirect3DModule.h" // for export macro
#include "vtkDirect3DBufferObject.h"

#include <map>    // For member variables.

class vtkMatrix4x4;
class vtkMatrix3x3;

class VTKRENDERINGDIRECT3D_EXPORT vtkDirect3DConstantBufferObject :
  public vtkDirect3DBufferObject
{
public:
  static vtkDirect3DConstantBufferObject *New();
  vtkTypeMacro(vtkDirect3DConstantBufferObject, vtkDirect3DBufferObject);
  void PrintSelf(ostream& os, vtkIndent indent);

  /**
   * Bind the buffer object ready for rendering.
   * @note Only one ARRAY_BUFFER and one ELEMENT_ARRAY_BUFFER may be bound at
   * any time.
   */
  virtual bool Bind();

  void AddUniform(const char *name, const char *type, size_t size);

  size_t GetOffset(const char *name);

  template <class T>
  void SetUniform(const char *name, const T value);

  template <class T>
  void SetUniformArray(const char *name, const T *value, size_t count);

  void SetUniformMatrix(const char *name, vtkMatrix3x3 *v);
  void SetUniformMatrix(const char *name, vtkMatrix4x4 *v);

  // maps of std::string are super slow when calling find
  // with a string literal or const char * as find
  // forces construction/copy/destruction of a
  // std::sting copy of the const char *
  // In spite of the doubters this can really be a
  // huge CPU hog.
  struct cmp_str
  {
     bool operator()(const char *a, const char *b) const
     {
        return strcmp(a, b) < 0;
     }
  };

  struct UniformData
  {
    std::string Name;
    std::string Type;
    size_t Size; // in bytes
    size_t Offset; // in bytes
  };

  std::string GetDeclaration();

  size_t GetNumberOfConstants() {
    return this->Uniforms.size(); }

protected:
  vtkDirect3DConstantBufferObject();
  ~vtkDirect3DConstantBufferObject();

  // map of name to data
  std::map<const char *, UniformData *, cmp_str> Uniforms;
  std::vector<UniformData *> UniformsInOrder;

  std::vector<unsigned char> PackedBuffer; // the data
  size_t TotalOffset;

private:
  vtkDirect3DConstantBufferObject(const vtkDirect3DConstantBufferObject&) VTK_DELETE_FUNCTION;
  void operator=(const vtkDirect3DConstantBufferObject&) VTK_DELETE_FUNCTION;
};

template <class T>
inline void vtkDirect3DConstantBufferObject::SetUniform(const char *name, const T value)
{
  std::map<const char *, vtkDirect3DConstantBufferObject::UniformData *>::iterator
    found = this->Uniforms.find(name);
  if (found != this->Uniforms.end())
  {
    UniformData *ud = found->second;
    if (sizeof(T) != ud->Size)
      {
      vtkErrorMacro("Bad SetUniform Call");
      return;
      }
    if (ud->Offset + ud->Size > this->PackedBuffer.size())
    {
      this->PackedBuffer.reserve(ud->Offset + ud->Size);
    }
    this->PackedBuffer.insert(this->PackedBuffer.begin() + ud->Offset,
      reinterpret_cast<const unsigned char *>(&value),
      reinterpret_cast<const unsigned char *>(&value) + ud->Size);
  }
}

template <class T>
inline void vtkDirect3DConstantBufferObject::SetUniformArray(
  const char *name, const T *value, size_t count)
{
  std::map<const char *, vtkDirect3DConstantBufferObject::UniformData *>::iterator
    found = this->Uniforms.find(name);
  if (found != this->Uniforms.end())
  {
    UniformData *ud = found->second;
    if (sizeof(T)*count != ud->Size)
      {
      vtkErrorMacro("Bad SetUniform Call");
      return;
      }
    if (ud->Offset + ud->Size > this->PackedBuffer.size())
    {
      this->PackedBuffer.reserve(ud->Offset + ud->Size);
    }
    this->PackedBuffer.insert(this->PackedBuffer.begin() + ud->Offset,
      reinterpret_cast<const unsigned char *>(value),
      reinterpret_cast<const unsigned char *>(value) + ud->Size);
  }
}

#endif
