/*=========================================================================

  Program:   Visualization Toolkit

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkDirect3DConstantBufferObject.h"
#include "vtkObjectFactory.h"

#include "vtkPoints.h"
#include "vtkDirect3DRenderWindow.h"
#include "vtkMatrix4x4.h"
#include "vtkMatrix3x3.h"

#include <sstream>

vtkStandardNewMacro(vtkDirect3DConstantBufferObject)

typedef std::map<const char *, vtkDirect3DConstantBufferObject::UniformData *>::iterator UIter;

vtkDirect3DConstantBufferObject::vtkDirect3DConstantBufferObject()
{
  this->TotalOffset = 0;
}

vtkDirect3DConstantBufferObject::~vtkDirect3DConstantBufferObject()
{
}

bool vtkDirect3DConstantBufferObject::Bind()
{
  // if the buffer is not created then create it
  if (!this->Buffer)
  {
    // make sure we are a multiple of 16 bytes
    this->TotalOffset = this->TotalOffset + (16 - this->TotalOffset%16)%16;
    this->PackedBuffer.reserve(this->TotalOffset);

    this->UploadInternalWithBinding(&(this->PackedBuffer[0]),
      this->TotalOffset,
      D3D11_BIND_CONSTANT_BUFFER);
  }

  this->Context->GetImmediateContext()->UpdateSubresource(
    this->Buffer, 0, nullptr, &(this->PackedBuffer[0]), 0, 0 );

  return true;
}

//-----------------------------------------------------------------------------
void vtkDirect3DConstantBufferObject::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

void vtkDirect3DConstantBufferObject::AddUniform(
  const char *name, const char *type, size_t size)
{
  // does it already exist?
  UIter found = this->Uniforms.find(name);
  if (found == this->Uniforms.end())
  {
    // create one
    vtkDirect3DConstantBufferObject::UniformData *ud = new UniformData;
    ud->Name = name;
    ud->Size = size;
    ud->Type = type;

    // entries that cross a 16 byte boundary must start one a 16 byte boundary
    size_t endingLoc = this->TotalOffset + size - 1;
    if (endingLoc/16 != this->TotalOffset/16)
      {
      this->TotalOffset = this->TotalOffset + (16 - this->TotalOffset%16)%16;
      }

    ud->Offset = this->TotalOffset;
    this->Uniforms.insert(std::make_pair(ud->Name.c_str(), ud));
    this->TotalOffset += size;
    // if total offset is not a 4 byte boundary move it to one
    this->TotalOffset = this->TotalOffset + (4 - this->TotalOffset%4)%4;
    this->UniformsInOrder.push_back(ud);
    return;
  }
}

size_t vtkDirect3DConstantBufferObject::GetOffset(const char *name)
{
  UIter found = this->Uniforms.find(name);
  if (found != this->Uniforms.end())
  {
    return found->second->Offset;
  }
  return 0;
}

void vtkDirect3DConstantBufferObject::SetUniformMatrix(
  const char *name,
  vtkMatrix4x4 *matrix)
{
  float data[16];
  for (int i = 0; i < 16; ++i)
  {
    data[i] = matrix->GetElement(i / 4, i % 4);
  }
  this->SetUniformArray(name, data, 16);
}

void vtkDirect3DConstantBufferObject::SetUniformMatrix(
  const char *name,
  vtkMatrix3x3 *matrix)
{
  float data[9];
  for (int i = 0; i < 9; ++i)
  {
    data[i] = matrix->GetElement(i / 3, i % 3);
  }
  this->SetUniformArray(name, data, 9);
}

std::string vtkDirect3DConstantBufferObject::GetDeclaration()
{
  std::stringstream ss;
  for (size_t i = 0; i < this->UniformsInOrder.size(); i++)
  {
    ss << this->UniformsInOrder[i]->Type << " "
      << this->UniformsInOrder[i]->Name
      << ";\n";
  }

  return ss.str();
}