/*=========================================================================

  Copyright (c) Kitware, Inc.
  All rights reserved.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include <sstream>

#include "mvActor.h"
#include "mvAudioHandler.h"
#include "mvFlagpole.h"
#include "mvLobby.h"
#include "mvLobbyClient.h"
#include "mvPhotoSphere.h"
#include "mvPoseInterpolator.h"
#include "mvTourStop.h"
#include "mvView.h"

#include "vtksys/SystemTools.hxx"

#include "vtkActor.h"
#include "vtkCamera.h"
#include "vtkFFMPEGVideoSource.h"
#include "vtkFlagpoleLabel.h"
#include "vtkGlobFileNames.h"
#include "vtkImageData.h"
#include "vtkImageResize.h"
#include "vtkJPEGReader.h"
#include "vtkLineSource.h"
#include "vtkLookupTable.h"
#include "vtkMath.h"
#include "vtkMultiThreader.h"
#include "vtkMutexLock.h"
#include "vtkOpenGLMovieSphere.h"
#include "vtkOpenGLPolyDataMapper.h"
#include "vtkOpenGLRenderer.h"
#include "vtkOpenGLRenderWindow.h"
#include "vtkOpenGLTexture.h"
#include "vtkOpenVRPanelRepresentation.h"
#include "vtkPlaneSource.h"
#include "vtkPointData.h"
#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkQuadricClustering.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkSkybox.h"
#include "vtkSphereSource.h"
#include "vtkTextActor3D.h"
#include "vtkTextProperty.h"
#include "vtkTextureObject.h"
#include "vtkTimerLog.h"
#include "vtkXMLDataElement.h"
#include "vtkXMLImageDataReader.h"
#include "vtkXMLImageDataWriter.h"
#include "vtkXMLPolyDataReader.h"
#include "vtkXMLPolyDataWriter.h"
#include "vtkXMLUtilities.h"
#include "vtkWindows.h"
#include <mmsystem.h>

#include "vtkOpenVRCamera.h"
#include "vtkOpenVRInteractorStyle.h"
#include "vtkOpenVRModel.h"
#include "vtkOpenVRPanelWidget.h"
#include "vtkOpenVRRenderWindow.h"
#include "vtkOpenVROverlayInternal.h"
#include "vtkOpenVROverlay.h"

namespace
{
const double PRESS_DELAY = 0.350;
}

mvTexture::~mvTexture()
{
  this->OpenGLTexture = nullptr;
  this->TextureObject = nullptr;
}

mvView::mvView()
{
  this->InitialTourStop = 0;
  this->CurrentTourStop = -1;
  this->SphereSource->SetThetaResolution(24);
  this->SphereSource->SetPhiResolution(12);
  this->InPhotoSphere = false;
  this->SavedPose = new vtkOpenVRCameraPose();
  this->NextPose = new vtkOpenVRCameraPose();
  this->CurrentPhotoSphere = nullptr;
  this->InLocalPhotoSphere = false;
  this->Animating = false;
  this->Active = true;
  this->ButtonPressTime = 0;
  this->Latitude = 0.0;
  this->Longitude = 0.0;
}

mvView::~mvView()
{
  this->Flagpoles.clear();
  this->Actors.clear();
  this->PosesXML->Delete();
  delete this->SavedPose;
  delete this->NextPose;
}

vtkActor *mvView::GetPosterActor()
{
  return this->PosterActor;
}

void mvView::LobbyOff()
{
  this->PosterActor->VisibilityOff();
  this->LineActor->VisibilityOff();
  this->MyLobby->GetRenderer()->RemoveActor(this->BackgroundActor);
}

void mvView::LobbyOn()
{
  this->PosterActor->VisibilityOn();
  this->LineActor->VisibilityOn();
}

void mvView::SetPosition(double x1, double y1, double x2, double y2, double height)
{
  this->Plane->SetOrigin(x1, y1, 0.5);
  this->Plane->SetPoint1(x2, y2, 0.5);
  this->Plane->SetPoint2(x1, y1, 0.5 + height);
  this->BackgroundPlane->SetOrigin(x2, y2, 0.5);
  this->BackgroundPlane->SetPoint1(x2 + 0.1*(x2 - x1), y2 + 0.1*(y2 - y1), 0.5);
  this->BackgroundPlane->SetPoint2(x2, y2, 0.5 + height);
  this->Line->SetPoint1((x1 + x2)*0.5, (y1 + y2)*0.5, 0.5);
}

void mvView::LoadAudioClips()
{
  for (auto &ts : this->TourStops)
  {
    std::ostringstream tmps;
      tmps << this->RootDirectory << "/Location" << ts.PoseNumber << ".wav";
    if (ts.AudioFile.size() == 0 &&
        vtksys::SystemTools::FileExists(tmps.str().c_str(),true))
    {
      ts.AudioFile = tmps.str();
    }
  }
}

namespace
{
  bool alessb(mvTourStop &a, mvTourStop &b)
  {
    return (a.PoseNumber < b.PoseNumber);
  }
}

void mvView::LoadExtraData(vtkXMLDataElement *topel)
{
  topel->GetScalarAttribute("Latitude", this->Latitude);
  topel->GetScalarAttribute("Longitude", this->Longitude);

  // get long lat and image
  this->ViewFile = this->RootDirectory;
  this->ViewFile += "/";
  this->ViewFile += topel->GetAttribute("ViewImage");

  // get Photo Spheres
  vtkXMLDataElement *el = topel->FindNestedElementWithName("PhotoSpheres");
  int numSpheres = el ? el->GetNumberOfNestedElements() : 0;
  this->PhotoSpheres.resize(numSpheres);
  for (size_t i = 0; i < numSpheres; i++)
  {
    vtkXMLDataElement *sphereel = el->GetNestedElement(static_cast<int>(i));
    mvPhotoSphere &sphere = this->PhotoSpheres[i];
    sphere.LoadData(sphereel, this->RootDirectory);
  }

  // get Flagpoles
  el = topel->FindNestedElementWithName("Flagpoles");
  int numFlags = el ? el->GetNumberOfNestedElements() : 0;
  this->Flagpoles.resize(numFlags);
  for (size_t i = 0; i < numFlags; i++)
  {
    vtkXMLDataElement *flagel = el->GetNestedElement(static_cast<int>(i));
    mvFlagpole &flagpole = this->Flagpoles[i];
    flagpole.LoadData(flagel);
  }

  // get pose specific data (tourstops)
  el = topel->FindNestedElementWithName("CameraPoses");
  int numPoses = el ? el->GetNumberOfNestedElements() : 0;
  for (size_t i = 0; i < numPoses; i++)
  {
    vtkXMLDataElement *poseel = el->GetNestedElement(static_cast<int>(i));

    float poseNum;
    poseel->GetScalarAttribute("PoseNumber", poseNum);

    bool newTS = true;
    mvTourStop *currTS = nullptr;
    for (auto &ts : this->TourStops)
    {
      if (ts.PoseNumber == poseNum)
      {
        newTS = false;
        currTS = &ts;
        continue;
      }
    }
    if (newTS)
    {
      // insert the pose and resort the vector
      this->TourStops.resize(this->TourStops.size() + 1);
      currTS = &this->TourStops[this->TourStops.size() - 1];
      currTS->PoseNumber = poseNum;

      // resort the vector
      std::sort(this->TourStops.begin(), this->TourStops.end(), alessb);
      for (auto &ts : this->TourStops)
      {
        if (ts.PoseNumber == poseNum)
        {
          currTS = &ts;
          continue;
        }
      }
    }

    currTS->LoadData(poseel, this->RootDirectory, this->CameraPoses);
  }
}

void mvView::LoadIndexData(vtkXMLDataElement *topel)
{
  this->NumberOfPolyDatas = 0;
  this->NumberOfTextures = 0;

  // backwards compatibility
  topel->GetScalarAttribute("Latitude", this->Latitude);
  topel->GetScalarAttribute("Longitude", this->Longitude);

  vtkXMLDataElement *el = topel->FindNestedElementWithName("CameraPoses");
  this->PosesXML = el;
  this->PosesXML->Register(nullptr);

  int numPoses = el->GetNumberOfNestedElements();
  this->CameraPoses.resize(numPoses);
  int totalActorCount = 0;
  for (size_t i = 0; i < numPoses; i++)
  {
    vtkXMLDataElement *poseel = el->GetNestedElement(static_cast<int>(i));

    mvCameraPose &pose = this->CameraPoses[i];

    poseel->GetScalarAttribute("PoseNumber", pose.PoseNumber);

    int numActors = poseel->GetNumberOfNestedElements();
    pose.ActorIDs.resize(numActors);
    for (size_t j = 0; j < numActors; j++)
    {
      vtkXMLDataElement *actorel = poseel->GetNestedElement(static_cast<int>(j));
      actorel->GetScalarAttribute("ActorID", pose.ActorIDs[j]);

      if (pose.ActorIDs[j] >= totalActorCount)
      {
        totalActorCount = pose.ActorIDs[j] + 1;
      }
    }

    // add a tour stop for every pose by default
    this->TourStops.resize(this->TourStops.size() + 1);
    auto &currTS = this->TourStops[this->TourStops.size() - 1];
    currTS.CameraPose = &pose;
    currTS.PoseNumber = pose.PoseNumber;

    // resort the vector
    std::sort(this->TourStops.begin(), this->TourStops.end(), alessb);
  }

  // realloc vectors
  this->Actors.resize(totalActorCount);
  vtkXMLDataElement *actorsel = topel->FindNestedElementWithName("Actors");
  for (int i = 0; i < this->Actors.size(); i++)
  {
    vtkXMLDataElement *actorel = actorsel->GetNestedElement(i);
    int aid;
    actorel->GetScalarAttribute("ActorID", aid);
    mvActor &actor = this->Actors[aid];
    actor.LoadData(actorel);
    if (actor.PolyDataID >= this->NumberOfPolyDatas)
    {
      this->NumberOfPolyDatas = actor.PolyDataID + 1;
    }
    if (actor.TextureID >= this->NumberOfTextures)
    {
      this->NumberOfTextures = actor.TextureID + 1;
    }
  }

  this->PolyDataSizes.resize(this->NumberOfPolyDatas);
  this->Textures.resize(this->NumberOfTextures);

  // now apply actor properties
  for (int i = 0; i < this->Actors.size(); i++)
  {
    vtkXMLDataElement *actorel = actorsel->GetNestedElement(i);
    int aid;
    actorel->GetScalarAttribute("ActorID", aid);

    actorel->GetScalarAttribute("PolyDataSize",
      this->PolyDataSizes[this->Actors[aid].PolyDataID]);
    if (this->Actors[aid].TextureID >= 0)
    {
      actorel->GetScalarAttribute("TextureSize",
        this->Textures[this->Actors[aid].TextureID].Size);
    }
  }
}

void mvView::Initialize(std::string fname, mvLobby *lobby)
{
  this->MyLobby = lobby;

  this->RootDirectory = vtksys::SystemTools::GetFilenamePath(fname);
  this->Name = vtksys::SystemTools::GetFilenameName(this->RootDirectory);

  vtkNew<vtkXMLUtilities> xml;
  vtkXMLDataElement *topel = xml->ReadElementFromFile(fname.c_str());

  this->LoadIndexData(topel);

  const char *timage = topel->GetAttribute("ViewImage");
  if (timage)
  {
    this->ViewFile = this->RootDirectory;
    this->ViewFile += "/";
    this->ViewFile += timage;
  }
  topel->Delete();

  // also load the extra file if found
  std::string extraname = this->RootDirectory  + "/extra.xml";
  if (vtksys::SystemTools::FileExists(extraname.c_str(), true))
  {
    vtkXMLDataElement *etopel =
      xml->ReadElementFromFile(extraname.c_str());
    this->LoadExtraData(etopel);
    etopel->Delete();
  }

  // draw a line from the view to the map
  this->Line->SetPoint2(2.0*this->Longitude/180.0, 2.0*this->Latitude/180.0, 0.0);
  vtkNew<vtkPolyDataMapper> lineMapper;
  this->LineActor->SetMapper(lineMapper);
  lineMapper->SetInputConnection(this->Line->GetOutputPort());
  this->LineActor->GetProperty()->SetAmbient(1.0);
  this->LineActor->GetProperty()->SetDiffuse(0.0);
  if (this->MyLobby->GetDisplayMap() &&
      (this->Latitude != 0.0 || this->Longitude != 0.0))
  {
    this->MyLobby->GetRenderer()->AddActor(this->LineActor);
  }

  vtkNew<vtkPolyDataMapper> bmapper;
  bmapper->SetInputConnection(this->BackgroundPlane->GetOutputPort());
  this->BackgroundActor->SetMapper(bmapper);
  this->BackgroundActor->GetProperty()->SetColor(0.6,1.0,0.6);
  this->BackgroundActor->GetProperty()->SetAmbient(1.0);
  this->BackgroundActor->GetProperty()->SetDiffuse(0.0);

  // check if the viewfile is present, if not take the first jpeg found
  if (!vtksys::SystemTools::FileExists(this->ViewFile.c_str(), true))
  {
    vtkNew<vtkGlobFileNames> glob;
    glob->SetDirectory(this->RootDirectory.c_str());
    glob->RecurseOff();
    glob->AddFileNames("*.jpg");
    int numjpg = glob->GetNumberOfFileNames();
    if (numjpg > 0)
    {
      this->ViewFile = glob->GetNthFileName(0);
    }
  }

  vtkNew<vtkTexture> tex;
  vtkNew<vtkJPEGReader> reader;
  reader->SetFileName(this->ViewFile.c_str());
  tex->SetInputConnection(reader->GetOutputPort());
  tex->InterpolateOn();
  tex->MipmapOn();
  vtkNew<vtkPolyDataMapper> mapper;
  mapper->SetInputConnection(this->Plane->GetOutputPort());
  this->PosterActor->SetMapper(mapper);
  this->PosterActor->SetTexture(tex);
  this->PosterActor->GetProperty()->SetAmbient(1.0);
  this->PosterActor->GetProperty()->SetDiffuse(0.0);
  this->MyLobby->GetRenderer()->AddActor(this->PosterActor);

  this->LoadAudioClips();

  this->MessageRepresentation->GetTextActor()->GetTextProperty()->SetFontFamilyToTimes();
  this->MessageRepresentation->GetTextActor()->GetTextProperty()->SetFontSize(28);
  this->MessageRepresentation->GetTextActor()->GetTextProperty()->SetFrame(1);
  this->MessageRepresentation->GetTextActor()->GetTextProperty()->SetFrameWidth(4);
  this->MessageRepresentation->GetTextActor()->GetTextProperty()->SetBackgroundOpacity(1.0);
  this->MessageRepresentation->GetTextActor()->GetTextProperty()->SetBackgroundColor(1, 1, 1);
  this->MessageRepresentation->GetTextActor()->GetTextProperty()->SetColor(0, 0, 0);
  this->MessageRepresentation->GetTextActor()->GetTextProperty()->SetFrameColor(0.2,0.2,0.2);

  this->MessageRepresentation->SetCoordinateSystemToWorld();
}

void mvView::SetActive(bool act) {
  this->Active = act;
  this->PosterActor->GetProperty()->SetAmbient(act ? 1.0 : 0.3);
}

void mvView::GenerateLODs()
{
  for (int i = 0; i < this->NumberOfPolyDatas; ++i)
  {
    // if it is big then generate a LOD. In this case big means
    // more than 10MB
    if (this->PolyDataSizes[i] > 10000)
    {
      std::ostringstream tmps;
      tmps << this->RootDirectory << "/data/pdata" << i;
      std::string baseName = tmps.str();
      baseName += ".vtp";
      vtkNew<vtkXMLPolyDataReader> rdr;
      rdr->SetFileName(baseName.c_str());
      rdr->Update();

      vtkNew<vtkQuadricClustering> cc;
      cc->UseInputPointsOn();
      cc->CopyCellDataOn();
      cc->SetInputConnection(rdr->GetOutputPort());

      vtkNew<vtkXMLPolyDataWriter> wtr;
      wtr->SetInputConnection(cc->GetOutputPort());
      tmps << ".lod.vtp";
      wtr->SetFileName(tmps.str().c_str());
      wtr->Write();
    }
  }

  // and textures, we want LODs for mip mapping
  for (int i = 0; i < this->NumberOfTextures; ++i)
  {
    if (this->Textures[i].Size > 10000)
    {
      std::ostringstream tmps;
      tmps << this->RootDirectory << "/data/tdata" << i;
      std::string baseName = tmps.str();
      baseName += ".vti";
      vtkNew<vtkXMLImageDataReader> rdr;
      rdr->SetFileName(baseName.c_str());
      rdr->Update();

      vtkSmartPointer<vtkImageData> id = rdr->GetOutput();
      int dims[3];
      id->GetDimensions(dims);

      // write out multiple LODs until we get down to
      // less than 1000
      unsigned int currentSize = this->Textures[i].Size;
      int level = 1;
      while (currentSize > 1000)
      {
        vtkNew<vtkImageResize> shrink;
        shrink->SetInputData(id);
        dims[0] = dims[0] > 1 ? dims[0] / 2 : 1;
        dims[1] = dims[1] > 1 ? dims[1] / 2 : 1;
        shrink->SetOutputDimensions(dims[0],dims[1],1);

        std::ostringstream tmps2;
        tmps2 << tmps.str() << ".lod" << level << ".vti";
        vtkNew<vtkXMLImageDataWriter> wtr;
        wtr->SetInputConnection(shrink->GetOutputPort());
        wtr->SetFileName(tmps2.str().c_str());
        wtr->Write();
        id = shrink->GetOutput();
        currentSize /= 4;
        level++;
      }
    }
  }
}

namespace {
VTK_THREAD_RETURN_TYPE LODLoadFunction(void *data)
{
  vtkMultiThreader::ThreadInfo *tinfo =
    static_cast<vtkMultiThreader::ThreadInfo*>(data);

  mvView *self =
    static_cast<mvView *>(tinfo->UserData);
  self->LODLoad(data);
  return VTK_THREAD_RETURN_VALUE;
}
}

//----------------------------------------------------------------------------
// functions to convert progress calls.
class ViewLODProgress : public vtkCommand
{
public:
  // generic new method
  static ViewLODProgress *New()
    { return new ViewLODProgress; }

  // the execute
  void Execute(vtkObject *caller,
     unsigned long event, void* vtkNotUsed(v)) override
  {
    this->ThreadInfo->ActiveFlagLock->Lock();
    if (*this->ThreadInfo->ActiveFlag == 0)
    {
      vtkAlgorithm *alg = vtkAlgorithm::SafeDownCast(caller);
      alg->AbortExecuteOn();
    }
    this->ThreadInfo->ActiveFlagLock->Unlock();
  }

  mvView *Self;
  vtkMultiThreader::ThreadInfo *ThreadInfo;
};


void mvView::LODLoad(void *data)
{
  vtkMultiThreader::ThreadInfo *tinfo =
    static_cast<vtkMultiThreader::ThreadInfo*>(data);

  double startTime = vtkTimerLog::GetUniversalTime();

  vtkNew<ViewLODProgress> cb;
  cb->Self = this;
  cb->ThreadInfo = tinfo;

  // iterate over the polydata loading full resolution
  for (int i = 0; i < this->NumberOfPolyDatas; ++i)
  {
    tinfo->ActiveFlagLock->Lock();
    if (*tinfo->ActiveFlag == 0)
    {
      tinfo->ActiveFlagLock->Unlock();
      return;
    }
    tinfo->ActiveFlagLock->Unlock();

    std::ostringstream tmps;
    tmps << this->RootDirectory << "/data/pdata" << i;

    // look for lod file first
    std::string lodFile = tmps.str();
    lodFile += ".lod.vtp";
    if (vtksys::SystemTools::FileExists(lodFile.c_str(),true))
    {
      vtkXMLPolyDataReader *rdr = vtkXMLPolyDataReader::New();
      rdr->AddObserver(vtkCommand::ProgressEvent, cb);
      std::string baseFile = tmps.str();
      baseFile += ".vtp";
      rdr->SetFileName(baseFile.c_str());
      rdr->Update();

      tinfo->ActiveFlagLock->Lock();
      if (*tinfo->ActiveFlag == 0)
      {
        rdr->Delete();
        tinfo->ActiveFlagLock->Unlock();
        return;
      }
      tinfo->ActiveFlagLock->Unlock();

      this->MyLobby->RenderLock();
      for (int j = 0; j < this->Actors.size(); ++j)
      {
        if (this->Actors[j].PolyDataID == i)
        {
          this->Actors[j].VTKActor->GetMapper()->SetInputConnection(rdr->GetOutputPort());
          this->Actors[j].VTKActor->GetMapper()->Modified();
        }
      }
      this->PolyDataReaders[i]->Delete();
      this->PolyDataReaders[i] = rdr;
      this->MyLobby->RenderUnlock();
    }
  }

  // iterate over texture maps
  // work up from lower LOD to higher. Aka load medium
  // levels of all textures before higher of any
  for (int level = this->MaxLODLevel; level >= 0; --level)
  {
    // look for any textures at this level, not yet loaded
    for (int i = 0; i < this->NumberOfTextures; ++i)
    {
      if (this->Textures[i].SmallestLOD >= level)
      {
        std::ostringstream tmps;
        tmps << this->RootDirectory << "/data/tdata" << i;
        std::string lodFile = this->GetTextureName(tmps.str(), level);
        vtkNew<vtkXMLImageDataReader> rdr;

        if (!vtksys::SystemTools::FileExists(lodFile.c_str(),true))
        {
          cerr << "Missing LOD file " << lodFile << "\n";
          return;
        }

        rdr->AddObserver(vtkCommand::ProgressEvent, cb);
        rdr->SetFileName(lodFile.c_str());
        rdr->Update();

        tinfo->ActiveFlagLock->Lock();
        if (*tinfo->ActiveFlag == 0)
        {
          tinfo->ActiveFlagLock->Unlock();
          return;
        }
        tinfo->ActiveFlagLock->Unlock();

        RenderOperation op;
        op.Reader = rdr;
        op.TextureIndex = i;
        op.Level = level;
        this->Textures[i].LastLoadedLOD = level;
        this->MyLobby->RenderLock();
        this->RenderOperations.push(op);
        this->MyLobby->RenderUnlock();
      }
    }
  }

  double endTime = vtkTimerLog::GetUniversalTime();

  cerr << "LODLoading of " << vtksys::SystemTools::GetFilenameName(this->RootDirectory) << " completed in " << floor(endTime - startTime) << " seconds.\n";
  this->LODLoadThreadId = -1;
}

std::string mvView::GetTextureName(const std::string &root, int level)
{
  std::ostringstream tmps;
  tmps << root;
  if (level)
  {
    tmps << ".lod" << level << ".vti";
  }
  else
  {
    tmps << ".vti";
  }
  return tmps.str();
}

// handle and prep for loading, but must be fast as it is on the main thread
// leave heavy lifting to the LoadBackgroundThread method
void mvView::LoadMainThread()
{
  // reset current pose to unknown
  this->CurrentTourStop = -1;
  this->InitialTourStop = 0;
  this->LoadProgress = 0;
  this->LODLoadThreadId = -1;
}

// load the data, runs in a background thread so the GUI
// doesn't hang
void mvView::LoadBackgroundThread(void *data)
{
  vtkMultiThreader::ThreadInfo *tinfo =
    static_cast<vtkMultiThreader::ThreadInfo*>(data);

  // poylydata and textures take the time, the rest is fast
  double totalCount = this->NumberOfPolyDatas
   + this->NumberOfTextures + 1;
  double accumulatedCount = 0;

  this->Textures.resize(0);

  // load up all the polydata
  this->PolyDataReaders.resize(this->NumberOfPolyDatas);
  for (int i = 0; i < this->NumberOfPolyDatas; ++i)
  {
    tinfo->ActiveFlagLock->Lock();
    if (*tinfo->ActiveFlag == 0)
    {
      tinfo->ActiveFlagLock->Unlock();
      this->LoadProgress = 0;
      // free partially loaded readers
      for (int j = 0; j < i; ++j)
      {
        this->PolyDataReaders[j]->Delete();
      }
      this->PolyDataReaders.resize(0);
      return;
    }
    tinfo->ActiveFlagLock->Unlock();

    std::ostringstream tmps;
    tmps << this->RootDirectory << "/data/pdata" << i;

    // look for lod file first
    std::string lodFile = tmps.str();
    lodFile += ".lod.vtp";
    this->PolyDataReaders[i] = vtkXMLPolyDataReader::New();
    if (vtksys::SystemTools::FileExists(lodFile.c_str(),true))
    {
      this->PolyDataReaders[i]->SetFileName(lodFile.c_str());
    }
    else
    {
      std::string baseFile = tmps.str();
      baseFile += ".vtp";
      this->PolyDataReaders[i]->SetFileName(baseFile.c_str());
    }
    this->PolyDataReaders[i]->Update();

    accumulatedCount++;
    this->LoadProgress = static_cast<int>(100.0*accumulatedCount/totalCount);
  }

  // and textures, we check to find the lowest LOD level available
  // and then load that.
  this->Textures.resize(this->NumberOfTextures);
  for (int i = 0; i < this->NumberOfTextures; ++i)
  {
    tinfo->ActiveFlagLock->Lock();
    if (*tinfo->ActiveFlag == 0)
    {
      tinfo->ActiveFlagLock->Unlock();
      this->LoadProgress = 0;
      this->Textures.clear();
      return;
    }
    tinfo->ActiveFlagLock->Unlock();

    std::ostringstream tmps;
    tmps << this->RootDirectory << "/data/tdata" << i;
    vtkNew<vtkXMLImageDataReader> rdr;

    // find the smallest LOD and use that
    for (int level = this->MaxLODLevel; level >= 0; --level)
    {
      std::string lodFile = this->GetTextureName(tmps.str(), level);
      if (vtksys::SystemTools::FileExists(lodFile.c_str(),true))
      {
        this->Textures[i].SmallestLOD = level;
        rdr->SetFileName(lodFile.c_str());
        break;
      }
    }

    rdr->Update();

    this->Textures[i].OpenGLTexture.TakeReference(vtkOpenGLTexture::New());
    this->Textures[i].OpenGLTexture->SetInputConnection(rdr->GetOutputPort());
    this->Textures[i].OpenGLTexture->InterpolateOn();
    this->Textures[i].OpenGLTexture->MipmapOn();
    accumulatedCount++;
    this->LoadProgress = static_cast<int>(100.0*accumulatedCount/totalCount);
  }

  // and photospheres
  for (auto &sphere : this->PhotoSpheres)
  {
    if (sphere.TextureFile.size())
    {
      vtkNew<vtkPolyDataMapper> smap;
      smap->SetInputConnection(this->SphereSource->GetOutputPort());
      sphere.Actor = vtkActor::New();
      sphere.Actor->SetMapper(smap);
      sphere.Actor->GetProperty()->SetColor(sphere.Color);
      sphere.Actor->GetProperty()->SetOpacity(sphere.Opacity);
      // sphere.Actor->GetProperty()->BackfaceCullingOn();

      sphere.Texture = vtkOpenGLTexture::New();
      sphere.Texture->InterpolateOn();
      // pose.Sphere.Texture->MipmapOn();
      sphere.Texture->RepeatOn();

      // watch for mpeg files
      std::string ext =
        vtksys::SystemTools::GetFilenameLastExtension(sphere.TextureFile);
      if (ext == ".mp4" || ext == ".mkv" || ext == ".webm")
      {
        sphere.IsVideo = true;
        sphere.VideoSource.TakeReference(vtkFFMPEGVideoSource::New());
        sphere.VideoSource->SetFileName(sphere.TextureFile.c_str());
        sphere.VideoSource->SetDecodingThreads(4);
        sphere.Texture->SetInputConnection(sphere.VideoSource->GetOutputPort());
      }
      // otherwise jpeg
      else
      {
        sphere.IsVideo = false;
      }
    }
  }

  this->MyLobby->RenderLock();

  for (int i = 0; i < this->Actors.size(); ++i)
  {
    this->Actors[i].VTKActor->GetMapper()->SetInputConnection(
      this->PolyDataReaders[this->Actors[i].PolyDataID]->GetOutputPort());
    this->Actors[i].VTKActor->GetMapper()->StaticOn();
    if (this->Actors[i].TextureID >= 0)
    {
      this->Actors[i].VTKActor->SetTexture(
        this->Textures[this->Actors[i].TextureID].OpenGLTexture);
    }
    this->MyLobby->GetRenderer()->AddActor(this->Actors[i].VTKActor);
    this->Actors[i].VTKActor->VisibilityOff();
  }

  // add photospheres
  for (auto &sphere : this->PhotoSpheres)
  {
    this->MyLobby->GetRenderer()->AddActor(sphere.Actor);
  }

  // add flags
  for (auto &flag : this->Flagpoles)
  {
    this->MyLobby->GetRenderer()->AddActor(flag.FlagpoleLabel);
  }

  this->MyLobby->RenderUnlock();

  tinfo->ActiveFlagLock->Lock();
  if (*tinfo->ActiveFlag == 0)
  {
    tinfo->ActiveFlagLock->Unlock();
    this->LoadProgress = 0;
    return;
  }
  tinfo->ActiveFlagLock->Unlock();

  while (this->RenderOperations.size())
  {
    this->RenderOperations.pop();
  }
  this->LODLoadThreadId =
    this->Threader->SpawnThread( LODLoadFunction, this);

  this->MyLobby->GetDashboardOverlay()->ReadCameraPoses(this->PosesXML);

  // 100% done
  this->LoadProgress = 100;
}

void mvView::GoToNextTourStop()
{
  int newStop = (this->CurrentTourStop + 1) % this->TourStops.size();

  // load up the actors for that stop
  if (this->CurrentTourStop >= 0)
  {
    mvTourStop &ts = this->TourStops[this->CurrentTourStop];

    if (ts.AdvanceToLobby)
    {
      this->MyLobby->ReturnFromView();
      return;
    }
  }

  this->GoToTourStop(newStop, true);
}

void mvView::GoToFirstTourStop()
{
  // allows other views to trigger GoToPose before we are loaded.
  int pose = this->InitialTourStop;
  this->InitialTourStop = -1;
  this->GoToTourStop(pose);
}

void mvView::ExitPhotoSphere()
{
  if (this->InPhotoSphere)
  {
    if (this->CurrentPhotoSphere->IsVideo)
    {
      this->CurrentPhotoSphere->VideoSource->Stop();
      this->CurrentPhotoSphere->VideoSource->ReleaseSystemResources();
    }
    this->MyLobby->GetRenderer()->RemoveActor(this->Skybox);
    this->MyLobby->GetRenderer()->RemoveActor(this->MovieSphere);
    this->InPhotoSphere = false;
    this->InLocalPhotoSphere = false;
  }
}

void mvView::UpdatePropVisibilityAndOpacity(mvTourStop &ts, float blendAmount)
{
  vtkRenderer *ren = this->MyLobby->GetRenderer();
  vtkOpenVRRenderWindow *renWin =
    static_cast<vtkOpenVRRenderWindow *>(ren->GetVTKWindow());

  std::vector<bool> usedActors;
  usedActors.resize(this->Actors.size(), false);

  // change visibility of actors
  if (ts.CameraPose)
  {
    for (auto actor : ts.CameraPose->ActorIDs)
    {
      this->Actors[actor].VTKActor->VisibilityOn();
      double targetOpacity = this->Actors[actor].Opacity;
      double opacity = this->Actors[actor].VTKActor->GetProperty()->GetOpacity();
      this->Actors[actor].VTKActor->GetProperty()->SetOpacity(
        blendAmount*targetOpacity > opacity ? blendAmount*targetOpacity : opacity);
      this->Actors[actor].VTKActor->SetForceOpaque(
        (this->Actors[actor].VTKActor->GetProperty()->GetOpacity() >= 1.0));
      usedActors[actor] = true;
    }
  }

  for (int i = 0; i < this->Actors.size(); ++i)
  {
    if (!usedActors[i])
    {
      this->Actors[i].VTKActor->ForceOpaqueOff();
      if (blendAmount >= 1.0)
      {
        this->Actors[i].VTKActor->GetProperty()->SetOpacity(0.0);
        this->Actors[i].VTKActor->VisibilityOff();
      }
      else
      {
        double opacity = this->Actors[i].VTKActor->GetProperty()->GetOpacity();
        this->Actors[i].VTKActor->GetProperty()->SetOpacity(
          opacity < (1.0 - blendAmount) ? opacity : 1.0 - blendAmount);
      }
    }
  }

  // display photo spheres
  for (auto &sphere : this->PhotoSpheres)
  {
    if (sphere.Actor)
    {
      if (sphere.TextureFile.size() && (
            sphere.NumberOfLocations == 0 ||
            std::find(sphere.Locations.begin(), sphere.Locations.end(), ts.PoseNumber) != sphere.Locations.end()))
      {
        if (sphere.Size != 0.0)
        {
          sphere.Actor->SetScale(sphere.Size, sphere.Size, sphere.Size);
        }
        else
        {
        double scale = renWin->GetPhysicalScale();
        sphere.Actor->SetScale(scale*0.2, scale*0.2, scale*0.2);
        }

        sphere.Actor->SetPosition(sphere.Position);
        sphere.Actor->VisibilityOn();
      }
      else
      {
        sphere.Actor->VisibilityOff();
      }
    }
  }

  // display ands update flagpoles
  std::vector<bool> usedFlags;
  usedFlags.resize(this->Flagpoles.size(), false);

  for (int i = 0; i < this->Flagpoles.size(); ++i)
  {
    mvFlagpole &flag = this->Flagpoles[i];
    if (flag.FlagpoleLabel->GetInput() && (
          flag.NumberOfLocations == 0 ||
          std::find(flag.Locations.begin(), flag.Locations.end(), ts.PoseNumber)
          != flag.Locations.end()))
    {
      flag.FlagpoleLabel->VisibilityOn();
      double opacity = flag.FlagpoleLabel->GetProperty()->GetOpacity();
      flag.FlagpoleLabel->GetProperty()->SetOpacity(
        blendAmount > opacity ? blendAmount : opacity);
      flag.FlagpoleLabel->SetForceOpaque(
        (flag.FlagpoleLabel->GetProperty()->GetOpacity() >= 1.0));
      usedFlags[i] = true;
    }
  }

  for (int i = 0; i < this->Flagpoles.size(); ++i)
  {
    if (!usedFlags[i])
    {
      mvFlagpole &flag = this->Flagpoles[i];
      flag.FlagpoleLabel->ForceOpaqueOff();
      if (blendAmount >= 1.0)
      {
        flag.FlagpoleLabel->GetProperty()->SetOpacity(0.0);
        flag.FlagpoleLabel->VisibilityOff();
      }
      else
      {
        double opacity = flag.FlagpoleLabel->GetProperty()->GetOpacity();
        flag.FlagpoleLabel->GetProperty()->SetOpacity(
          opacity < (1.0 - blendAmount) ? opacity : 1.0 - blendAmount);
      }
    }
  }
  this->MyLobby->GetRenderer()->ResetCameraClippingRange();
}

void mvView::HandleAnimation()
{
  if (!this->Animating)
  {
    return;
  }

  // 0 to 1.0
  double percentComplete =
    (vtkTimerLog::GetUniversalTime() - this->AnimationStartTime)/this->AnimationDuration;

  double tickDuration = vtkTimerLog::GetUniversalTime() - this->AnimationLastTick;
  double factor = tickDuration/this->AnimationDuration; // 0 to 1.0
  double total = 0.4 + 0.6*sqrt(vtkMath::Pi())/6.0;
  if (percentComplete < 0.3)
  {
    double eexp = 3.0*percentComplete/0.3 - 3.0;
    double currentValue = exp(-eexp*eexp);
    factor *= currentValue/total;
  }
  else if (percentComplete > 0.7)
  {
    double eexp = 3.0*(1.0 - percentComplete)/0.3 - 3.0;
    double currentValue = exp(-eexp*eexp);
    factor *= currentValue/total;
  }
  else
  {
    factor /= total;
  }

  factor *= 1.01;
  factor += this->AnimationSum;
  if (factor > 1.0)
  {
    factor = 1.0;
  }

  vtkRenderer *ren = this->MyLobby->GetRenderer();
  vtkOpenVRRenderWindow *renWin =
    static_cast<vtkOpenVRRenderWindow *>(ren->GetVTKWindow());

  // this->PoseInterpolatorInstance.Fly(this->SavedPose, this->NextPose, ren, factor);
  mvPoseInterpolator::Interpolate(
    this->SavedPose,
    this->NextPose, factor, this->AnimationSum, 0.6,
    static_cast<vtkOpenVRCamera*>(ren->GetActiveCamera()),
    renWin);
  this->MyLobby->GetRenderer()->ResetCameraClippingRange();
  this->AnimationSum = factor;

  if (percentComplete > 0.25)
  {
    std::vector<bool> usedActors;
    usedActors.resize(this->Actors.size(), false);
    mvTourStop &ts = this->TourStops[this->CurrentTourStop];

    double opacityAdjust = 2.0*(percentComplete - 0.25);
    if (opacityAdjust > 1.0)
    {
      opacityAdjust = 1.0;
    }

    this->UpdatePropVisibilityAndOpacity(ts, opacityAdjust);
  }

  this->AnimationLastTick += tickDuration;
  if (this->AnimationLastTick >  this->AnimationStartTime + this->AnimationDuration)
  {
    this->Animating = false;
  }
}

// poseNum is the vector index, which may be different
// from the actual pose number
// fromCollab indicates a message from a collaborator to match their pose.
void mvView::GoToTourStop(int index, bool broadcast, bool fromCollab,
  double *collabTrans, double *collabDir)
{
  if (index < 0 || index >= this->TourStops.size())
  {
    return;
  }
  // if we aren't loaded yet, InitialPose will be >= 0, set it and return
  if (this->InitialTourStop >= 0)
  {
    this->InitialTourStop = index;
    return;
  }

  // are we going from a model view to another model view?
  bool fromPoseToPose = !this->InPhotoSphere;

  // exit any photospheres we are in
  this->ExitPhotoSphere();

  vtkRenderer *ren = this->MyLobby->GetRenderer();
  vtkOpenVRRenderWindow *renWin =
    static_cast<vtkOpenVRRenderWindow *>(ren->GetVTKWindow());

  // load up the actors for that stop
  mvTourStop &ts = this->TourStops[index];

  cerr << "Now viewing tour stop " << ts.PoseNumber << "\n";

  // if there is an audio clip then play it
  if (ts.AudioFile.size())
  {
    PlaySound(ts.AudioFile.c_str(), nullptr, SND_ASYNC | SND_FILENAME | SND_NODEFAULT);
  }

  // if the tourstop should show a photosphere then do that instead
  if (ts.ShowPhotoSphere.size())
  {
    for (auto &sphere : this->PhotoSpheres)
    {
      if (sphere.TextureFile == ts.ShowPhotoSphere)
      {
        this->CurrentTourStop = index;
        this->Animating = false;
        this->GoToPhotoSphere(sphere);
        if (fromCollab)
        {
          renWin->SetPhysicalViewDirection(collabDir);
        }
        break;
      }
    }
    if (broadcast)
    {
      this->MyLobby->GetCollaborationClient()->
        SendMessage("P", index,
          renWin->GetPhysicalTranslation(),
          renWin->GetPhysicalViewDirection());
    }
    return;
  }

  std::vector<bool> usedActors;
  usedActors.resize(this->Actors.size(), false);

  this->SavedPose->Set(
    static_cast<vtkOpenVRCamera*>(ren->GetActiveCamera()), renWin);
  this->MyLobby->NewPose(static_cast<int>(floor(ts.CameraPose->PoseNumber)));
  this->NextPose->Set(
    static_cast<vtkOpenVRCamera*>(ren->GetActiveCamera()), renWin);


  if (fromCollab)
  {
    renWin->SetPhysicalTranslation(collabTrans);
    renWin->SetPhysicalViewDirection(collabDir);
    renWin->UpdateHMDMatrixPose();
    this->NextPose->Set(
      static_cast<vtkOpenVRCamera*>(ren->GetActiveCamera()), renWin);
    renWin->UpdateHMDMatrixPose();
  }
  else
  {
    if (broadcast)
    {
      this->MyLobby->GetCollaborationClient()->SendMessage("P", index,
          renWin->GetPhysicalTranslation(),
          renWin->GetPhysicalViewDirection());
    }
  }

  if (!fromPoseToPose || this->CurrentTourStop == -1 || ts.AnimationDuration <= 0.0)
  {
    this->CurrentTourStop = index;
    this->UpdatePropVisibilityAndOpacity(ts,1.0);
  }
  else
  {
    this->CurrentTourStop = index;

    // fly between old and new pose
    this->AnimationStartTime = vtkTimerLog::GetUniversalTime();
    this->AnimationLastTick = this->AnimationStartTime;
    this->AnimationDuration = ts.AnimationDuration;
    this->AnimationSum = 0.0;
    this->SavedPose->Apply(
      static_cast<vtkOpenVRCamera*>(ren->GetActiveCamera()),
       renWin);
    renWin->UpdateHMDMatrixPose();

    // enter flying mode
    this->Animating = true;
  }

  renWin->UpdateHMDMatrixPose();

  // Get coordinate system so we can place objects
  // in the user's view easily
  vtkCamera *cam = this->MyLobby->GetRenderer()->GetActiveCamera();
  vtkVector3d physvup(renWin->GetPhysicalViewUp());
  vtkVector3d camdop(cam->GetDirectionOfProjection());
  vtkVector3d camvright = camdop.Cross(physvup);
  camvright.Normalize();
  double scale = renWin->GetPhysicalScale();

  // display a message box?
  if (ts.MessageBoxText.size())
  {
    this->MessageWidget->SetInteractor(renWin->GetInteractor());
    this->MessageWidget->SetRepresentation(this->MessageRepresentation.Get());
    this->MessageRepresentation->SetText(ts.MessageBoxText.c_str());

    // have to compute a good placement via
    // vup, normal, bounds, scale
    vtkVector3d messvup;
    double wxyz[4];
    wxyz[0] = -3.1416/6.0;
    wxyz[1] = camvright[0];
    wxyz[2] = camvright[1];
    wxyz[3] = camvright[2];
    vtkMath::RotateVectorByWXYZ(physvup.GetData(), wxyz, messvup.GetData());
    vtkVector3d messnorm;
    wxyz[0] = 3.1416/3.0;
    vtkMath::RotateVectorByWXYZ(physvup.GetData(), wxyz, messnorm.GetData());

    vtkVector3d campos(cam->GetPosition());
    campos = campos + camdop*0.6*scale - camvright*0.4*scale - physvup*0.6*scale;
    double bnds[6] =
    {
      campos[0], campos[0],
      campos[1], campos[1],
      campos[2], campos[2]
    };

    this->MessageRepresentation->PlaceWidgetExtended(
      bnds, messnorm.GetData(), messvup.GetData(), scale*1.5);

    this->MessageWidget->SetEnabled(1);
  }
  else
  {
    this->MessageWidget->SetEnabled(0);
  }

  // after all actors are updated, reset CR
  this->MyLobby->GetRenderer()->ResetCameraClippingRange();
}

void mvView::ReturnFromView()
{
  while (this->RenderOperations.size())
  {
    this->RenderOperations.pop();
  }

  // stop any audio
  PlaySound(nullptr, nullptr, SND_ASYNC);

  if (this->LODLoadThreadId >= 0)
  {
    this->Threader->TerminateThread(this->LODLoadThreadId);
  }

  this->ExitPhotoSphere();

  // free up the memory
  for (auto &sphere : this->PhotoSpheres)
  {
    if (sphere.Texture)
    {
      sphere.Texture->Delete();
      sphere.Texture = nullptr;
    }
    if (sphere.Actor)
    {
      this->MyLobby->GetRenderer()->RemoveActor(sphere.Actor);
      sphere.Actor->Delete();
      sphere.Actor = nullptr;
    }
  }

  for (auto &flag : this->Flagpoles)
  {
    this->MyLobby->GetRenderer()->RemoveActor(flag.FlagpoleLabel);
  }

  // actors
  for (auto &actor : this->Actors)
  {
    this->MyLobby->GetRenderer()->RemoveActor(actor.VTKActor);
    if (actor.VTKActor->GetTexture())
    {
      actor.VTKActor->SetTexture(nullptr);
    }
    actor.VTKActor->GetMapper()->SetInputConnection(nullptr);
  }

  //readers
  for (auto rdr : this->PolyDataReaders)
  {
    rdr->Delete();
  }
  this->PolyDataReaders.resize(0);

  // and textures
  this->Textures.clear();

  this->MyLobby->GetRenderer()->RemoveActor(this->BackgroundActor);

  this->MessageWidget->SetEnabled(0);
}

void mvView::EventCallback(
  unsigned long eventID,
  void* calldata)
{
  if (eventID == vtkCommand::Button3DEvent)
  {
    vtkEventData* edata = static_cast<vtkEventData*>(calldata);
    vtkEventDataDevice3D* edd = edata->GetAsEventDataDevice3D();
    if (!edd)
    {
      return;
    }

    if (edd->GetInput() == vtkEventDataDeviceInput::ApplicationMenu &&
      edd->GetAction() == vtkEventDataAction::Release)
    {
      this->MyLobby->ReturnFromView();
      return;
    }

    if (edd->GetInput() == vtkEventDataDeviceInput::Trigger &&
        edd->GetAction() == vtkEventDataAction::Press)
    {
      // trigger press, record the time. If release is short
      // do a "click", otherwise do hold (show a ray)
      // this->MyLobby->GetStyle()->ShowRay(edd->GetDevice());
      this->ButtonPressTime = vtkTimerLog::GetUniversalTime();
      this->ButtonPressDevice = edd->GetDevice();
      vtkRenderer *ren = this->MyLobby->GetRenderer();
      vtkOpenVRRenderWindow *renWin =
        static_cast<vtkOpenVRRenderWindow *>(ren->GetVTKWindow());
      vtkOpenVRModel* mod = renWin->GetTrackedDeviceModel(edd->GetDevice());
      if (mod)
      {
        double scale = renWin->GetPhysicalScale();
        vtkCamera *cam = ren->GetActiveCamera();
        // Ray may need to be longer than 200 meters for large mine models.
        mod->SetRayLength(RAY_LENGTH * scale);
      }
    }
    if (edd->GetInput() == vtkEventDataDeviceInput::Trigger &&
        edd->GetAction() == vtkEventDataAction::Release)
    {
      this->MyLobby->GetStyle()->HideRay(edd->GetDevice());
      this->MyLobby->GetCollaborationClient()->
        SendMessage("HR", static_cast<int>(edd->GetDevice()));
      double releaseTime = vtkTimerLog::GetUniversalTime();
      if (this->ButtonPressTime != 0 &&
        releaseTime - this->ButtonPressTime > PRESS_DELAY)
      {
        // if this wasn't a "click", just hide the ray.
        this->ButtonPressTime = 0;
        return;
      }
      this->ButtonPressTime = 0;
      if (!this->InPhotoSphere)
      {
        for (auto &sphere : this->PhotoSpheres)
        {
          if (sphere.Actor && sphere.Actor->GetVisibility())
          {
            // if they click in a sphere then go there
            double *pos = sphere.Actor->GetPosition();
            double radius = sphere.Actor->GetScale()[0];
            const double *epos = edd->GetWorldPosition();
            if (sqrt(vtkMath::Distance2BetweenPoints(pos, epos)) < radius)
            {
              this->InLocalPhotoSphere = true;
              this->GoToPhotoSphere(sphere);
              return;
            }
          }
        }
      }
      if (this->InLocalPhotoSphere)
      {
        this->ReturnFromPhotoSphere();
      }
      else
      {
        this->GoToNextTourStop();
      }
    }

    if (edd->GetInput() == vtkEventDataDeviceInput::TrackPad &&
        edd->GetAction() == vtkEventDataAction::Press)
    {
      if (!this->InPhotoSphere)
      {
        this->MyLobby->GetStyle()->StartDolly3D(edd);
      }
    }

    if (edd->GetInput() == vtkEventDataDeviceInput::TrackPad &&
        edd->GetAction() == vtkEventDataAction::Release)
    {
      this->MyLobby->GetStyle()->EndDolly3D(edd);
    }
    return;
  }

  if (eventID == vtkCommand::RenderEvent)
  {
    // process any opengl code we need done in the
    // main thread. Nothing here should take more than
    // 20ms or so. Must be fast
    this->HandleAnimation();
    this->ProcessRenderOperations();

    if (this->ButtonPressTime != 0)
    {
      double holdTime = vtkTimerLog::GetUniversalTime();
      if (holdTime - this->ButtonPressTime > PRESS_DELAY)
      {
        this->MyLobby->GetStyle()->ShowRay(this->ButtonPressDevice);
        this->MyLobby->GetCollaborationClient()->
          SendMessage("SR", static_cast<int>(this->ButtonPressDevice));
      }
    }
  }
}

// process any opengl code we need done in the
// main thread. Nothing here should take more than
// 20ms or so. Must be fast
void mvView::ProcessRenderOperations()
{
  static unsigned int count = 0;
  count++;
  // reset at 1000
  if (count > 1000)
  {
    count = 0;
  }

  // update progress, now and then
  if (count%10 == 0)
  {
    int prog = this->LoadProgress.load();
    // 0 to 100 percent as an int
    if (prog <= 0 || prog >= 100)
    {
      this->MyLobby->GetRenderer()->RemoveActor(this->BackgroundActor);
    }
    else
    {
      this->MyLobby->GetRenderer()->AddActor(this->BackgroundActor);
      double *p1 = this->Plane->GetPoint1();
      double *p2 = this->Plane->GetPoint2();
      this->BackgroundPlane->SetPoint2(
        p1[0], p1[1], p1[2] + 0.01 * prog * (p2[2] - p1[2]));
    }
  }

  // update the audio direction if needed, also check if the video ended
  if (this->InPhotoSphere && this->CurrentPhotoSphere->IsVideo)
  {
    // if at the end of the movie, possibly autoadvance
    mvTourStop &ts = this->TourStops[this->CurrentTourStop];
    if (this->CurrentPhotoSphere->VideoSource->GetEndOfFile() && ts.AutoAdvance)
    {
      this->GoToNextTourStop();
      return;
    }
    vtkRenderer *ren = this->MyLobby->GetRenderer();
    vtkCamera *cam = ren->GetActiveCamera();
    this->MyLobby->GetAudioHandler()
      ->SetListenerPose(cam->GetViewUp(), cam->GetDirectionOfProjection());
  }

  if (this->RenderOperations.size() && count%4 == 0)
  {
    RenderOperation &op = this->RenderOperations.front();

    mvTexture &texture = this->Textures[op.TextureIndex];

    vtkImageData *id = op.Reader->GetOutput();
    int *dims = id->GetDimensions();
    int numComp = id->GetNumberOfScalarComponents();

    // make sure we create and assign a texture object
    if (texture.TextureObject == nullptr)
    {
      texture.TextureObject.TakeReference(vtkTextureObject::New());
      texture.TextureObject->SetAutoParameters(0);
      vtkRenderer *ren = this->MyLobby->GetRenderer();
      vtkOpenVRRenderWindow *renWin =
        static_cast<vtkOpenVRRenderWindow *>(ren->GetVTKWindow());
      texture.TextureObject->SetContext(renWin);
      texture.TextureObject->SetMagnificationFilter(vtkTextureObject::Linear);
      texture.TextureObject->SetMinificationFilter(vtkTextureObject::LinearMipmapLinear);
      texture.TextureObject->SetMaxLevel(texture.SmallestLOD);
      // cerr << "creating level " << texture.SmallestLOD << "\n";
    }

    if (op.CurrentRow == 0)
    {
      // cerr << "allocating  level " << op.Level << " " << dims[0] << " by " << dims[1] << "\n";
      texture.TextureObject->SetBaseLevel(op.Level);
      texture.TextureObject->SetMinLOD(op.Level+1);
      texture.TextureObject->Allocate2D(
        dims[0], dims[1], numComp, VTK_UNSIGNED_CHAR, op.Level);
      texture.TextureObject->Bind();
      texture.TextureObject->SendParameters();
    }

    if (op.CurrentRow == dims[1])
    {
      // cerr << "completed level " << op.Level << "\n";
      if (op.Level < texture.SmallestLOD)
      {
        texture.TextureObject->SetMinLOD(op.Level);
        texture.TextureObject->Bind();
        texture.TextureObject->SendParameters();
      }

      // wait until we have 2 textures before showing
      if (op.Level == texture.SmallestLOD - 1)
      {
        // sets for the smallest LOD, no op after that
        this->Textures[op.TextureIndex].OpenGLTexture
          ->SetTextureObject(texture.TextureObject);
      }

      this->RenderOperations.pop();
      return;
    }

    // load some data to the texture in chunks of ~4MB RGBA
    int height = 1000000/dims[0];
    if (op.CurrentRow + height > dims[1])
    {
      height = dims[1] - op.CurrentRow;
    }
    texture.TextureObject->Bind();
    vtkDataArray *da = id->GetPointData()->GetScalars();
    glTexSubImage2D(
      texture.TextureObject->GetTarget(),
      op.Level, // level
      0, // xoffset
      op.CurrentRow, // yoffset
      dims[0], // width
      height,
      numComp == 4 ? GL_RGBA : GL_RGB,
      GL_UNSIGNED_BYTE,
      da->GetVoidPointer(op.CurrentRow*dims[0]*numComp)
      );
    op.CurrentRow += height;
  }
}

void mvView::GoToPhotoSphere(mvPhotoSphere &sphere)
{
  // turn off other stuff
  for (auto &actor : this->Actors)
  {
    actor.VTKActor->VisibilityOff();
  }
  for (auto &sphere : this->PhotoSpheres)
  {
    if (sphere.Actor)
    {
      sphere.Actor->VisibilityOff();
    }
  }
  for (auto &flag : this->Flagpoles)
  {
    flag.FlagpoleLabel->VisibilityOff();
  }
  this->MessageWidget->SetEnabled(0);

  vtkRenderer *ren = this->MyLobby->GetRenderer();

  vtkOpenVRRenderWindow *renWin =
    static_cast<vtkOpenVRRenderWindow *>(ren->GetVTKWindow());

  // save current pose?
  this->SavedPose->Set(
    static_cast<vtkOpenVRCamera*>(ren->GetActiveCamera()),
     renWin);

  double plane[4];
  renWin->GetPhysicalViewUp(plane);

  double right[3];
  // cross to the the right vector
  vtkMath::Cross(sphere.Front, plane, right);

  if (sphere.IsVideo)
  {
    ren->AddActor(this->MovieSphere);
    this->MovieSphere->SetVideoSource(sphere.VideoSource);
    this->MyLobby->GetAudioHandler()->SetupAudioPlayback(sphere.VideoSource);
    this->MyLobby->GetAudioHandler()->SetAudioPose(plane, sphere.Front);
    sphere.VideoSource->Record();
    if (sphere.VideoSource->GetStereo3D() || sphere.Stereo3D)
    {
      this->MovieSphere->SetProjectionToStereoSphere();
    }
    else
    {
      this->MovieSphere->SetProjectionToSphere();
    }
    this->MovieSphere->SetFloorPlane(plane[0], plane[1], plane[2], 0.0);
    this->MovieSphere->SetFloorRight(right[0], right[1], right[2]);
    this->MovieSphere->Reset();
  }
  else
  {
    ren->AddActor(this->Skybox);
    this->Skybox->SetTexture(sphere.Texture);
    vtkNew<vtkJPEGReader> rdr;
    rdr->SetFileName(sphere.TextureFile.c_str());
    rdr->Update();
    sphere.Texture->SetInputConnection(rdr->GetOutputPort());
    if (sphere.Stereo3D)
    {
      this->Skybox->SetProjectionToStereoSphere();
    }
    else
    {
      this->Skybox->SetProjectionToSphere();
    }
    this->Skybox->SetFloorPlane(plane[0], plane[1], plane[2], 0.0);
    this->Skybox->SetFloorRight(right[0], right[1], right[2]);
  }

  renWin->SetPhysicalTranslation(0.0,0.0,0.0);
  renWin->SetPhysicalScale(1.0);
  ren->GetActiveCamera()->SetPosition(0,0,0);
  ren->GetActiveCamera()->SetFocalPoint(right);
  // Match the length of avatar's ray.
  ren->GetActiveCamera()->SetClippingRange(0.1, RAY_LENGTH);
  this->InPhotoSphere = true;
  this->CurrentPhotoSphere = &sphere;
}

void mvView::ReturnFromPhotoSphere()
{
  this->GoToTourStop(this->CurrentTourStop);

  vtkOpenVRRenderWindow *renWin =
    static_cast<vtkOpenVRRenderWindow *>(this->MyLobby->GetRenderer()->GetVTKWindow());

  // restore current pose
  this->SavedPose->Apply(
    static_cast<vtkOpenVRCamera*>(this->MyLobby->GetRenderer()->GetActiveCamera()),
     renWin);
  renWin->UpdateHMDMatrixPose();
  this->MyLobby->GetRenderer()->ResetCameraClippingRange();
}
