//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#include "qtMaterialItem.h"

#include "smtk/simulation/truchas/qt/ctkCollapsibleButton.h"
#include "smtk/simulation/truchas/qt/qtMaterialAttribute.h"

#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/GroupItem.h"
#include "smtk/attribute/StringItem.h"
#include "smtk/extension/qt/qtAttributeItemInfo.h"
#include "smtk/extension/qt/qtUIManager.h"

#include <QColor>
#include <QDebug>
#include <QFrame>
#include <QLabel>
#include <QMargins>
#include <QPalette>
#include <QTextStream>
#include <QVBoxLayout>

typedef smtk::extension::qtAttributeItemInfo qtItemInfo;

class qtMaterialItemInternals
{
public:
  qtMaterialAttribute* m_qAttribute;
  smtk::attribute::GroupItemPtr m_phasesItem;
  smtk::attribute::GroupItemPtr m_transitionsItem;
};

static smtk::extension::qtItem* createItemWidget(
  const qtItemInfo& info, qtMaterialAttribute* attribute)
{
  return new qtMaterialItem(info, attribute);
}

qtMaterialItem::qtMaterialItem(const qtItemInfo& phasesInfo, qtMaterialAttribute* qAttribute)
  : smtk::extension::qtItem(phasesInfo)
{
  this->Internals = new qtMaterialItemInternals;
  this->Internals->m_qAttribute = qAttribute;
  auto att = qAttribute->attribute();
  this->Internals->m_phasesItem = att->findGroup("phases");
  this->Internals->m_transitionsItem = att->findGroup("transitions");

  this->createWidget();
}

qtMaterialItem::~qtMaterialItem()
{
  delete this->Internals;
}

void qtMaterialItem::createWidget()
{
  auto parentWidget = this->Internals->m_qAttribute->widget();
  m_widget = new QFrame(parentWidget);
  auto layout = new QVBoxLayout();
  m_widget->setLayout(layout);

  std::size_t phasesCount = 0;
  std::size_t transitionsCount = 0;

  // Check some validity conditions for the material attribute
  QString errorMessage;

  // Check that phases and transitions items are defined
  if (this->Internals->m_phasesItem == nullptr)
  {
    errorMessage = "Error: no \"phases\" item";
  }
  else if (this->Internals->m_transitionsItem == nullptr)
  {
    errorMessage = "Error: no \"transitions\" item";
  }
  else
  {
    // Check that number of phases is 1 more than number of transitions
    phasesCount = this->Internals->m_phasesItem->numberOfGroups();
    transitionsCount = this->Internals->m_transitionsItem->numberOfGroups();
    if (phasesCount > 0 && phasesCount != transitionsCount + 1)
    {
      QTextStream qs(&errorMessage);
      qs << "Error: invalid number of transitions should be 1 less than number of phases"
         << ", phases: " << phasesCount << ", transitions: " << transitionsCount;
    }
  }

  if (!errorMessage.isEmpty())
  {
    QLabel* label = new QLabel(errorMessage, m_widget);
    layout->addWidget(label);
    return;
  }

  QWidget* phaseWidget;
  QWidget* transitionWidget;
  QString title;
  QTextStream qs(&title);

  if (phasesCount > 1)
  {
    auto nameItem = this->Internals->m_phasesItem->findAs<smtk::attribute::StringItem>(0, "name");
    qs << "Phase 1: " << nameItem->value().c_str();
  }
  phaseWidget = this->createElementWidget(this->Internals->m_phasesItem, 0, title);
  layout->addWidget(phaseWidget);
  for (std::size_t i = 0; i < transitionsCount; ++i)
  {
    title = "Transition";
    transitionWidget = this->createElementWidget(this->Internals->m_transitionsItem, i, title);
    layout->addWidget(transitionWidget);

    auto nameItem = this->Internals->m_phasesItem->findAs<smtk::attribute::StringItem>(i + 1, "name");
    title = "";
    qs << "Phase " << (i + 2) << ": " << nameItem->value().c_str();
    phaseWidget = this->createElementWidget(this->Internals->m_phasesItem, i + 1, title);
    layout->addWidget(phaseWidget);
  }
}

QWidget* qtMaterialItem::createElementWidget(
  smtk::attribute::GroupItemPtr groupItem, std::size_t element, const QString& labelString)
{
  QWidget* frame = nullptr;
  int verticalPadding = 6;  // for esthetics (multiple-phase only)
  bool isSinglePhase = groupItem->name() == "phases" && groupItem->numberOfGroups() == 1;
  if (isSinglePhase)
  {
    // Display single-phase material with simple frame
    frame = new QFrame();
  }
  else
  {
    // Display multiple-phase material with collapsible frames
    auto collapsibleFrame = new ctkCollapsibleButton();
    collapsibleFrame->setVerticalPadding(verticalPadding);  // set this *before* text
    collapsibleFrame->setContentsFrameShape(QFrame::StyledPanel);
    collapsibleFrame->setCollapsedHeight(0);
    collapsibleFrame->setText(labelString);
    collapsibleFrame->setCollapsed(true);

    // Alternate row colors
    QPalette pal = collapsibleFrame->palette();
    QString hex = groupItem->name() == "phases" ? "#a0a0a0" : "#808080";
    pal.setColor(QPalette::Button, QColor(hex));
    collapsibleFrame->setAutoFillBackground(true);
    collapsibleFrame->setPalette(pal);
    collapsibleFrame->update();

    frame = collapsibleFrame;
  }

  auto layout = new QVBoxLayout(frame);
  // Adjust margin to match vertical padding - dont know why this is needed
  auto margins = layout->contentsMargins();
  int top = margins.top();
  margins.setTop(top + verticalPadding);
  layout->setContentsMargins(margins);

  const std::size_t numItems = groupItem->numberOfItemsPerGroup();
  for (std::size_t i = 0; i < numItems; i++)
  {
    auto item = groupItem->item(element, static_cast<int>(i));
    if (isSinglePhase && item->name() == "name")
    {
      // Omit phase name for single-phase materials
      continue;
    }

    smtk::view::View::Component comp; // create a default style (an empty component)
    qtItemInfo info(item, comp, m_widget, m_itemInfo.baseView());
    qtItem* childItem = m_itemInfo.uiManager()->createItem(info);
    if (childItem)
    {
      layout->addWidget(childItem->widget());
      // Todo
      // itemList.push_back(childItem);
      // connect(childItem, SIGNAL(modified()), this, SLOT(onChildItemModified()));
    }
  }
  return frame;
}
