//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#include "smtkTruchasMaterialsView.h"

#include "smtk/attribute/Attribute.h"
#include "smtk/extension/qt/qtTableWidget.h"
#include "smtk/extension/qt/qtUIManager.h"
#include "smtk/view/View.h"

#include "smtk/simulation/truchas/qt/qtMaterialAttribute.h"

#include <QDebug>
#include <QHBoxLayout>
#include <QHeaderView>
#include <QLabel>
#include <QPointer>
#include <QPushButton>
#include <QSizePolicy>
#include <QSplitter>
#include <QVBoxLayout>
#include <QVariant>

#include <string>
#include <vector>

class smtkTruchasMaterialsViewInternals
{
public:
  // Internal member data
  smtk::extension::qtTableWidget* ListTable;

  QPushButton* AddButton;
  QPushButton* DeleteButton;
  QPushButton* CopyButton;

  // QFrame* FiltersFrame;
  QFrame* ButtonsFrame;
  QFrame* TopFrame;
  QFrame* BottomFrame;

  QPointer<QFrame> AttFrame;
  QPointer<qtMaterialAttribute> CurrentAtt;
}; // Internals

qtBaseView* smtkTruchasMaterialsView::createViewWidget(const ViewInfo& info)
{
  smtkTruchasMaterialsView* view = new smtkTruchasMaterialsView(info);
  view->buildUI();
  return view;
}

smtkTruchasMaterialsView::smtkTruchasMaterialsView(const ViewInfo& info)
  : qtBaseView(info)
{
  this->Internals = new smtkTruchasMaterialsViewInternals();
}

smtkTruchasMaterialsView::~smtkTruchasMaterialsView()
{
  delete this->Internals;
}

void smtkTruchasMaterialsView::createWidget()
{
  // Display a simplified version of qtAttributeView top frame
  auto attResource = this->uiManager()->attResource();
  auto materialDefinition = attResource->findDefinition("material");

  QSplitter* frame = new QSplitter(this->parentWidget());
  frame->setOrientation(Qt::Vertical);

  QFrame* TopFrame = new QFrame(frame);
  QFrame* BottomFrame = new QFrame(frame);

  this->Internals->TopFrame = TopFrame;
  this->Internals->BottomFrame = BottomFrame;
  QSizePolicy sizeFixedPolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

  QVBoxLayout* TopLayout = new QVBoxLayout(TopFrame);
  TopLayout->setMargin(0);
  TopFrame->setSizePolicy(sizeFixedPolicy);
  QVBoxLayout* BottomLayout = new QVBoxLayout(BottomFrame);
  BottomLayout->setMargin(0);
  BottomFrame->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

  QSizePolicy tableSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
  // create a list box for all the entries
  this->Internals->ListTable = new smtk::extension::qtTableWidget(frame);
  this->Internals->ListTable->setSelectionMode(QAbstractItemView::SingleSelection);
  this->Internals->ListTable->setSelectionBehavior(QAbstractItemView::SelectRows);
  this->Internals->ListTable->setSizePolicy(tableSizePolicy);
  this->Internals->ListTable->setColumnCount(2);

  QStringList headerLabels;
  headerLabels << "Name"
               << "Type";
  this->Internals->ListTable->setHorizontalHeaderLabels(headerLabels);

  // Buttons frame
  this->Internals->ButtonsFrame = new QFrame(frame);
  QHBoxLayout* buttonLayout = new QHBoxLayout(this->Internals->ButtonsFrame);
  buttonLayout->setMargin(0);
  this->Internals->ButtonsFrame->setSizePolicy(sizeFixedPolicy);

  this->Internals->AddButton = new QPushButton("New", this->Internals->ButtonsFrame);
  this->Internals->AddButton->setSizePolicy(sizeFixedPolicy);
  this->Internals->DeleteButton = new QPushButton("Delete", this->Internals->ButtonsFrame);
  this->Internals->DeleteButton->setSizePolicy(sizeFixedPolicy);
  this->Internals->CopyButton = new QPushButton("Copy", this->Internals->ButtonsFrame);
  this->Internals->CopyButton->setSizePolicy(sizeFixedPolicy);

  buttonLayout->addWidget(this->Internals->AddButton);
  buttonLayout->addWidget(this->Internals->CopyButton);
  buttonLayout->addWidget(this->Internals->DeleteButton);

  // TopLayout->addWidget(this->Internals->FiltersFrame);
  TopLayout->addWidget(this->Internals->ButtonsFrame);
  TopLayout->addWidget(this->Internals->ListTable);

  // Attribute frame
  this->Internals->AttFrame = new QFrame(frame);
  new QVBoxLayout(this->Internals->AttFrame);
  BottomLayout->addWidget(this->Internals->AttFrame);
  //  this->Internals->AttFrame->setVisible(0);

  frame->addWidget(TopFrame);
  frame->addWidget(BottomFrame);

  this->Internals->ListTable->horizontalHeader()->setSectionResizeMode(
    QHeaderView::ResizeToContents);
  this->Internals->ListTable->horizontalHeader()->setDefaultAlignment(Qt::AlignLeft);

  // We want the signals that may change the attribute to be displayed Queued instead of
  // Direct so that QLineEdit::editingFinished signals are processed prior to these.
  QObject::connect(this->Internals->ListTable, SIGNAL(itemClicked(QTableWidgetItem*)), this,
    SLOT(onListBoxClicked(QTableWidgetItem*)), Qt::QueuedConnection);

  QObject::connect(this->Internals->ListTable,
    &smtk::extension::qtTableWidget::itemSelectionChanged, this,
    &smtkTruchasMaterialsView::onAttributeSelectionChanged, Qt::QueuedConnection);

  QObject::connect(this->Internals->ListTable, SIGNAL(itemChanged(QTableWidgetItem*)), this,
    SLOT(onAttributeNameChanged(QTableWidgetItem*)));
  // we need this so that the attribute name will also be changed
  // when a recorded test is play back, which is using setText
  // on the underline QLineEdit of the cell.
  QObject::connect(this->Internals->ListTable, SIGNAL(cellChanged(int, int)), this,
    SLOT(onAttributeCellChanged(int, int)), Qt::QueuedConnection);
  QObject::connect(this->Internals->AddButton, SIGNAL(clicked()), this, SLOT(onCreateNew()));
  QObject::connect(this->Internals->CopyButton, SIGNAL(clicked()), this, SLOT(onCopySelected()));
  QObject::connect(
    this->Internals->DeleteButton, SIGNAL(clicked()), this, SLOT(onDeleteSelected()));

  this->Widget = frame;

  // From qtAttributeView::onViewByWithDefinition()
  std::vector<smtk::attribute::AttributePtr> result;
  attResource->findAttributes("material", result);
  if (result.size())
  {
    std::vector<smtk::attribute::AttributePtr>::iterator it;
    for (it = result.begin(); it != result.end(); ++it)
    {
      QTableWidgetItem* item = this->addAttributeListItem(*it);
      if ((*it)->definition()->advanceLevel())
      {
        item->setFont(this->uiManager()->advancedFont());
      }
    }
  }
} // createWidget()

QTableWidgetItem* smtkTruchasMaterialsView::addAttributeListItem(
  smtk::attribute::AttributePtr childData)
{
  QTableWidgetItem* item =
    new QTableWidgetItem(QString::fromUtf8(childData->name().c_str()), smtk_USER_DATA_TYPE);
  Qt::ItemFlags nonEditableFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
  qDebug() << "Item text" << item->text();

  QVariant vdata;
  vdata.setValue(static_cast<void*>(childData.get()));
  item->setData(Qt::UserRole, vdata);
  item->setFlags(nonEditableFlags | Qt::ItemIsEditable);

  int numRows = this->Internals->ListTable->rowCount();
  this->Internals->ListTable->setRowCount(++numRows);
  this->Internals->ListTable->setItem(numRows - 1, 0, item);

  // add the type column too.
  std::string txtDef = childData->definition()->displayedTypeName();

  QTableWidgetItem* defitem =
    new QTableWidgetItem(QString::fromUtf8(txtDef.c_str()), smtk_USER_DATA_TYPE);
  defitem->setFlags(nonEditableFlags);
  this->Internals->ListTable->setItem(numRows - 1, 1, defitem);
  qDebug() << "defitem text" << defitem->text();

  return item;
}

smtk::attribute::AttributePtr smtkTruchasMaterialsView::getAttributeFromItem(QTableWidgetItem* item)
{
  smtk::attribute::Attribute* rawPtr = item
    ? static_cast<smtk::attribute::Attribute*>(item->data(Qt::UserRole).value<void*>())
    : nullptr;
  return rawPtr ? rawPtr->shared_from_this() : smtk::attribute::AttributePtr();
}

QTableWidgetItem* smtkTruchasMaterialsView::getSelectedItem()
{
  return this->Internals->ListTable->selectedItems().count() > 0
    ? this->Internals->ListTable->selectedItems().value(0)
    : nullptr;
}

void smtkTruchasMaterialsView::onAttributeSelectionChanged()
{
  this->Internals->AttFrame->setVisible(true);
  if (this->Internals->CurrentAtt != nullptr)
  {
    this->Internals->AttFrame->layout()->removeWidget(this->Internals->CurrentAtt->widget());
  }
  delete this->Internals->CurrentAtt;
  this->Internals->CurrentAtt = nullptr;

  QTableWidgetItem* currentItem = this->getSelectedItem();
  if (currentItem == nullptr)
  {
    return;
  }

  // (else) Initialize qtMaterialAttribute
  auto att = this->getAttributeFromItem(currentItem);
  if (att != nullptr)
  {
    smtk::view::View::Component comp;
    this->Internals->CurrentAtt =
      new qtMaterialAttribute(att, comp, this->Internals->AttFrame, this);

    this->Internals->CurrentAtt->createBasicLayout(false);
    if (this->Internals->CurrentAtt->widget())
    {
      this->Internals->AttFrame->layout()->addWidget(this->Internals->CurrentAtt->widget());
      if (this->advanceLevelVisible())
      {
        this->Internals->CurrentAtt->showAdvanceLevelOverlay(true);
      }
    }
  } // (att != nullptr)
}
