//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#include "qtMaterialItem.h"

#include "smtk/simulation/truchas/qt/ctkCollapsibleButton.h"
#include "smtk/simulation/truchas/qt/qtMaterialAttribute.h"

#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/DoubleItem.h"
#include "smtk/attribute/GroupItem.h"
#include "smtk/attribute/StringItem.h"
#include "smtk/extension/qt/qtAttributeItemInfo.h"
#include "smtk/extension/qt/qtUIManager.h"
#include "smtk/simulation/truchas/qt/qtMaterialMultiPhaseMoveButtons.h"
#include "smtk/simulation/truchas/qt/qtMaterialMultiPhaseValidityWidget.h"

#include <QCheckBox>
#include <QColor>
#include <QDebug>
#include <QFrame>
#include <QHeaderView>
#include <QLabel>
#include <QMargins>
#include <QMessageBox>
#include <QPushButton>
#include <QTableWidget>
#include <QTableWidgetItem>
#include <QTextStream>
#include <QVBoxLayout>

#include <memory>

typedef smtk::extension::qtAttributeItemInfo qtItemInfo;

class qtMaterialItemInternals
{
public:
  qtMaterialAttribute* m_qAttribute;
  smtk::attribute::GroupItemPtr m_phasesItem;
  smtk::attribute::GroupItemPtr m_transitionsItem;
  QTableWidget* m_tableWidget;
  QWidget* m_dataWidget;
};

static smtk::extension::qtItem* createItemWidget(
  const qtItemInfo& info, qtMaterialAttribute* attribute)
{
  return new qtMaterialItem(info, attribute);
}

qtMaterialItem::qtMaterialItem(const qtItemInfo& phasesInfo, qtMaterialAttribute* qAttribute)
  : smtk::extension::qtItem(phasesInfo)
{
  this->Internals = new qtMaterialItemInternals;
  this->Internals->m_qAttribute = qAttribute;
  auto att = qAttribute->attribute();
  this->Internals->m_phasesItem = att->findGroup("phases");
  this->Internals->m_transitionsItem = att->findGroup("transitions");
  this->Internals->m_tableWidget = nullptr;
  this->Internals->m_dataWidget = nullptr;

  this->createWidget();
}

qtMaterialItem::~qtMaterialItem()
{
  delete this->Internals;
}

void qtMaterialItem::createWidget()
{
  auto parentWidget = this->Internals->m_qAttribute->widget();
  m_widget = new QFrame(parentWidget);
  m_widget->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Expanding);
  auto layout = new QVBoxLayout();
  m_widget->setLayout(layout);

  std::size_t phasesCount = 0;
  std::size_t transitionsCount = 0;

  // Check some validity conditions for the material attribute
  QString errorMessage;

  // Check that phases and transitions items are defined
  if (this->Internals->m_phasesItem == nullptr)
  {
    errorMessage = "Error: no \"phases\" item";
  }
  else if (this->Internals->m_transitionsItem == nullptr)
  {
    errorMessage = "Error: no \"transitions\" item";
  }
  else
  {
    // Check that number of phases is 1 more than number of transitions
    phasesCount = this->Internals->m_phasesItem->numberOfGroups();
    transitionsCount = this->Internals->m_transitionsItem->numberOfGroups();
    if (phasesCount > 0 && phasesCount != transitionsCount + 1)
    {
      QTextStream qs(&errorMessage);
      qs << "Error: invalid number of transitions should be 1 less than number of phases"
         << ", phases: " << phasesCount << ", transitions: " << transitionsCount;
    }
  }

  if (!errorMessage.isEmpty())
  {
    QLabel* label = new QLabel(errorMessage, m_widget);
    layout->addWidget(label);
    return;
  }

  // Control panel
  QWidget* controlWidget = new QWidget;
  QHBoxLayout* controlLayout = new QHBoxLayout;
  QPushButton* addPhaseButton = new QPushButton("Add Phase");
  QPushButton* removeSelectedButton = new QPushButton("Remove Selected");
  controlWidget->setLayout(controlLayout);
  controlLayout->addWidget(addPhaseButton);
  controlLayout->addWidget(removeSelectedButton);
  controlLayout->addStretch();
  QObject::connect(addPhaseButton, &QPushButton::clicked, this, &qtMaterialItem::addPhase);
  QObject::connect(
    removeSelectedButton, &QPushButton::clicked, this, &qtMaterialItem::removeSelected);

  layout->addWidget(controlWidget);

  this->Internals->m_dataWidget = this->createDataWidget();

  layout->addWidget(this->Internals->m_dataWidget);
}

QWidget* qtMaterialItem::createElementWidget(
  smtk::attribute::GroupItemPtr groupItem, std::size_t element, const QString& labelString)
{
  QWidget* frame = nullptr;
  int verticalPadding = 6; // for esthetics (multiple-phase only)
  bool isSinglePhase = groupItem->name() == "phases" && groupItem->numberOfGroups() == 1;
  if (isSinglePhase)
  {
    // Display single-phase material with simple frame
    frame = new QFrame();
  }
  else
  {
    // Display multiple-phase material with collapsible frames
    auto collapsibleFrame = new ctkCollapsibleButton();
    collapsibleFrame->setVerticalPadding(verticalPadding); // set this *before* text
    collapsibleFrame->setContentsFrameShape(QFrame::StyledPanel);
    collapsibleFrame->setCollapsedHeight(0);
    collapsibleFrame->setText(labelString);
    collapsibleFrame->setCollapsed(true);

    // Alternate row colors
    QPalette pal = collapsibleFrame->palette();
    QString hex = groupItem->name() == "phases" ? "#a0a0a0" : "#808080";
    pal.setColor(QPalette::Button, QColor(hex));
    collapsibleFrame->setAutoFillBackground(true);
    collapsibleFrame->setPalette(pal);
    collapsibleFrame->update();

    frame = collapsibleFrame;
  }

  auto layout = new QVBoxLayout(frame);
  // Adjust margin to match vertical padding - dont know why this is needed
  auto margins = layout->contentsMargins();
  int top = margins.top();
  margins.setTop(top + verticalPadding);
  layout->setContentsMargins(margins);

  const std::size_t numItems = groupItem->numberOfItemsPerGroup();
  for (std::size_t i = 0; i < numItems; i++)
  {
    auto item = groupItem->item(element, static_cast<int>(i));
    if (isSinglePhase && item->name() == "name")
    {
      // Omit phase name for single-phase materials
      continue;
    }

    smtk::view::View::Component comp; // create a default style (an empty component)
    qtItemInfo info(item, comp, m_widget, m_itemInfo.baseView());
    qtItem* childItem = m_itemInfo.uiManager()->createItem(info);
    if (childItem)
    {
      layout->addWidget(childItem->widget());
      if (!isSinglePhase)
      {
        if (item->name() == "name")
        {

          QObject::connect(childItem, &qtItem::modified, [frame, element, childItem]() {
            auto* dataFrame = static_cast<ctkCollapsibleButton*>(frame);
            QString title;
            QTextStream qs(&title);
            qs << "Phase" << element + 1 << ": "
               << childItem->itemAs<smtk::attribute::StringItem>()->value().c_str();
            dataFrame->setText(title);
          });
        }
        QObject::connect(childItem, SIGNAL(modified()), this, SLOT(checkValidity()));
      }
      // Todo
      // itemList.push_back(childItem);
      // connect(childItem, SIGNAL(modified()), this, SLOT(onChildItemModified()));
    }
  }
  return frame;
}

void qtMaterialItem::updateTableItemSizes()
{
  this->Internals->m_tableWidget->resizeRowsToContents();
  this->Internals->m_tableWidget->resizeColumnsToContents();
}

void qtMaterialItem::setTableRowFromPhase(int row, size_t element)
{
  QString title;
  QTextStream qs(&title);
  auto nameItem =
    this->Internals->m_phasesItem->findAs<smtk::attribute::StringItem>(element, "name");
  qs << "Phase" << element + 1 << ": " << nameItem->value().c_str();
  auto* phaseWidget = this->createElementWidget(this->Internals->m_phasesItem, element, title);
  this->setTableRowWidgets(row, phaseWidget);
}

void qtMaterialItem::setTableRowFromTransition(int row, size_t element)
{
  QString title = "transition";
  auto* transitionWidget =
    this->createElementWidget(this->Internals->m_transitionsItem, element, title);
  this->setTableRowWidgets(row, transitionWidget);
}

void qtMaterialItem::setTableRowWidgets(int row, QWidget* widget)
{
  if (row >= this->Internals->m_tableWidget->rowCount())
  {
    this->Internals->m_tableWidget->insertRow(row);
  }

  auto* selectBoxWidget = new QWidget();
  auto* selectBoxLayout = new QVBoxLayout(selectBoxWidget);
  auto* selectBox = new QCheckBox();
  selectBoxLayout->addWidget(selectBox);
  selectBoxLayout->setAlignment(Qt::AlignCenter);
  selectBoxLayout->setContentsMargins(0, 0, 0, 0);
  auto* selectBoxItem = new QTableWidgetItem();
  auto* validity = new qtMaterialMultiPhaseValidityWidget();
  auto* validityItem = new QTableWidgetItem();
  auto* moveButton = new qtMaterialMultiPhaseMoveButtons(nullptr, row,
    static_cast<size_t>(row) == 2 * (this->Internals->m_phasesItem->numberOfGroups() - 1));
  auto* moveButtonItem = new QTableWidgetItem();
  auto* tableItem = new QTableWidgetItem();

  selectBoxItem->setSizeHint(selectBox->sizeHint());
  validityItem->setSizeHint(validity->sizeHint());
  tableItem->setSizeHint(widget->sizeHint());
  moveButtonItem->setSizeHint(moveButton->sizeHint());

  this->Internals->m_tableWidget->setItem(row, 0, selectBoxItem);
  this->Internals->m_tableWidget->setItem(row, 1, validityItem);
  this->Internals->m_tableWidget->setItem(row, 2, tableItem);
  this->Internals->m_tableWidget->setItem(row, 3, moveButtonItem);
  this->Internals->m_tableWidget->setCellWidget(row, 0, selectBoxWidget);
  this->Internals->m_tableWidget->setCellWidget(row, 1, validity);
  this->Internals->m_tableWidget->setCellWidget(row, 2, widget);
  this->Internals->m_tableWidget->setCellWidget(row, 3, moveButton);

  if (row % 2 != 0)
  {
    selectBoxWidget->setVisible(false);
    selectBoxWidget->setEnabled(false);
    selectBox->setVisible(false);
    selectBox->setEnabled(false);
  }

  this->updateTableItemSizes();

  QObject::connect(static_cast<ctkCollapsibleButton*>(widget),
    &ctkCollapsibleButton::contentsCollapsed, this, &qtMaterialItem::updateTableItemSizes);
  QObject::connect(
    moveButton, &qtMaterialMultiPhaseMoveButtons::moveUp, this, &qtMaterialItem::movePhaseUp);
  QObject::connect(
    moveButton, &qtMaterialMultiPhaseMoveButtons::moveDown, this, &qtMaterialItem::movePhaseDown);
}

void qtMaterialItem::swapPhase(size_t phaseA, size_t phaseB)
{
  if (phaseA == phaseB)
  {
    return;
  }
  if (phaseA > phaseB)
  {
    std::swap(phaseA, phaseB);
  }
  auto* phaseATableWidget =
    static_cast<ctkCollapsibleButton*>(this->Internals->m_tableWidget->cellWidget(2 * phaseA, 2));
  auto* phaseBTableWidget =
    static_cast<ctkCollapsibleButton*>(this->Internals->m_tableWidget->cellWidget(2 * phaseB, 2));
  bool phaseACollapsed = phaseATableWidget->collapsed();
  bool phaseBCollapsed = phaseBTableWidget->collapsed();

  this->Internals->m_phasesItem->rotate(phaseA, phaseB);
  if (phaseB - phaseA > 1)
  {
    this->Internals->m_phasesItem->rotate(phaseB - 1, phaseA);
  }

  setTableRowFromPhase(2 * static_cast<int>(phaseA), phaseA);
  setTableRowFromPhase(2 * static_cast<int>(phaseB), phaseB);

  setRowCollapsed(2 * static_cast<int>(phaseB), phaseACollapsed);
  setRowCollapsed(2 * static_cast<int>(phaseA), phaseBCollapsed);
}

void qtMaterialItem::setTransitionToDefault(int row)
{
  assert(row % 2 != 0);
  auto transitionsItem = this->Internals->m_transitionsItem;
  size_t element = static_cast<size_t>(row) / 2;
  for (size_t i = 0; i < transitionsItem->numberOfItemsPerGroup(); i++)
  {
    auto item = transitionsItem->item(element, i);
    std::static_pointer_cast<smtk::attribute::ValueItem>(item)->setToDefault();
  }
  this->setTableRowFromTransition(row, element);
}

void qtMaterialItem::setTransitionsToDefault(const std::vector<int>& rows)
{
  for (int row : rows)
  {
    if (row >= 0 && row < this->Internals->m_tableWidget->rowCount())
    {
      setTransitionToDefault(row);
    }
  }
}

QWidget* qtMaterialItem::createDataWidget()
{
  int phasesCount = this->Internals->m_phasesItem->numberOfGroups();
  if (phasesCount == 1)
  {
    QString title;
    QTextStream qs(&title);

    return this->createElementWidget(this->Internals->m_phasesItem, 0, title);
  }
  else
  {
    auto& tableWidget = this->Internals->m_tableWidget;
    if (tableWidget != nullptr)
    {
      tableWidget->setRowCount(0);
    }
    else
    {
      tableWidget = new QTableWidget;
      tableWidget->setEditTriggers(QAbstractItemView::NoEditTriggers);
      tableWidget->setFocusPolicy(Qt::NoFocus);
      tableWidget->setSelectionMode(QAbstractItemView::NoSelection);

      tableWidget->setColumnCount(4);
      tableWidget->setHorizontalHeaderLabels({ "Select", "Valid", "Phase/Transition", "Move" });
    }

    int row = 0;
    setTableRowFromPhase(row, 0);
    row++;
    for (std::size_t i = 0; i < this->Internals->m_transitionsItem->numberOfGroups(); ++i)
    {
      setTableRowFromTransition(row, i);
      row++;
      setTableRowFromPhase(row, i + 1);
      row++;
    }
    return tableWidget;
  }
}

void qtMaterialItem::recreateDataWidget()
{
  if (this->Internals->m_dataWidget != nullptr)
  {

    delete this->Internals->m_dataWidget;
    this->Internals->m_dataWidget = nullptr;
    this->Internals->m_tableWidget = nullptr;
    this->Internals->m_dataWidget = this->createDataWidget();
    this->m_widget->layout()->addWidget(this->Internals->m_dataWidget);
  }
}

void qtMaterialItem::movePhaseDown(int row)
{
  assert(row % 2 == 0);
  size_t phase = static_cast<size_t>(row) / 2;
  this->swapPhase(phase, phase + 1);
  std::vector<int> affectedTransitions;
  affectedTransitions.push_back(row - 1);
  affectedTransitions.push_back(row + 1);
  affectedTransitions.push_back(row + 3);
  setTransitionsToDefault(affectedTransitions);
}

void qtMaterialItem::movePhaseUp(int row)
{
  assert(row % 2 == 0);
  size_t phase = static_cast<size_t>(row) / 2;
  this->swapPhase(phase, phase - 1);
  std::vector<int> affectedTransitions;
  affectedTransitions.push_back(row + 1);
  affectedTransitions.push_back(row - 1);
  affectedTransitions.push_back(row - 3);
  setTransitionsToDefault(affectedTransitions);
}

void qtMaterialItem::addPhase()
{
  auto* internals = this->Internals;
  internals->m_phasesItem->appendGroup();
  internals->m_transitionsItem->appendGroup();
  if (internals->m_phasesItem->numberOfGroups() == 2)
  {
    this->recreateDataWidget();
  }
  else
  {
    int row = internals->m_tableWidget->rowCount();
    auto* lastRowMoveButtons = static_cast<qtMaterialMultiPhaseMoveButtons*>(
      internals->m_tableWidget->cellWidget(row - 1, 3));
    lastRowMoveButtons->setLastRow(false);
    this->setTableRowFromTransition(row, internals->m_phasesItem->numberOfGroups() - 2);
    this->setTableRowFromPhase(row + 1, internals->m_phasesItem->numberOfGroups() - 1);
  }
}

void qtMaterialItem::removeSelected()
{
  auto* table = this->Internals->m_tableWidget;
  std::vector<int> selectedRows;
  for (int i = 0; i < table->rowCount(); i++)
  {
    auto* checkBox =
      static_cast<QCheckBox*>(table->cellWidget(i, 0)->layout()->itemAt(0)->widget());
    if (checkBox->isChecked())
    {
      selectedRows.push_back(i);
    }
  }
  if (!selectedRows.size())
  {
    return;
  }
  for (int i = 0; i < selectedRows.size(); i++)
  {
    size_t phase = static_cast<size_t>(selectedRows[i]) / 2;
    this->Internals->m_phasesItem->removeGroup(phase);
    if (selectedRows[i] == 0)
    {
      if (selectedRows[i + 1] != 2)
      {
        this->Internals->m_transitionsItem->removeGroup(0);
      }
    }
    else
    {
      this->Internals->m_transitionsItem->removeGroup(phase - 1);
    }
  }
  this->recreateDataWidget();
}

void qtMaterialItem::setRowCollapsed(int row, bool collapsed)
{
  auto* collapsibleButton =
    static_cast<ctkCollapsibleButton*>(this->Internals->m_tableWidget->cellWidget(row, 2));
  collapsibleButton->setCollapsed(collapsed);
}

void qtMaterialItem::checkValidity()
{
  auto* table = this->Internals->m_tableWidget;
  for (int i = 0; i < table->rowCount(); i++)
  {
    auto* validityWidget =
      static_cast<qtMaterialMultiPhaseValidityWidget*>(table->cellWidget(i, 1));
    size_t element = static_cast<size_t>(i) / 2;
    auto group = i % 2 ? this->Internals->m_transitionsItem : this->Internals->m_phasesItem;
    bool internalValid = true;
    bool transitionValid = true;
    for (size_t j = 0; j < group->numberOfItemsPerGroup(); j++)
    {
      //      std::cout << group->item(element, j)->name() << std::endl;
      if (!group->item(element, j)->isValid())
      {
        //        std::cout << "invalid!" << std::endl;
        internalValid = false;
        break;
      }
    }
    if (group == this->Internals->m_transitionsItem)
    {
      double highTemp =
        group->findAs<smtk::attribute::DoubleItem>(element, "upper-transition-temperature")
          ->value();
      double lowTemp =
        group->findAs<smtk::attribute::DoubleItem>(element, "lower-transition-temperature")
          ->value();
      if (lowTemp > highTemp)
      {
        transitionValid = false;
      }
    }
    validityWidget->setValid(qtMaterialMultiPhaseValidityWidget::Internal, internalValid);
    validityWidget->setValid(qtMaterialMultiPhaseValidityWidget::Transition, transitionValid);
  }
}
