//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#ifndef smtk_extension_qt_InvokeOnMainThreadBehavior_h
#define smtk_extension_qt_InvokeOnMainThreadBehavior_h

#include "smtk/extension/qt/Exports.h"

#include <QApplication>
#include <QObject>
#include <QThread>

#include <smtk/operation/Observer.h>
#include <smtk/resource/Observer.h>
#include <smtk/view/SelectionObserver.h>

namespace smtk
{
namespace extension
{

class qtInvokeOnMainThreadBehaviorPrivate;

class SMTKQTEXT_EXPORT qtInvokeOnMainThreadBehavior : public QObject
{
  Q_OBJECT
    using Superclass = QObject;
public:
  static qtInvokeOnMainThreadBehavior* instance(QObject* parent = nullptr);
  ~qtInvokeOnMainThreadBehavior() override;

  template<typename Func, typename... Args>
  static auto invokeOnMainThread(Func&& fn, Args&&... args) -> std::enable_if_t<
    // Non-void returning function specialization.
    !std::is_void_v<decltype(fn(std::forward<Args>(args)...))>,
    decltype(fn(
      std::forward<Args>(args)...))>
  {
    using ReturnType = decltype(fn(std::forward<Args>(args)...));
    ReturnType result;

    if (qApp->thread() != QThread::currentThread())
    {
      // Not on main thread. Run the function on the main thread while this one blocks until it
      // returns.
      QMetaObject::invokeMethod(
        qApp,
        [&]() { result = std::invoke(std::forward<Func>(fn), std::forward<Args>(args)...); },
        Qt::BlockingQueuedConnection);
    }
    else
    {
      // Call on this thread directly.
      result = std::invoke(std::forward<Func>(fn), std::forward<Args>(args)...);
    }

    return result;
  }

  template<typename Func, typename... Args>
  static auto invokeOnMainThread(Func&& fn, Args&&... args)
    // Void returning function specialization.
    -> std::enable_if_t<std::is_void_v<decltype(fn(std::forward<Args>(args)...))>, void>
  {
    if (qApp->thread() != QThread::currentThread())
    {
      // Not on main thread. Run the function on the main thread while this one blocks until it
      // returns.
      QMetaObject::invokeMethod(
        qApp,
        [&]() { std::invoke(std::forward<Func>(fn), std::forward<Args>(args)...); },
        Qt::BlockingQueuedConnection);
    }
    else
    {
      // Call on this thread directly.
      std::invoke(std::forward<Func>(fn), std::forward<Args>(args)...);
    }
  }

  void addObservers(const smtk::common::Managers& managers) const;
  smtk::resource::Observers& resourceObservers() const;
  smtk::operation::Observers& operationObservers() const;
  smtk::view::SelectionObservers& selectionObservers() const;

protected:
  explicit qtInvokeOnMainThreadBehavior(QObject* parent = nullptr);

private:
  Q_DISABLE_COPY(qtInvokeOnMainThreadBehavior);

  qtInvokeOnMainThreadBehaviorPrivate *m_private;
};

}
}

#endif //smtk_extension_qt_InvokeOnMainThreadBehavior_h
