//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#include "smtk/extension/qt/qtInvokeOnMainThreadBehavior.h"

#include "smtk/operation/Manager.h"
#include "smtk/resource/Manager.h"
#include "smtk/view/Selection.h"

#include <unordered_map>

namespace smtk
{
namespace extension
{

static qtInvokeOnMainThreadBehavior* g_instance = nullptr;

class qtInvokeOnMainThreadBehaviorPrivate : public QObject
{
public:
  explicit qtInvokeOnMainThreadBehaviorPrivate(QObject* parent)
    : QObject(parent)
  {
  }

  smtk::resource::Observers m_resourceObservers;
  smtk::operation::Observers m_operationObservers;
  smtk::view::SelectionObservers m_selectionObservers;

  std::unordered_map<smtk::resource::ManagerPtr, smtk::resource::Observers::Key>
    m_resourceObserverKeys;
  std::unordered_map<smtk::operation::ManagerPtr, smtk::operation::Observers::Key>
    m_operationObserverKeys;
  std::unordered_map<smtk::view::SelectionPtr, smtk::view::SelectionObservers::Key>
    m_selectionObserverKeys;
};

qtInvokeOnMainThreadBehavior* qtInvokeOnMainThreadBehavior::instance(QObject* parent)
{
  if (!g_instance)
  {
    g_instance = new qtInvokeOnMainThreadBehavior(parent);
  }

  if (g_instance->parent() == nullptr && parent)
  {
    g_instance->setParent(parent);
  }

  return g_instance;
}
qtInvokeOnMainThreadBehavior::~qtInvokeOnMainThreadBehavior() {}

smtk::resource::Observers& qtInvokeOnMainThreadBehavior::resourceObservers() const
{
  return this->m_private->m_resourceObservers;
}

smtk::operation::Observers&
qtInvokeOnMainThreadBehavior::qtInvokeOnMainThreadBehavior::operationObservers() const
{
  return this->m_private->m_operationObservers;
}

smtk::view::SelectionObservers&
qtInvokeOnMainThreadBehavior::qtInvokeOnMainThreadBehavior::selectionObservers() const
{
  return this->m_private->m_selectionObservers;
}

qtInvokeOnMainThreadBehavior::qtInvokeOnMainThreadBehavior(QObject* parent)
  : Superclass(parent)
  , m_private(new qtInvokeOnMainThreadBehaviorPrivate(this))
{
}

void qtInvokeOnMainThreadBehavior::addObservers(const smtk::common::Managers& managers) const
{
  const auto resourceManager = managers.get<smtk::resource::ManagerPtr>();
  const auto operationManager = managers.get<smtk::operation::ManagerPtr>();
  const auto selection = managers.get<smtk::view::SelectionPtr>();

  // Add delegating observers as lowest priority so the GUI is updated last.
  this->m_private->m_resourceObserverKeys[resourceManager] = resourceManager->observers().insert(
    [this](const smtk::resource::Resource& rsrc, smtk::resource::EventType event) {
      invokeOnMainThread(
        [&]() { this->m_private->m_resourceObservers.callObserversDirectly(rsrc, event); });
    },
    INT_MIN,
    false,
    "qtInvokeOnMainThreadBehavior resource observer");

  this->m_private->m_operationObserverKeys[operationManager] = operationManager->observers().insert(
    [&](
      smtk::common::Context ctx,
      const smtk::operation::Operation& oper,
      smtk::operation::EventType event,
      smtk::operation::Operation::Result result) -> int {
      return invokeOnMainThread([&]() -> int {
        return this->m_private->m_operationObservers.callObserversDirectly(
          ctx, oper, event, result);
      });
    },
    INT_MIN,
    false,
    "qtInvokeOnMainThreadBehavior operation observer");

  this->m_private->m_selectionObserverKeys[selection] = selection->observers().insert(
    [&](const std::string& str, smtk::view::Selection::Ptr selection) {
      invokeOnMainThread(
        [&]() { this->m_private->m_selectionObservers.callObserversDirectly(str, selection); });
    },
    INT_MIN,
    false,
    "qtInvokeOnMainThreadBehavior selection observer");
}

} // namespace extension
} // namespace smtk
