//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#ifndef smtk_common_Context_h
#define smtk_common_Context_h

#include "smtk/CoreExports.h"

//#include <optional>
#include <typeindex>
#include <unordered_map>

#include <boost/any.hpp>

// TODO: Replace this with std::optional before merging this.
#include <boost/optional.hpp>

#define DECLARE_CONTEXT_KEY_TYPE(NAME)                                                                     \
struct NAME                                                                                      \
{                                                                                                \
};

namespace smtk
{
namespace common
{

class SMTKCORE_EXPORT Context final
{
  using key_type = std::size_t;
  using value_type = boost::any;
  using value_map = std::unordered_map<key_type, value_type>;

public:
  template<typename keyT, typename valueT>
  boost::optional<valueT> value(const keyT& key) const
  {
    const auto keyHash = std::hash<keyT>{}(key);
    const auto it = this->m_valueEntries.find(keyHash);
    if (it != this->m_valueEntries.end())
    {
      try
      {
        return boost::any_cast<valueT>(it->second);
      }
      catch (const boost::bad_any_cast&)
      {
        return boost::optional<valueT>();
      }
    }
    return boost::optional<valueT>();
  }

  template<typename keyT, typename valueT>
  boost::optional<valueT> value() const
  {
    const auto keyHash = std::hash<std::type_index>{}(std::type_index(typeid(keyT)));
    const auto it = this->m_typeEntries.find(keyHash);
    if (it != this->m_typeEntries.end())
    {
      try
      {
        return boost::any_cast<valueT>(it->second);
      }
      catch (const boost::bad_any_cast&)
      {
        return boost::optional<valueT>();
      }
    }
    return boost::optional<valueT>();
  }

  template<typename keyT, typename valueT>
  Context withValue(const keyT& key, const valueT& value) const
  {
    // Create a copy of the current context.
    Context result = *this;

    // Add the entry to the copy.
    result.m_valueEntries[std::hash<keyT>{}(key)] = value;

    // Return the copy.
    return result;
  }

  template<typename keyT, typename valueT>
  Context withValue(const valueT& value) const
  {
    // Create a copy of the current context.
    Context result = *this;

    // Add the entry to the copy using the type id of keyT as key value.
    result.m_typeEntries[std::hash<std::type_index>{}(std::type_index(typeid(keyT)))] = value;

    // Return the copy.
    return result;
  }

private:
  value_map m_valueEntries;
  value_map m_typeEntries;
};

} // namespace common
} // namespace smtk

#endif // smtk_common_Context_h