import os
import sys

import smtk
import smtk.attribute
import smtk.io
import smtk.mesh
import smtk.model
import smtk.operation
import smtk.resource
import smtk.session.mesh
import smtk.session.polygon

# On macOS, importing xmsmesh before smtkxmsmesh works around
# an error loading _smtkPybindXMSMesh.so. The load fails due
# to missing symbol boost::archive_exception::~archive_exception()
import xmsmesh
import smtkxmsmesh

script_dir = os.path.abspath(os.path.dirname(__file__))
DATA_DIR =  os.path.join(script_dir, os.pardir, 'data')

class PolygonMesher:
    """Uses xmsmesh to generate mesh from polygon model"""
    def __init__(self):
        self.res_manager = smtk.resource.Manager.create()
        self.op_manager = smtk.operation.Manager.create()

        # Register polygon session
        smtk.session.polygon.Registrar.registerTo(self.res_manager)
        smtk.session.polygon.Registrar.registerTo(self.op_manager)

        # Register mesh.xms session
        smtk.mesh.xms.Registrar.registerTo(self.op_manager)

        smtk.operation.Registrar.registerTo(self.op_manager)
        self.op_manager.registerResourceManager(self.res_manager)

    def load_model(self, filename):
        """Loads smtk file storing polygon resource with 1 model"""
        read_op = smtk.session.polygon.LegacyRead.create()
        if not read_op:
            raise RuntimeError('No read operator')

        read_op.parameters().find('filename').setValue(filename)
        read_result = read_op.operate()
        if read_result.findInt('outcome').value(0) != int(smtk.operation.Operation.SUCCEEDED):
          raise RuntimeError('Error reading input file')

        # Access the resource
        # FYI no py binding for smtk::polygon::Resource, but doesn't seem to be needed
        resource = smtk.model.Resource.CastTo(read_result.find('resource').value())

        # Access the model
        model_ents = resource.entitiesMatchingFlags(smtk.model.MODEL_ENTITY, True)
        model_ent = model_ents.pop()
        model = smtk.model.Model(resource, model_ent)

        self.resource = resource
        return model

    def generate_face_mesh(self, model):
        """Uses xmsmesh to generate the face mesh"""
        # Create the mesh operator
        mesh_op = smtk.mesh.xms.GenerateMesh.create()

        # Associate it to the loaded model
        associated = mesh_op.parameters().associateEntity(model)
        if not associated:
            raise RuntimeError('Associate returned false')

        # Execute the operation
        res = mesh_op.operate()

        # Test for success
        outcome = res.find('outcome').value()
        if outcome != int(smtk.operation.Operation.Outcome.SUCCEEDED):
            print(mesh_op.log().convertToString())
            raise RuntimeError('"generate mesh" operator failed, outcome: ', outcome)

        # Print an overview of the output mesh
        mesh_collection = res.find('resource').value()
        return mesh_collection


if __name__ == '__main__':
    mesher = PolygonMesher()

    model_file = os.path.join(DATA_DIR, 'boxWithHole.smtk')
    print('loading model file', model_file)
    model = mesher.load_model(model_file)

    print('generating face mesh')
    mesh_collection  = mesher.generate_face_mesh(model)
    # print('meshes: %d\tcells: %d\tpoints: %d' % (
    #     mesh_collection.meshes().size(),
    #     mesh_collection.cells().size(),
    #     mesh_collection.points().size()))
    # print('3d cells: %d\t2d cells: %d\t1d cells: %d' % (
    #     mesh_collection.cells(smtk.mesh.Dims3).size(),
    #     mesh_collection.cells(smtk.mesh.Dims2).size(),
    #     mesh_collection.cells(smtk.mesh.Dims1).size()))

    mesh_filename = 'boxWithHole.2dm'
    print('exporting mesh file', mesh_filename)
    if not smtk.io.exportMesh(mesh_filename, mesh_collection):
        raise RuntimeError('Error exporting 2dm file')

    print('finis')
