//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

// .NAME vtkModelItemListIterator - Class for iterating over associated vtkModelItems
// .SECTION Description
// Abstract class for iterating over associated objects of a vtkModelItem.
// The intended use is:
// \code
//   vtkModelItemListIterator* iterator = ModelItem->NewIterator(itemType);
//   for(iterator->Begin();!iterator->IsAtEnd();iterator->Next())
//     {
//       vtkModelItem* associatedItem = iterator->GetCurrentItem();
//     }
//   iterator->Delete();
// \endcode
// Note that the iterator must be deleted when the user is done
// using it to avoid a memory leak.

#ifndef __smtkdiscrete_vtkModelItemListIterator_h
#define __smtkdiscrete_vtkModelItemListIterator_h

#include "smtk/bridge/discrete/kernel/vtkSMTKDiscreteModelModule.h" // For export macro
#include "vtkModelItemIterator.h"

class vtkModelItem;
struct vtkModelItemListIteratorInternals;

class VTKSMTKDISCRETEMODEL_EXPORT vtkModelItemListIterator : public vtkModelItemIterator
{
public:
  static vtkModelItemListIterator* New();
  vtkTypeMacro(vtkModelItemListIterator, vtkModelItemIterator);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  // Description:
  // Set/get functions for Root.  Root is set by the function that
  // creates the iterator.
  void SetRoot(vtkModelItem* item);
  vtkGetMacro(Root, vtkModelItem*);

  // Description:
  // Set/get functions for the itemType to iterate over.  This is set by the function
  // that creates the iterator.
  void SetItemType(int itemType);
  int GetItemType();

  // Description:
  // Go to the first item with given type.
  void Begin() override;

  // Description:
  // Is the iterator at the end of the list.
  int IsAtEnd() override;

  // Description:
  // Move to the next iterator.
  void Next() override;

  // Description:
  // Returns the current item.
  vtkModelItem* GetCurrentItem() override;

  // Description:
  // Returns the number of items being iterated over.
  int Size() override;

protected:
  vtkModelItemListIterator();
  ~vtkModelItemListIterator() override;

private:
  vtkModelItemListIterator(const vtkModelItemListIterator&); // Not implemented.
  void operator=(const vtkModelItemListIterator&);           // Not implemented.

  // Description:
  // Root is the object that will have its associated model entities
  // iterated over.
  vtkModelItem* Root;
  vtkModelItemListIteratorInternals* Internal;
};

#endif
