//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#include "smtk/simulation/ace3p/qt/qtNerscFileItem.h"

// Plugin includes
#include "smtk/newt/qtNewtFileBrowserDialog.h"
#include "smtk/newt/qtNewtInterface.h"
#include "smtk/simulation/ace3p/qt/qtProjectRuntime.h"

// SMTK includes
#include "smtk/attribute/Item.h"
#include "smtk/attribute/StringItem.h"
#include "smtk/extension/qt/qtBaseAttributeView.h"
#include "smtk/extension/qt/qtUIManager.h"

#include <QDebug>
#include <QFrame>
#include <QHBoxLayout>
#include <QHeaderView>
#include <QLabel>
#include <QLineEdit>
#include <QMessageBox>
#include <QPushButton>
#include <QString>
#include <QTableWidget>

namespace smtk
{
namespace simulation
{
namespace ace3p
{

class qtNerscFileItem::Internal
{
public:
  smtk::attribute::StringItemPtr m_item;

  QLineEdit* m_lineEdit = nullptr;
  newt::qtNewtInterface* m_newt = newt::qtNewtInterface::instance();
};

smtk::extension::qtItem* qtNerscFileItem::createItemWidget(
  const smtk::extension::qtAttributeItemInfo& info)
{
  return new qtNerscFileItem(info);
}

qtNerscFileItem::qtNerscFileItem(const smtk::extension::qtAttributeItemInfo& info)
  : qtItem(info)
{
  this->setObjectName("NERSC File Item");
  m_internal = new qtNerscFileItem::Internal;
  m_internal->m_item = info.itemAs<smtk::attribute::StringItem>();
  m_internal->m_lineEdit = new QLineEdit(m_widget);
  this->createWidget();
}

qtNerscFileItem::~qtNerscFileItem()
{
  delete m_internal;
}

void qtNerscFileItem::clearChildWidgets()
{
  m_internal->m_lineEdit->clear();
}

void qtNerscFileItem::createWidget()
{
  smtk::attribute::ItemPtr item = m_itemInfo.item();
  auto iview = m_itemInfo.baseView();
  if (iview && !iview->displayItem(item))
  {
    return;
  }

  this->clearChildWidgets();
  this->updateUI();
}

void qtNerscFileItem::setBackground()
{
  if (m_internal->m_item->isValid() && !m_internal->m_item->value().empty())
  {
    this->uiManager()->setWidgetColorToNormal(m_internal->m_lineEdit);
  }
  else
  {
    this->uiManager()->setWidgetColorToInvalid(m_internal->m_lineEdit);
  }
}

void qtNerscFileItem::updateUI()
{
  auto item = m_itemInfo.item();
  auto iview = m_itemInfo.baseView();
  if (iview && !iview->displayItem(item))
  {
    return;
  }

  if (m_widget)
  {
    delete m_widget;
  }

  m_widget = new QFrame(m_itemInfo.parentWidget());
  QHBoxLayout* layout = new QHBoxLayout(m_widget);
  layout->setMargin(0);

  if (m_internal->m_item == nullptr)
  {
    QLabel* label = new QLabel(item->label().c_str(), m_widget);
    layout->addWidget(label);
    QString text("Invalid NERSCDIRECTORY ItemView NOT assigned to StringItem");
    QLabel* msg = new QLabel(text, m_widget);
    layout->addWidget(msg);
    return;
  }

  QLabel* label = new QLabel(item->label().c_str(), m_widget);
  QFontMetrics metric(label->font());
  label->setFixedWidth(metric.boundingRect(label->text()).width() + 2);
  label->setWordWrap(true);
  label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
  layout->addWidget(label);

  if (qtProjectRuntime::instance()->ace3pProject())
  {
    QIcon arrowIcon(":/icons/arrow-up.svg");
    QPushButton* getResultsPathButton = new QPushButton(arrowIcon, "", m_widget);
    getResultsPathButton->setToolTip("Select from upstream stage");
    layout->addWidget(getResultsPathButton);
    QObject::connect(
      getResultsPathButton, &QPushButton::clicked, this, &qtNerscFileItem::onGetResultsPath);
  }

  m_internal->m_lineEdit = new QLineEdit(m_widget);
  layout->addWidget(m_internal->m_lineEdit);
  QPushButton* browseButton = new QPushButton("Browse", m_widget);
  layout->addWidget(browseButton);

  QObject::connect(
    m_internal->m_lineEdit, &QLineEdit::editingFinished, this, &qtNerscFileItem::onEditingFinished);
  QObject::connect(browseButton, &QPushButton::clicked, this, &qtNerscFileItem::onBrowse);

  if (m_internal->m_item->isSet())
  {
    m_internal->m_lineEdit->setText(m_internal->m_item->value().c_str());
  }
  this->setBackground();
}

void qtNerscFileItem::onBrowse()
{
  // First check if user is signed in to NERSC
  if (!m_internal->m_newt->isLoggedIn())
  {
    QMessageBox::information(
      this->parentWidget(), "Not Signed in", "You must first log into NERSC from the application.");
    return;
  }

  // Instantiate browser
  newt::qtNewtFileBrowserDialog dialog(this->parentWidget());
  QObject::connect(
    &dialog, &newt::qtNewtFileBrowserDialog::applyPath, this, &qtNerscFileItem::onDialogApply);
  dialog.exec();
}

void qtNerscFileItem::onGetResultsPath()
{
  auto project = qtProjectRuntime::instance()->ace3pProject();
  if (project == nullptr)
  {
    return;
  }
  int currentStageIndex = project->currentStageIndex();
  std::shared_ptr<smtk::simulation::ace3p::JobsManifest> manifest = project->jobsManifest();

  struct ResultPath
  {
    QString jobName;
    QString analysisType;
    QString path;
  };
  QVector<ResultPath> results;

  // get the list of all upstream Job results
  for (int i = 0; i < currentStageIndex; i++)
  {
    std::shared_ptr<Stage> stage = project->stage(i);
    std::shared_ptr<smtk::attribute::Resource> attResource = stage->attributeResource();
    std::string stageID = attResource->id().toString();

    for (int j = 0; j < manifest->size(); j++)
    {
      std::string analysisID;
      manifest->getField(j, "analysis_id", analysisID);

      if (stageID == analysisID)
      {
        std::string name, analysis, path1, path2;
        manifest->getField(j, "job_name", name);
        manifest->getField(j, "analysis", analysis);
        manifest->getField(j, "runtime_job_folder", path1);
        manifest->getField(j, "results_subfolder", path2);
        std::string path = path1 + "/" + path2;

        ResultPath result;
        result.jobName = QString().fromStdString(name);
        result.analysisType = QString().fromStdString(analysis);
        result.path = QString().fromStdString(path);
        results.push_back(result);
      }
    }
  }

  // first, handle the trivial cases
  if (results.size() == 0)
  {
    QMessageBox::warning(
      this->parentWidget(),
      "No Jobs",
      "There are no jobs from prior stages present in the Project.");
    return;
  }

  // display all Job results options for user selection
  QWidget* selectionWindow = new QWidget(qtProjectRuntime::instance()->mainWidget());
  selectionWindow->setWindowFlags(selectionWindow->windowFlags() | Qt::Window);
  selectionWindow->setWindowModality(Qt::WindowModal);
  selectionWindow->setWindowTitle("Job Results");
  QVBoxLayout* mainLayout = new QVBoxLayout();

  QTableWidget* tableWidget = new QTableWidget(selectionWindow);
  tableWidget->setSelectionBehavior(QAbstractItemView::SelectRows);
  tableWidget->setSelectionMode(QAbstractItemView::SingleSelection);
  tableWidget->setEditTriggers(QAbstractItemView::NoEditTriggers);
  tableWidget->setColumnCount(3);
  tableWidget->setRowCount(results.size());
  tableWidget->setHorizontalHeaderLabels(QStringList{ "Job Name", "ACE3P Code", "Results Path" });
  tableWidget->verticalHeader()->setVisible(false);
  tableWidget->horizontalHeader()->setSectionResizeMode(0, QHeaderView::ResizeToContents);
  tableWidget->horizontalHeader()->setSectionResizeMode(1, QHeaderView::ResizeToContents);
  tableWidget->horizontalHeader()->setSectionResizeMode(2, QHeaderView::Stretch);
  for (int i = 0; i < results.size(); i++)
  {
    tableWidget->setItem(i, 0, new QTableWidgetItem(results[i].jobName));
    tableWidget->setItem(i, 1, new QTableWidgetItem(results[i].analysisType));
    tableWidget->setItem(i, 2, new QTableWidgetItem(results[i].path));
  }
  mainLayout->addWidget(tableWidget);

  QHBoxLayout* buttonsLayout = new QHBoxLayout();

  // setup the Cancel button
  QPushButton* cancelButton = new QPushButton(selectionWindow);
  cancelButton->setText(tr("Cancel"));
  QObject::connect(
    cancelButton, &QPushButton::clicked, [selectionWindow]() { selectionWindow->close(); });
  buttonsLayout->addWidget(cancelButton);

  buttonsLayout->addSpacerItem(
    new QSpacerItem(10, cancelButton->height(), QSizePolicy::MinimumExpanding));

  // setup the Select button
  QPushButton* selectButton = new QPushButton(selectionWindow);
  selectButton->setText(tr("Select"));
  QObject::connect(
    selectButton, &QPushButton::clicked, [this, results, tableWidget, selectionWindow]() {
      QList<QTableWidgetItem*> items = tableWidget->selectedItems();
      int index = items[0]->row();
      this->m_internal->m_lineEdit->setText(results[index].path);
      this->onEditingFinished();
      selectionWindow->close();
    });
  buttonsLayout->addWidget(selectButton);
  mainLayout->addLayout(buttonsLayout);

  selectionWindow->setLayout(mainLayout);
  selectionWindow->resize(640, 320);
  selectionWindow->show();
}

void qtNerscFileItem::onDialogApply(const QString& path)
{
  if (m_internal->m_item == nullptr)
  {
    return;
  }

  if (path.toStdString() == m_internal->m_item->value())
  {
    return;
  }

  bool isChanged = m_internal->m_item->setValue(path.toStdString());
  if (!isChanged)
  {
    qWarning() << "Failed to set NERSC Directory (path) to" << path;
    return;
  }

  m_internal->m_lineEdit->setText(path);
  this->setBackground();
  emit this->modified();
}

void qtNerscFileItem::onEditingFinished()
{
  if (m_internal->m_item == nullptr)
  {
    return;
  }

  auto inputString = m_internal->m_lineEdit->text().toStdString();
  if (m_internal->m_item->isSet())
  {
    if (inputString == m_internal->m_item->value())
    {
      return;
    }
  }
  else if (inputString.empty())
  {
    return;
  }

  bool isChanged = m_internal->m_item->setValue(inputString);
  if (!isChanged)
  {
    qWarning() << "Failed to set NERSC Directory (path) to" << inputString.c_str();
  }

  this->setBackground();
  emit this->modified();
}
} // namespace ace3p
} // namespace simulation
} // namespace smtk
