//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

// Local incluides
#include "smtk/simulation/ace3p/Metadata.h"
#include "smtk/simulation/ace3p/Registrar.h"
#include "smtk/simulation/ace3p/Project.h"
#include "smtk/simulation/ace3p/operations/Export.h"
#include "smtk/simulation/ace3p/operations/Read.h"

#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/FileItem.h"
#include "smtk/attribute/DirectoryItem.h"
#include "smtk/attribute/IntItem.h"
#include "smtk/attribute/ResourceItem.h"
#include "smtk/attribute/StringItem.h"
#include "smtk/model/Registrar.h"
#include "smtk/model/Resource.h"
#include "smtk/operation/Manager.h"
#include "smtk/operation/Operation.h"
#include "smtk/project/Manager.h"
#include "smtk/project/Project.h"
#include "smtk/project/Registrar.h"
#include "smtk/resource/Manager.h"
#include "smtk/session/vtk/Registrar.h"

#include <boost/filesystem.hpp>

#include <cassert>
#include <iostream>
#include <string>
#include <vector>

const int OP_SUCCEEDED = static_cast<int>(smtk::operation::Operation::Outcome::SUCCEEDED);

int TestExportProject(int /*argc*/, char* /*argv*/ [])
{
  // Remove output folder
  std::stringstream ss;
  ss << SCRATCH_DIR << "/cxx"
     << "/export_project";
  std::string outputFolder = ss.str();
  if (boost::filesystem::is_directory(outputFolder))
  {
    boost::filesystem::remove_all(outputFolder);
  }
  boost::filesystem::create_directories(outputFolder);

  // Create sundry managers
  smtk::resource::ManagerPtr resourceManager = smtk::resource::Manager::create();
  smtk::attribute::Registrar::registerTo(resourceManager);
  smtk::model::Registrar::registerTo(resourceManager);
  smtk::session::vtk::Registrar::registerTo(resourceManager);

  smtk::operation::ManagerPtr operationManager = smtk::operation::Manager::create();
  smtk::operation::Registrar::registerTo(operationManager);
  smtk::attribute::Registrar::registerTo(operationManager);
  smtk::session::vtk::Registrar::registerTo(operationManager);

  smtk::project::ManagerPtr projectManager =
    smtk::project::Manager::create(resourceManager, operationManager);
  smtk::project::Registrar::registerTo(projectManager);
  smtk::simulation::ace3p::Registrar::registerTo(projectManager);

  // Todo Next line is currently a no-op. Is it needed?
  smtk::project::Registrar::registerTo(operationManager);
  operationManager->registerResourceManager(resourceManager);

  // Application must set workflows directory
  smtk::simulation::ace3p::Metadata::WORKFLOWS_DIRECTORY = WORKFLOWS_SOURCE_DIR;

  // Read project from file system
  std::shared_ptr<smtk::simulation::ace3p::Project> project;
  std::shared_ptr<smtk::project::Project> projectIntermediate;
  {
    boost::filesystem::path scratchFolder(SCRATCH_DIR);
    boost::filesystem::path projectFolder = scratchFolder / "cxx" / "create_project";
    boost::filesystem::path projectFilename = projectFolder / "create_project.project.smtk";
    std::cout << "Read project file " << projectFilename.string() << std::endl;

    auto readOp = operationManager->create<smtk::simulation::ace3p::Read>();
    assert(readOp != nullptr);

    std::cout << "readOp has " << readOp->parameters()->numberOfItems() << " items\n";
    for(int i = 0; i < readOp->parameters()->numberOfItems(); ++i) {
      std::cout << "item "<< i << ": " << readOp->parameters()->item(i)->name() << "\n";
    }

    assert(readOp->parameters()->findFile("filename")->setValue(projectFilename.string()));
    auto result = readOp->operate();
    int outcome = result->findInt("outcome")->value();
    std::cout << "Read Outcome: "
              << (outcome == OP_SUCCEEDED ? "OP_SUCCEEDED" : std::to_string(outcome))
              << "\n";

    if (outcome != OP_SUCCEEDED)
    {
      std::cout << readOp->log().convertToString() << std::endl;
    }
    assert(outcome == OP_SUCCEEDED);

    smtk::attribute::ResourceItemPtr projectItem = result->findResource("resource");
    smtk::resource::ResourcePtr resource = projectItem->value();

    std::cout << "templated smtk::simulation::ace3p::Project : " << resource->isOfType<smtk::simulation::ace3p::Project>() << "\n";

    std::cout << "templated smtk::project::Project: " << resource->isOfType<smtk::project::Project>() << "\n";

    std::cout << "templated smtk::resource::Resource: " << resource->isOfType<smtk::resource::Resource>() << "\n";

    assert(resource != nullptr);
    projectIntermediate = std::dynamic_pointer_cast<smtk::project::Project>(resource);
    assert(projectIntermediate != nullptr);
    project = std::dynamic_pointer_cast<smtk::simulation::ace3p::Project>(projectIntermediate);
    assert(project != nullptr);
  }

  // Run export operator
  {
    std::cout << "Running export\n";
    auto exportOp = operationManager->create<smtk::simulation::ace3p::Export>();
    assert(exportOp->parameters()->associate(project));

    std::string analysisRole("Analysis1");
    auto attResource = project->resources().getByRole<smtk::attribute::Resource>(analysisRole);
    assert(attResource != nullptr);
    assert(exportOp->parameters()->findResource("analysis")->setValue(attResource));

    boost::filesystem::path scratchFolder(SCRATCH_DIR);
    boost::filesystem::path meshFile = scratchFolder / "cxx" / "create_project" / "assets"/ "mesh1.gen";
    std::cout << "MeshFile: " << meshFile.string() << "\n";

    exportOp->parameters()->findFile("MeshFile")->setIsEnabled(true);
    assert(exportOp->parameters()->findFile("MeshFile")->setValue(meshFile.string()));

    std::cout << "OutputFolder: " << outputFolder << "\n";
    exportOp->parameters()->findDirectory("OutputFolder")->setIsEnabled(true);
    assert(exportOp->parameters()->findDirectory("OutputFolder")->setValue(outputFolder));
    //
    std::string outFilePrefix = "foo";
    std::cout << "OutputFilePrefix: " << outFilePrefix << "\n";
    exportOp->parameters()->findString("OutputFilePrefix")->setIsEnabled(true);
    assert(exportOp->parameters()->findString("OutputFilePrefix")->setValue(outFilePrefix));


    auto result = exportOp->operate();
    int outcome = result->findInt("outcome")->value();
    std::cout << "Export Outcome: " << outcome << std::endl;
    if (outcome != OP_SUCCEEDED)
    {
      std::cout << exportOp->log().convertToString() << std::endl;
    }
    assert(outcome == OP_SUCCEEDED);
  }

  return 0;
}
