//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "pqACE3PExportBehavior.h"

#include "smtk/simulation/ace3p/operations/Export.h"
#include "smtk/simulation/ace3p/qt/qtProjectRuntime.h"
#include "smtk/simulation/ace3p/utility/AttributeUtils.h"

// SMTK
#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/FileItem.h"
#include "smtk/attribute/DirectoryItem.h"
#include "smtk/attribute/IntItem.h"
#include "smtk/attribute/Resource.h"
#include "smtk/attribute/ResourceItem.h"
#include "smtk/attribute/StringItem.h"
#include "smtk/extension/paraview/appcomponents/pqSMTKBehavior.h"
#include "smtk/extension/paraview/appcomponents/pqSMTKWrapper.h"
#include "smtk/extension/qt/qtOperationDialog.h"
#include "smtk/io/Logger.h"
#include "smtk/operation/Manager.h"
#include "smtk/project/Project.h"
#include "smtk/project/ResourceContainer.h"

// Paraview client
#include "pqActiveObjects.h"
#include "pqCoreUtilities.h"
#include "pqFileDialog.h"
#include "pqPresetDialog.h"
#include "pqServer.h"

// Qt
#include <QAction>
#include <QDebug>
#include <QGridLayout>
#include <QMessageBox>
#include <QPushButton>
#include <QSharedPointer>
#include <QSpacerItem>
#include <QString>
#include <QTextStream>
#include <QtGlobal>

#include "nlohmann/json.hpp"

#include "boost/filesystem.hpp"

#include <algorithm> // std::replace
#include <cassert>

using json = nlohmann::json;

namespace
{
const int OP_SUCCEEDED = static_cast<int>(smtk::operation::Operation::Outcome::SUCCEEDED);
}

//-----------------------------------------------------------------------------
pqACE3PExportReaction::pqACE3PExportReaction(QAction* parentObject)
  : Superclass(parentObject)
{
}

//-----------------------------------------------------------------------------
void pqACE3PExportReaction::exportProject()
{
  // Access the active server
  pqServer* server = pqActiveObjects::instance().activeServer();
  pqSMTKWrapper* wrapper = pqSMTKBehavior::instance()->resourceManagerForServer(server);
  auto opManager = wrapper->smtkOperationManager();

  // Instantiate the Export operator
  auto exportOp = opManager->create<smtk::simulation::ace3p::Export>();
  assert(exportOp != nullptr);

  auto project = qtProjectRuntime::instance()->project();
  exportOp->parameters()->associate(project);

//Check for single analysis resource
#if 0
  // Todo smtk::project::ResourceContainer::find() won't compile:
  auto attResourceSet = project->resources().find<smtk::attribute::Resource>();
#else
  // So do it brute force instead
  std::set<smtk::attribute::ResourcePtr> attResourceSet;
  for (auto iter = project->resources().begin(); iter != project->resources().end(); ++iter)
  {
    auto resource = *iter;
    if (resource->isOfType<smtk::attribute::Resource>())
    {
      auto attResource = std::dynamic_pointer_cast<smtk::attribute::Resource>(resource);
      attResourceSet.insert(attResource);
    }
  }
#endif
  if (attResourceSet.size() == 1)
  {
    auto resourceItem = exportOp->parameters()->findResource("analysis");
    const auto attResource = *attResourceSet.begin();
    resourceItem->setValue(attResource);

    // check if the user has specified an Analysis attribute
    smtk::simulation::ace3p::AttributeUtils attUtils;
    auto analysisAtt = attUtils.getAnalysisAtt(attResource);
    assert(analysisAtt != nullptr);
    smtk::attribute::ConstStringItemPtr analysisItem = analysisAtt->findString("Analysis");
    if (!analysisItem->isSet())
    {
      // Cannot export if no analysis is set so tell the user as much and exit.
      QSharedPointer<QMessageBox> noAnalysisDialog =
        QSharedPointer<QMessageBox>(new QMessageBox(pqCoreUtilities::mainWidget()));
      noAnalysisDialog->setWindowTitle("No Analysis Set!");
      noAnalysisDialog->setText("Please set an Analysis attribute before exporting.");
      auto okButton = noAnalysisDialog->addButton("Continue", QMessageBox::RejectRole);
      noAnalysisDialog->setDefaultButton(okButton);
      noAnalysisDialog->exec();
      return;
    }

#ifndef NDEBUG
    // Check for analysis attribute
    analysisAtt = attResource->findAttribute("analysis");
    std::cout << "found analysisAtt ?" << (analysisAtt != nullptr) << std::endl;
#endif
  }



  // Set default path for output file
  std::string projectLocation = project->location();
  if (!projectLocation.empty())
  {
    boost::filesystem::path projectFilePath(projectLocation);
    boost::filesystem::path projectPath = projectFilePath.parent_path();
    boost::filesystem::path outputFilePath = projectPath / "export";
    std::string outputFolder = outputFilePath.string();

    // TODO make outFilePrefix configurable
    const std::string outFilePrefix = "ace3p";

    // TODO make the mesh name match the existing mesh file name
    std::string mshName = "mesh.gen";

    boost::filesystem::path meshFilePath = projectFilePath / "assets"/ mshName;

    std::string meshFile = meshFilePath.string();

    exportOp->parameters()->findFile("MeshFile")->setIsEnabled(true);
    assert(exportOp->parameters()->findFile("MeshFile")->setValue(meshFile));

    exportOp->parameters()->findDirectory("OutputFolder")->setIsEnabled(true);
    assert(exportOp->parameters()->findDirectory("OutputFolder")->setValue(outputFolder));

    exportOp->parameters()->findString("OutputFilePrefix")->setIsEnabled(true);
    assert(exportOp->parameters()->findString("OutputFilePrefix")->setValue(outFilePrefix));
  }

  // Construct a modal dialog for the operation spec
  QSharedPointer<smtk::extension::qtOperationDialog> dialog =
    QSharedPointer<smtk::extension::qtOperationDialog>(
      new smtk::extension::qtOperationDialog(
        exportOp,
        wrapper->smtkResourceManager(),
        wrapper->smtkViewManager(),
        pqCoreUtilities::mainWidget()
      )
    );
  dialog->setObjectName("ExportACE3PDialog");
  dialog->setWindowTitle("Specify Export Properties");
  dialog->exec();

#if 0
  // Make dialog contents scrollable.
  QScrollArea* scroll = new QScrollArea(exportDialog);
  QWidget* viewport = new QWidget(exportDialog);
  scroll->setWidget(viewport);
  scroll->setWidgetResizable(true);
  QVBoxLayout* viewLayout = new QVBoxLayout(viewport);
  viewport->setLayout(viewLayout);

  // Create operation view for the export operation.
  auto uiManager = new smtk::extension::qtUIManager(
    exportOp, wrapper->smtkResourceManager(), wrapper->smtkViewManager());
  smtk::view::ConfigurationPtr vconfig = uiManager->findOrCreateOperationView();

  smtk::extension::qtOperationView* opView =
    dynamic_cast<smtk::extension::qtOperationView*>(uiManager->setSMTKView(vconfig, viewport));
  opView->showInfoButton(false);

  QVBoxLayout* layout = new QVBoxLayout(exportDialog);
  exportDialog->setLayout(layout);
  exportDialog->layout()->addWidget(scroll);
  exportDialog->resize(640, 480);
  exportDialog->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

  // Alert the user if the operation fails. Close the dialog if the operation
  // succeeds.
  smtk::operation::Operation::Result result;
  connect(opView, &smtk::extension::qtOperationView::operationExecuted,
    [=](const smtk::operation::Operation::Result& result) {
    });

  // Launch the modal dialog and wait for the operation to succeed.
  exportDialog->exec();
  delete uiManager;
  exportDialog->deleteLater();
#endif
} // exportProject()

//-----------------------------------------------------------------------------
void pqACE3PExportBehavior::onOperationExecuted(
  const smtk::operation::Operation::Result& result)
{
  if (result->findInt("outcome")->value() != OP_SUCCEEDED)
  {
    QMessageBox msgBox(pqCoreUtilities::mainWidget());
    msgBox.setStandardButtons(QMessageBox::Ok);
    // Create a spacer so it doesn't look weird
    QSpacerItem* horizontalSpacer =
      new QSpacerItem(300, 0, QSizePolicy::Minimum, QSizePolicy::Expanding);
    msgBox.setText("Export failed. See the \"Output Messages\" view for more details.");
    QGridLayout* layout = (QGridLayout*)msgBox.layout();
    layout->addItem(horizontalSpacer, layout->rowCount(), 0, 1, layout->columnCount());
    msgBox.exec();
    return;
  }

  qInfo() << "Export operation completed";

  // Check for warnings - some scripts use a string item
  std::size_t numWarnings = 0;
  auto warningsItem = result->findString("warnings");
  if (warningsItem && warningsItem->numberOfValues() > 0)
  {
    numWarnings = warningsItem->numberOfValues();
    for (std::size_t i = 0; i < numWarnings; ++i)
    {
      qWarning() << QString::fromStdString(warningsItem->value(i));
    }
  }

  // Alternatively check log item
  if (warningsItem == nullptr)
  {
    auto logItem = result->findString("log");
    std::string logString = logItem->value(0);
    if (!logString.empty())
    {
      auto jsonLog = json::parse(logString);
      assert(jsonLog.is_array());
      for (json::iterator it = jsonLog.begin(); it != jsonLog.end(); ++it)
      {
        auto jsRecord = *it;
        assert(jsRecord.is_object());
        if (jsRecord["severity"] >= static_cast<int>(smtk::io::Logger::WARNING))
        {
          numWarnings++;
          std::string content = jsRecord["message"];
          // Escape any quote signs (formats better)
          std::replace(content.begin(), content.end(), '"', '\"');
          qWarning("%zu. %s", numWarnings, content.c_str());
        } // if (>= warning)
      }   // for (it)
    }     // if (logString)
  }       // if (warningsItem)
  qDebug() << "Number of warnings:" << numWarnings;

  if (numWarnings > 0)
  {
    QWidget* parentWidget = pqCoreUtilities::mainWidget();
    QString text;
    QTextStream qs(&text);
    qs << "WARNING: The generated file is INCOMPLETE or INVALID."
       << " You can find more details in the Output Messages view."
       << " You will generally need to correct all invalid input fields"
       << " in the Attribute Editor in order to generate a valid"
       << " ACE3P input file."
       << " Look in the Attribute Editor panel for red \"alert\" icons"
       << " and input fields with red background."
       << "\n\nNumber of errors: " << numWarnings;
    QMessageBox msgBox(
      QMessageBox::NoIcon, "Export Warnings", text, QMessageBox::NoButton, parentWidget);
    // Todo connect alert icon
    // msgBox.setIconPixmap(QIcon(ALERT_ICON_PATH).pixmap(32, 32));
    msgBox.exec();
  }
}

//-----------------------------------------------------------------------------
static pqACE3PExportBehavior* g_instance = nullptr;

pqACE3PExportBehavior::pqACE3PExportBehavior(QObject* parent)
  : Superclass(parent)
{
}

pqACE3PExportBehavior* pqACE3PExportBehavior::instance(QObject* parent)
{
  if (!g_instance)
  {
    g_instance = new pqACE3PExportBehavior(parent);
  }

  if (g_instance->parent() == nullptr && parent)
  {
    g_instance->setParent(parent);
  }

  return g_instance;
}

pqACE3PExportBehavior::~pqACE3PExportBehavior()
{
  if (g_instance == this)
  {
    g_instance = nullptr;
  }

  QObject::disconnect(this);
}
