//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "pqACE3PAnalysisPanel.h"
#include "pqACE3PAnalysisModel.h"
#include "pqACE3PResourceModel.h"
#include "pqACE3PNewAnalysisReaction.h"

#include "pqApplicationCore.h"
#include "smtk/extension/paraview/appcomponents/pqSMTKBehavior.h"
#include "smtk/extension/paraview/appcomponents/pqSMTKWrapper.h"
#include "smtk/extension/qt/qtDescriptivePhraseModel.h"
#include "smtk/io/Logger.h"
#include "smtk/view/ResourcePhraseModel.h"
#include "smtk/view/json/jsonView.h"

#include <QAction>
#include <QCheckBox>
#include <QDebug>
#include <QFormLayout>
#include <QLineEdit>
#include <QMargins>
#include <QMessageBox>
#include <QPushButton>
#include <QSslSocket>
#include <QString>
#include <QVBoxLayout>
#include <QWidget>
#include <QtGlobal>
#include <QHeaderView>
#include <QTableView>
#include <QStandardItemModel>

pqACE3PAnalysisPanel::pqACE3PAnalysisPanel(QWidget* parent)
  : QDockWidget(parent)
  , m_mainwidget(nullptr)
  , m_mainlayout(nullptr)
  , m_newanalysis(nullptr)
  , m_copyanalysis(nullptr)
  , m_analysismodel(nullptr)
  , m_analysistable(nullptr)
{
  this->setObjectName("Analysis Panel");
  this->setWindowTitle("Analysis Selection");


  // Finish main widget
  m_mainwidget = new QWidget(parent);
  m_mainlayout = new QVBoxLayout(m_mainwidget);
  m_mainlayout->setObjectName("analysis_panel_layout");
  m_mainwidget->setObjectName("analysis_panel_widget");

  QHBoxLayout* buttonlayout = new QHBoxLayout;

  m_mainwidget->setLayout(m_mainlayout);
  m_mainlayout->addLayout(buttonlayout);
  setWidget(m_mainwidget);

  auto pqCore = pqApplicationCore::instance();
  if (!pqCore)
  {
    qWarning() << "pqACE3PAnalysisPanel missing pqApplicationCore";
    return;
  }

  auto smtkBehavior = pqSMTKBehavior::instance();
  // Now listen for future connections.
  QObject::connect(smtkBehavior, SIGNAL(addedManagerOnServer(pqSMTKWrapper*, pqServer*)), this,
    SLOT(sourceAdded(pqSMTKWrapper*, pqServer*)));
  QObject::connect(smtkBehavior, SIGNAL(removingManagerFromServer(pqSMTKWrapper*, pqServer*)), this,
    SLOT(sourceRemoved(pqSMTKWrapper*, pqServer*)));

  // create a button for a new analysis
  m_newanalysis = new QPushButton("New Analysis");
  m_newanalysis->setEnabled(true);
  auto newAnalysisReaction = new pqACE3PNewAnalysisReaction(parent);
  connect(m_newanalysis,       &QPushButton::pressed,
          newAnalysisReaction, &pqACE3PNewAnalysisReaction::onTriggered);
  buttonlayout->addWidget(m_newanalysis);

  // create a button for copying analyses
  m_copyanalysis = new QPushButton("Copy Analysis");
  m_copyanalysis->setEnabled(true);
  buttonlayout->addWidget(m_copyanalysis);

  // create the table and the model
  m_analysistable = new QTableView;
  m_analysismodel = new pqACE3PAnalysisModel(parent);

  m_analysistable->setModel(m_analysismodel);
  m_analysistable->setSelectionBehavior(QAbstractItemView::SelectRows);
  m_analysistable->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  m_analysistable->setEditTriggers( QAbstractItemView::DoubleClicked );
  m_analysistable->setWordWrap(true);
  m_analysistable->resizeColumnsToContents();
  connect(m_analysistable, SIGNAL(doubleClicked(const QModelIndex&)),
          m_analysistable, SLOT(edit(const QModelIndex&)));
  m_mainlayout->addWidget(m_analysistable);


  // add resource table and model
  m_resourcetable = new QTableView;
  m_resourcemodel = new pqACE3PResourceModel(parent);
  m_resourcetable->setModel(m_resourcemodel);
  m_resourcetable->setSelectionBehavior(QAbstractItemView::SelectRows);
  m_resourcetable->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  m_resourcetable->setEditTriggers( QAbstractItemView::DoubleClicked );
  m_resourcetable->setWordWrap(true);
  m_resourcetable->resizeColumnsToContents();
  connect(m_resourcetable, SIGNAL(doubleClicked(const QModelIndex&)),
          m_resourcetable, SLOT(edit(const QModelIndex&)));
  m_mainlayout->addWidget(m_resourcetable);
}

pqACE3PAnalysisPanel::~pqACE3PAnalysisPanel()
{
  auto pqCore = pqApplicationCore::instance();
  pqCore->unRegisterManager(QString("analysis_panel"));
}

void pqACE3PAnalysisPanel::infoSlot(const QString& msg)
{
  // Would like to emit signal to main window status bar
  // but that is not currently working. Instead, qInfo()
  // messages are sent to the CMB Log Window:
  qInfo() << "AnalysisPanel:" << msg;
}

void pqACE3PAnalysisPanel::setView(const smtk::view::ConfigurationPtr& view)
{
  m_view = view;

  auto smtkBehavior = pqSMTKBehavior::instance();

  smtkBehavior->visitResourceManagersOnServers([this](pqSMTKWrapper* r, pqServer* s) {
    this->sourceAdded(r, s);
    return false;
  });
}

void pqACE3PAnalysisPanel::sourceAdded(pqSMTKWrapper* wrapper, pqServer* server)
{
  if (!wrapper || !server)
  {
    return;
  }

  smtk::resource::ManagerPtr rsrcMgr = wrapper->smtkResourceManager();
  smtk::view::ManagerPtr viewMgr = wrapper->smtkViewManager();
  if (!rsrcMgr || !viewMgr)
  {
    return;
  }
}

void pqACE3PAnalysisPanel::sourceRemoved(pqSMTKWrapper* mgr, pqServer* server)
{
  if (!mgr || !server)
  {
    return;
  }
}
