#include "vtkCamera.h"
#include "vtkContinuousPointRegistration.h"
#include "vtkDoubleArray.h"
#include "vtkFieldData.h"
#include "vtkInformation.h"
#include "vtkMatrix4x4.h"
#include "vtkNamedColors.h"
#include "vtkNew.h"
#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkRenderer.h"
#include "vtkRobustIterativeClosestPointTransform.h"
#include "vtkSphereSource.h"
#include "vtkSpsPolyDataFileSeriesReader.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkTestUtilities.h"
#include "vtkTesting.h"
#include "vtkTransform.h"
#include "vtkXMLPolyDataReader.h"
#include "vtkXMLPolyDataWriter.h"

#include <sstream>

namespace
{
int iColor = 0;

void AppendMesh(vtkContinuousPointRegistration* regFilter,
  vtkSmartPointer<vtkRenderer> renderers[2], bool scalarVisibility = false)
{
  static int iColor = 0;
  vtkNew<vtkNamedColors> colors;

  vtkNew<vtkPolyData> poly;
  poly->DeepCopy(regFilter->GetOutput());
  vtkNew<vtkActor> actor;
  vtkNew<vtkPolyDataMapper> mapper;
  mapper->SetInputData(poly);
  mapper->SetScalarVisibility(scalarVisibility);
  actor->SetMapper(mapper);
  vtkDoubleArray* arr =
    vtkDoubleArray::SafeDownCast(poly->GetFieldData()->GetArray("RegTransform"));
  vtkNew<vtkTransform> tf;

  if (arr)
  {
    std::cout << "Registration present" << std::endl;
    vtkNew<vtkMatrix4x4> mat;
    std::memcpy(mat->GetData(), arr->GetVoidPointer(0), 16 * sizeof(double));
    tf->SetMatrix(mat);
    tf->Update();
    mat->PrintSelf(std::cout, vtkIndent());
  }
  actor->SetUserTransform(tf);

  std::stringstream ss(colors->GetColorNames());
  std::string colorName;
  std::getline(ss, colorName, '\n');
  for (int i = 0; i < iColor; i++)
    std::getline(ss, colorName, '\n');

  actor->GetProperty()->SetColor(colors->GetColor3d(colorName).GetData());
  iColor = iColor + 2;

  renderers[0]->AddActor(actor);

  vtkNew<vtkActor> actor1;
  vtkNew<vtkPolyDataMapper> mapper1;
  mapper1->SetInputData(poly);
  mapper1->SetScalarVisibility(true);
  actor1->SetMapper(mapper1);
  actor1->SetUserTransform(tf);
  renderers[1]->AddActor(actor1);
  renderers[0]->GetRenderWindow()->Render();
}

void CreateMovingSpheres(vtkIdType nSpheres = 3)
{
  vtkNew<vtkSphereSource> sphere;
  sphere->SetPhiResolution(11);
  sphere->SetThetaResolution(11);
  sphere->SetRadius(10);
  for (vtkIdType iFile = 0; iFile < nSpheres; iFile++)
  {
    sphere->SetCenter(0.0, double(iFile), 0.0);
    sphere->SetStartPhi(90);
    sphere->Update();
    vtkPolyData* poly = sphere->GetOutput();

    vtkNew<vtkDoubleArray> indexArray;
    indexArray->SetName("Index");
    indexArray->SetNumberOfValues(1);
    indexArray->SetNumberOfComponents(1);
    indexArray->SetValue(0, double(iFile));
    poly->GetFieldData()->AddArray(indexArray);
    vtkNew<vtkDoubleArray> affineArray;
    affineArray->SetNumberOfComponents(4);
    affineArray->SetNumberOfValues(16);
    affineArray->Fill(0);
    for (int i = 0; i < 4; i++)
    {
      affineArray->SetValue(0 + i * 5, 1.0);
      affineArray->SetValue(7, double(iFile));
      affineArray->SetName("UserTransform");
      poly->GetFieldData()->AddArray(affineArray);
      vtkNew<vtkXMLPolyDataWriter> writer;
      std::stringstream ss;
      ss << "sphere";
      ss << std::setfill('0');
      ss << std::setw(2);
      ss << iFile;
      ss << ".vtp";
      writer->SetFileName(ss.str().c_str());
      writer->SetInputData(poly);
      writer->Write();
    }
  }
}
} // namespace

static char ContinuousPointRegistrationLog[] =

  "# StreamVersion 1.2\n"
  "RenderEvent 0 0 0 0 0 0 0\n"
  "EnterEvent 395 218 0 0 0 0 0\n"
  "MouseMoveEvent 395 218 0 0 0 0 0\n"
  "LeaveEvent 343 399 0 0 0 0 0\n"
  "ExitEvent 343 399 0 0 0 0 0\n";

int TestContinuousPointRegistration(int argc, char* argv[])
{
  int rval = EXIT_SUCCESS;
  vtkNew<vtkContinuousPointRegistration> contReg;
  contReg->DebugOn();

  vtkIdType nFiles = 3;

  vtkNew<vtkSpsPolyDataFileSeriesReader> seriesReader;
  vtkNew<vtkXMLPolyDataReader> reader;
  seriesReader->SetReader(reader);

  bool useSpheres;
  useSpheres = true;

  std::string filePrefix;

  if (useSpheres)
  {
    CreateMovingSpheres(nFiles);
    filePrefix = "sphere";
  }
  else
  {
    filePrefix = "mesh";
  }

  for (vtkIdType iFile = 0; iFile < nFiles; iFile++)
  {
    std::stringstream ss;
    ss << filePrefix;
    ss << std::setfill('0');
    ss << std::setw(2);
    ss << iFile;
    ss << ".vtp";
    seriesReader->AddFileName(ss.str().c_str());
  }
  seriesReader->Update();

  vtkInformation* info = seriesReader->GetOutputInformation(0);
  seriesReader->UpdateInformation();

  double* steps = info->Get(vtkStreamingDemandDrivenPipeline::TIME_STEPS());
  int nSteps = info->Length(vtkStreamingDemandDrivenPipeline::TIME_STEPS());

  for (int iStep = 0; iStep < nSteps; iStep++)
  {
    std::cout << "Time step: " << steps[iStep] << std::endl;
  }

  double* timeRange = info->Get(vtkStreamingDemandDrivenPipeline::TIME_RANGE());
  std::cout << "Time range: [" << timeRange[0] << ", " << timeRange[1] << "]" << std::endl;

  seriesReader->GetOutputInformation(0)->Set(
    vtkStreamingDemandDrivenPipeline::UPDATE_TIME_STEP(), 0.0);

  contReg->SetInputConnection(seriesReader->GetOutputPort());
  contReg->UpdateInformation();

  vtkNew<vtkRenderWindow> renWin;
  vtkSmartPointer<vtkRenderer> renderers[2];

  renderers[0] = vtkSmartPointer<vtkRenderer>::New();
  renderers[0]->SetViewport(0.0, 0.0, 0.5, 1.0);
  renWin->AddRenderer(renderers[0]);

  renderers[1] = vtkSmartPointer<vtkRenderer>::New();
  renderers[1]->SetViewport(0.5, 0.0, 1.0, 1.0);
  renWin->AddRenderer(renderers[1]);
  renderers[1]->SetActiveCamera(renderers[0]->GetActiveCamera());

  renWin->SetWindowName("TestContinuousPointRegistration");
  renWin->SetSize(300, 300);
  vtkNew<vtkRenderWindowInteractor> iren;
  iren->SetRenderWindow(renWin);

  for (vtkIdType iFile = 0; iFile < nFiles; iFile++)
  {
    contReg->GetOutputInformation(0)->Set(
      vtkStreamingDemandDrivenPipeline::UPDATE_TIME_STEP(), double(iFile));
    contReg->Update();
    AppendMesh(contReg.GetPointer(), renderers);
  }

  renderers[0]->ResetCamera();
  renWin->Render();
  iren->Initialize();
  return vtkTesting::InteractorEventLoop(argc, argv, iren, ContinuousPointRegistrationLog);
}
