#pragma once

#include "vtkCommonDataModelModule.h" // For export macro

#include "vtkLinearTransform.h"

#include "vtkGenericCell.h"          // For thread local storage
#include "vtkSMPThreadLocalObject.h" // For thread local storage

// TODO: Add line-search and threading
class vtkCellLocator;
class vtkLandmarkTransform;
class vtkDataSet;
class vtkPoints;

class VTK_EXPORT vtkRobustIterativeClosestPointTransform : public vtkLinearTransform
{
public:
  static const int VTK_ICP_MODE_RMS = 0;
  static const int VTK_ICP_MODE_AV = 1;

  static vtkRobustIterativeClosestPointTransform* New();
  vtkTypeMacro(vtkRobustIterativeClosestPointTransform, vtkLinearTransform);

  void PrintSelf(ostream& os, vtkIndent indent) override;

  //@{
  /**
   * Specify the source and target data sets.
   */
  void SetSource(vtkDataSet* source);
  void SetTarget(vtkDataSet* target);
  vtkGetObjectMacro(Source, vtkDataSet);
  vtkGetObjectMacro(Target, vtkDataSet);
  //@}

  //@{
  /**
   * Set/Get a spatial locator for speeding up the search process.
   * An instance of vtkCellLocator is used by default.
   */
  void SetLocator(vtkCellLocator* locator);
  vtkGetObjectMacro(Locator, vtkCellLocator);
  //@}

  //@{
  /**
   * Set/Get the maximum number of iterations. Default is 50.
   */
  vtkSetMacro(MaximumNumberOfIterations, int);
  vtkGetMacro(MaximumNumberOfIterations, int);
  //@}

  //@{
  /**
   * Get the number of iterations since the last update
   */
  vtkGetMacro(NumberOfIterations, int);
  //@}

  //@{
  /**
   * Force the algorithm to check the mean distance between two iterations.
   * Default is Off.
   */
  vtkSetMacro(CheckMeanDistance, vtkTypeBool);
  vtkGetMacro(CheckMeanDistance, vtkTypeBool);
  vtkBooleanMacro(CheckMeanDistance, vtkTypeBool);
  //@}

  vtkGetMacro(LastMeanDistance, double);

  //@{
  /**
   * Specify the mean distance mode. This mode expresses how the mean
   * distance is computed. The RMS mode is the square root of the average
   * of the sum of squares of the closest point distances. The Absolute
   * Value mode is the mean of the sum of absolute values of the closest
   * point distances. The default is VTK_ICP_MODE_RMS
   */
  vtkSetClampMacro(MeanDistanceMode, int, VTK_ICP_MODE_RMS, VTK_ICP_MODE_AV);
  vtkGetMacro(MeanDistanceMode, int);
  void SetMeanDistanceModeToRMS() { this->SetMeanDistanceMode(VTK_ICP_MODE_RMS); }
  void SetMeanDistanceModeToAbsoluteValue() { this->SetMeanDistanceMode(VTK_ICP_MODE_AV); }
  const char* GetMeanDistanceModeAsString();
  //@}

  //@{
  /**
   * Set/Get the maximum mean distance between two iteration. If the mean
   * distance is lower than this, the convergence stops. The default
   * is 0.01.
   */
  vtkSetMacro(MaximumMeanDistance, double);
  vtkGetMacro(MaximumMeanDistance, double);
  //@}

  //@{
  /**
   * Get the mean distance between the last two iterations.
   */
  vtkGetMacro(MeanDistance, double);
  //@}

  //@{
  /**
   * Set/Get the maximum number of landmarks sampled in your dataset.
   * If your dataset is dense, then you will typically not need all the
   * points to compute the ICP transform. The default is 200.
   */
  vtkSetMacro(MaximumNumberOfLandmarks, int);
  vtkGetMacro(MaximumNumberOfLandmarks, int);
  //@}

  //@{
  /**
   * Starts the process by translating source centroid to target centroid.
   * The default is Off.
   */
  vtkSetMacro(StartByMatchingCentroids, vtkTypeBool);
  vtkGetMacro(StartByMatchingCentroids, vtkTypeBool);
  vtkBooleanMacro(StartByMatchingCentroids, vtkTypeBool);
  //@}

  //@{
  /**
   * Get the internal landmark transform. Use it to constrain the number of
   * degrees of freedom of the solution (i.e. rigid body, similarity, etc.).
   */
  vtkGetObjectMacro(LandmarkTransform, vtkLandmarkTransform);
  //@}

  //@{
  /** Set/Get the threshold to declare a point to not have a corresponding
   * point in the other point set. This value is only used if
   * UseThresholdParameter is True (not the default).
   * This is useful to align partially overlapping surfaces.
   * If this value is negative, all points are considered to have a
   * corresponding point in the other point set. The default is 1.0.
   */
  vtkSetMacro(ThresholdParameter, double);
  vtkGetMacro(ThresholdParameter, double);
  //@}

  ///@{
  /**
   *
   */
  enum ThreasholdModeType : int
  {
    ThresholdNone = 0,
    ThresholdFar = 1,
    ThresholdStd = 2,
  };

  vtkSetClampMacro(ThresholdMode, int, ThresholdNone, ThresholdStd);
  vtkGetMacro(ThresholdMode, int);
  void SetThresholdModeToNone() { this->SetThresholdMode(ThresholdNone); }
  void SetThresholdModeToFar() { this->SetThresholdMode(ThresholdFar); }
  void SetThresholdModeToStd() { this->SetThresholdMode(ThresholdStd); }
  const char* GetThresholdModeAsString();
  ///@}

  /**
   * Invert the transformation.  This is done by switching the
   * source and target.
   */
  void Inverse() override;

  /**
   * Make another transform of the same type.
   */
  vtkAbstractTransform* MakeTransform() override;

protected:
  //@{
  /**
   * Release source and target
   */
  void ReleaseSource(void);
  void ReleaseTarget(void);
  //@}

  /**
   * Release locator
   */
  void ReleaseLocator(void);

  /**
   * Create default locator. Used to create one when none is specified.
   */
  void CreateDefaultLocator(void);

  /**
   * Get the MTime of this object also considering the locator.
   */
  vtkMTimeType GetMTime() override;

  vtkRobustIterativeClosestPointTransform();
  ~vtkRobustIterativeClosestPointTransform() override;

  void InternalUpdate() override;

  void UpdateLandmarks(
    vtkPoints* a, vtkIdType nb_points, vtkPoints* closestp, vtkPoints* sourceLandmarks);

  /**
   * This method does no type checking, use DeepCopy instead.
   */
  void InternalDeepCopy(vtkAbstractTransform* transform) override;

  double ClosestPoint(double point[3], double closestPoint[3]);

  vtkDataSet* Source;
  vtkDataSet* Target;
  //  vtkStaticCellLocator* Locator;
  vtkCellLocator* Locator;
  int MaximumNumberOfIterations;
  vtkTypeBool CheckMeanDistance;
  int MeanDistanceMode;
  double MaximumMeanDistance;
  int MaximumNumberOfLandmarks;
  vtkTypeBool StartByMatchingCentroids;
  vtkTypeBool StoreLastMeanDistance;

  int NumberOfIterations;
  double MeanDistance;
  vtkLandmarkTransform* LandmarkTransform;

  double LastMeanDistance;
  double ThresholdParameter;
  int ThresholdMode;

  vtkSMPThreadLocalObject<vtkGenericCell> TLCell;
  vtkSMPThreadLocalObject<vtkIdList> TLCellIds;
  double NoValue;
  double Tolerance;

private:
  vtkRobustIterativeClosestPointTransform(const vtkRobustIterativeClosestPointTransform&) = delete;
  void operator=(const vtkRobustIterativeClosestPointTransform&) = delete;
};
