#include "vtkRobustIterativeClosestPointTransform.h"

#include "vtkCellData.h"
#include "vtkCellLocator.h"
#include "vtkDataSet.h"
#include "vtkLandmarkTransform.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkPoints.h"
#include "vtkPolyData.h"
#include "vtkPolygon.h"
#include "vtkSMPThreadLocal.h"
#include "vtkSMPThreadLocalObject.h"
#include "vtkSMPTools.h"
#include "vtkStaticCellLocator.h"
#include "vtkTransform.h"

#include <vector>

vtkStandardNewMacro(vtkRobustIterativeClosestPointTransform);

//----------------------------------------------------------------------------

vtkRobustIterativeClosestPointTransform::vtkRobustIterativeClosestPointTransform()
  : vtkLinearTransform()
{
  this->Source = nullptr;
  this->Target = nullptr;
  this->Locator = nullptr;
  this->LandmarkTransform = vtkLandmarkTransform::New();
  this->MaximumNumberOfIterations = 50;
  this->CheckMeanDistance = 0;
  this->MeanDistanceMode = VTK_ICP_MODE_RMS;
  this->MaximumMeanDistance = 0.01;
  this->MaximumNumberOfLandmarks = 200;
  this->StartByMatchingCentroids = 0;
  this->StoreLastMeanDistance = 0;

  this->NumberOfIterations = 0;
  this->MeanDistance = 0.0;

  this->ThresholdParameter = 1.0;
  this->ThresholdMode = vtkRobustIterativeClosestPointTransform::ThresholdFar;
  this->NoValue = 0.0;
  this->Tolerance = 1e-12;
}

//----------------------------------------------------------------------------

const char* vtkRobustIterativeClosestPointTransform::GetMeanDistanceModeAsString()
{
  if (this->MeanDistanceMode == VTK_ICP_MODE_RMS)
  {
    return "RMS";
  }
  else
  {
    return "AbsoluteValue";
  }
}

//----------------------------------------------------------------------------

const char* vtkRobustIterativeClosestPointTransform::GetThresholdModeAsString()
{
  if (this->ThresholdMode == vtkRobustIterativeClosestPointTransform::ThresholdStd)
  {
    return "Std";
  }
  else if (this->ThresholdMode == vtkRobustIterativeClosestPointTransform::ThresholdFar)
  {
    return "Far";
  }
  else
  {
    return "None";
  }
}

//----------------------------------------------------------------------------

vtkRobustIterativeClosestPointTransform::~vtkRobustIterativeClosestPointTransform()
{
  ReleaseSource();
  ReleaseTarget();
  ReleaseLocator();
  this->LandmarkTransform->Delete();
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::SetSource(vtkDataSet* source)
{
  if (this->Source == source)
  {
    return;
  }

  if (this->Source)
  {
    this->ReleaseSource();
  }

  if (source)
  {
    source->Register(this);
  }

  this->Source = source;
  this->NoValue = this->Source->GetLength();
  this->Modified();
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::ReleaseSource()
{
  if (this->Source)
  {
    this->Source->UnRegister(this);
    this->Source = nullptr;
  }
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::SetTarget(vtkDataSet* target)
{
  if (this->Target == target)
  {
    return;
  }

  if (this->Target)
  {
    this->ReleaseTarget();
  }

  if (target)
  {
    target->Register(this);
  }

  this->Target = target;
  this->Modified();
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::ReleaseTarget()
{
  if (this->Target)
  {
    this->Target->UnRegister(this);
    this->Target = nullptr;
  }
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::SetLocator(vtkCellLocator* locator)
{
  if (this->Locator == locator)
  {
    return;
  }

  if (this->Locator)
  {
    this->ReleaseLocator();
  }

  if (locator)
  {
    locator->Register(this);
  }

  this->Locator = locator;
  this->Modified();
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::ReleaseLocator()
{
  if (this->Locator)
  {
    this->Locator->UnRegister(this);
    this->Locator = nullptr;
  }
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::CreateDefaultLocator()
{
  if (this->Locator)
  {
    this->ReleaseLocator();
  }
  //  this->Locator = vtkStaticCellLocator::New();
  this->Locator = vtkCellLocator::New();
}

//------------------------------------------------------------------------

vtkMTimeType vtkRobustIterativeClosestPointTransform::GetMTime()
{
  vtkMTimeType result = this->vtkLinearTransform::GetMTime();
  vtkMTimeType mtime;

  if (this->Source)
  {
    mtime = this->Source->GetMTime();
    if (mtime > result)
    {
      result = mtime;
    }
  }

  if (this->Target)
  {
    mtime = this->Target->GetMTime();
    if (mtime > result)
    {
      result = mtime;
    }
  }

  if (this->Locator)
  {
    mtime = this->Locator->GetMTime();
    if (mtime > result)
    {
      result = mtime;
    }
  }

  if (this->LandmarkTransform)
  {
    mtime = this->LandmarkTransform->GetMTime();
    if (mtime > result)
    {
      result = mtime;
    }
  }

  return result;
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::Inverse()
{
  vtkDataSet* tmp1 = this->Source;
  this->Source = this->Target;
  this->Target = tmp1;
  this->Modified();
}

//----------------------------------------------------------------------------

vtkAbstractTransform* vtkRobustIterativeClosestPointTransform::MakeTransform()
{
  return vtkRobustIterativeClosestPointTransform::New();
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::InternalDeepCopy(vtkAbstractTransform* transform)
{
  vtkRobustIterativeClosestPointTransform* t = (vtkRobustIterativeClosestPointTransform*)transform;

  this->SetSource(t->GetSource());
  this->SetTarget(t->GetTarget());
  this->SetLocator(t->GetLocator());
  this->SetMaximumNumberOfIterations(t->GetMaximumNumberOfIterations());
  this->SetCheckMeanDistance(t->GetCheckMeanDistance());
  this->SetMeanDistanceMode(t->GetMeanDistanceMode());
  this->SetMaximumMeanDistance(t->GetMaximumMeanDistance());
  this->SetMaximumNumberOfLandmarks(t->GetMaximumNumberOfLandmarks());

  this->Modified();
}

#if 0
namespace
{
struct CollectPointsFunctor
{
  vtkPoints* Points;
  vtkPoints* SourceLandmarks;
  vtkPoints* ClosestPoints;
  vtkCellLocator* CellLocator;

  using TLS = vtkSMPThreadLocalObject<vtkPoints>;
  TLS TLClosestPoints;
  TLS TLSourceLandmarks;
  vtkSMPThreadLocalObject<vtkGenericCell> TLCell;

  double FarThreshold2;
  CollectPointsFunctor(vtkPoints* points, vtkPoints* sourceLandmarks, vtkPoints* closestPoints,
    vtkCellLocator* cellLocator, double farThreshold2)
    : Points(points)
    , SourceLandmarks(sourceLandmarks)
    , ClosestPoints(closestPoints)
    , CellLocator(cellLocator)
    , FarThreshold2(farThreshold2)
  {
  }
  void Initialize()
  {
    this->TLClosestPoints.Local()->Initialize();
    this->TLSourceLandmarks.Local()->Initialize();
  }
  void operator()(vtkIdType beginPointId, vtkIdType endPointId)
  {
    double outPoint[3];
    vtkIdType cellId;
    int subId = 0;
    double dist2;
    auto cell = this->TLCell.Local();

    auto& closestPoints = TLClosestPoints.Local();
    auto& sourceLandmarks = TLSourceLandmarks.Local();
    double point[3];
    for (vtkIdType iPoint = beginPointId; iPoint < endPointId; iPoint++)
    {
      Points->GetPoint(iPoint, point);
      this->CellLocator->FindClosestPoint(point, outPoint, cell, cellId, subId, dist2);
      if (dist2 < FarThreshold2)
      {
        closestPoints->InsertNextPoint(outPoint);
        sourceLandmarks->InsertNextPoint(point);
      }
    }
  }
  void Reduce()
  {
    using TLSIter = TLS::iterator;
    TLSIter end = this->TLClosestPoints.end();
    TLSIter landmarksEnd = this->TLSourceLandmarks.end();

    vtkIdType nPoints = 0;
    for (TLSIter itr = this->TLClosestPoints.begin(); itr != end; ++itr)
    {
      nPoints += (*itr)->GetNumberOfPoints();
    }
    this->ClosestPoints->Initialize();

    TLSIter itClosestPoints = this->TLClosestPoints.begin();

    // TODO: Points are repeated!!!
    for (; itClosestPoints != end; itClosestPoints++, itClosestPoints++)
    {
      for (vtkIdType pointId = 0; pointId < (*itClosestPoints)->GetNumberOfPoints(); pointId++)
      {
        this->ClosestPoints->InsertNextPoint((*itClosestPoints)->GetPoint(pointId));
      }
    }

    TLSIter itSourceLandmarks = this->TLSourceLandmarks.begin();

    for (; itSourceLandmarks != landmarksEnd; itSourceLandmarks++)
    {
      for (vtkIdType pointId = 0; pointId < (*itSourceLandmarks)->GetNumberOfPoints(); pointId++)
      {
        this->SourceLandmarks->InsertNextPoint((*itSourceLandmarks)->GetPoint(pointId));
      }
    }
  }
  void Execute(int nPoints) { vtkSMPTools::For(0, static_cast<vtkIdType>(nPoints), *this); }
};

struct CollectPointsWorker
{
  CollectPointsWorker() {}

  // Dispatch 2 arrays
  void operator()(vtkPoints* points, vtkPoints* sourceLandmarks, vtkPoints* closestPoints,
    vtkCellLocator* cellLocator, double farThreshold2)
  {
    CollectPointsFunctor functor(
      points, sourceLandmarks, closestPoints, cellLocator, farThreshold2);
    vtkSMPTools::For(0, points->GetNumberOfPoints(), functor);
  }
};
}
#endif

double vtkRobustIterativeClosestPointTransform::ClosestPoint(double x[3], double closestPoint[3])
{
  double ret = 0.0;

  double p[3];
  vtkIdType cellId;
  int subId;
  double vlen2;
  double g[3];

  vtkDataArray* cnorms = nullptr;
  if (this->Source->GetCellData() && this->Source->GetCellData()->GetNormals())
  {
    cnorms = this->Source->GetCellData()->GetNormals();
  }

  auto cell = this->TLCell.Local();
  this->Locator->FindClosestPoint(x, p, cell, cellId, subId, vlen2);

  if (cellId != -1) // point located
  {
    // dist = | point - x |
    ret = std::sqrt(vlen2);
    for (int i = 0; i < 3; i++)
    {
      g[i] = (p[i] - x[i]) / (ret == 0. ? 1. : ret);
    }

    double dist2, weights[3], pcoords[3], awnorm[3] = { 0, 0, 0 };
    cell->EvaluatePosition(p, closestPoint, subId, pcoords, dist2, weights);

    auto idList = this->TLCellIds.Local();
    int count = 0;
    for (int i = 0; i < 3; i++)
    {
      count += (std::abs(weights[i]) < this->Tolerance ? 1 : 0);
    }
    // Face case - weights contains no 0s
    if (count == 0)
    {
      // Compute face normal.
      if (cnorms)
      {
        cnorms->GetTuple(cellId, awnorm);
      }
      else
      {
        vtkPolygon::ComputeNormal(cell->Points, awnorm);
      }
    }
    // Edge case - weights contain one 0
    else if (count == 1)
    {
      // ... edge ... get two adjacent faces, compute average normal
      int a = -1, b = -1;
      for (int edge = 0; edge < 3; edge++)
      {
        if (std::abs(weights[edge]) < this->Tolerance)
        {
          a = cell->PointIds->GetId((edge + 1) % 3);
          b = cell->PointIds->GetId((edge + 2) % 3);
          break;
        }
      }

      if (a == -1)
      {
        vtkErrorMacro(<< "Could not find edge when closest point is "
                      << "expected to be on an edge.");
        return this->NoValue;
      }

      // The first argument is the cell ID. We pass a bogus cell ID so that
      // all face IDs attached to the edge are returned in the idList.

      // Only works for polydata!
      vtkPolyData* PolySource = vtkPolyData::SafeDownCast(this->Source);
      if (PolySource)
      {
        PolySource->GetCellEdgeNeighbors(VTK_ID_MAX, a, b, idList);
        for (int i = 0; i < idList->GetNumberOfIds(); i++)
        {
          double norm[3];
          if (cnorms)
          {
            cnorms->GetTuple(idList->GetId(i), norm);
          }
          else
          {
            PolySource->GetCell(idList->GetId(i), cell);
            vtkPolygon::ComputeNormal(cell->GetPoints(), norm);
          }
          awnorm[0] += norm[0];
          awnorm[1] += norm[1];
          awnorm[2] += norm[2];
        }
        vtkMath::Normalize(awnorm);
      }
    }

    // Vertex case - weights contain two 0s
    else if (count == 2)
    {
      // ... vertex ... this is the expensive case, get all adjacent
      // faces and compute sum(a_i * n_i) Angle-Weighted Pseudo
      // Normals, J. Andreas Baerentzen and Henrik Aanaes
      int a = -1;
      for (int i = 0; i < 3; i++)
      {
        if (std::abs(weights[i]) > this->Tolerance)
        {
          a = cell->PointIds->GetId(i);
        }
      }

      if (a == -1)
      {
        vtkErrorMacro(<< "Could not find point when closest point is "
                      << "expected to be a point.");
        return this->NoValue;
      }

      this->Source->GetPointCells(a, idList);
      for (int i = 0; i < idList->GetNumberOfIds(); i++)
      {
        double norm[3];
        this->Source->GetCell(idList->GetId(i), cell);
        if (cnorms)
        {
          cnorms->GetTuple(idList->GetId(i), norm);
        }
        else
        {
          vtkPolygon::ComputeNormal(cell->GetPoints(), norm);
        }

        // Compute angle at point a
        int b = cell->GetPointId(0);
        int c = cell->GetPointId(1);
        if (a == b)
        {
          b = cell->GetPointId(2);
        }
        else if (a == c)
        {
          c = cell->GetPointId(2);
        }
        double pa[3], pb[3], pc[3];
        this->Source->GetPoint(a, pa);
        this->Source->GetPoint(b, pb);
        this->Source->GetPoint(c, pc);
        for (int j = 0; j < 3; j++)
        {
          pb[j] -= pa[j];
          pc[j] -= pa[j];
        }
        vtkMath::Normalize(pb);
        vtkMath::Normalize(pc);
        double alpha = std::acos(vtkMath::Dot(pb, pc));
        awnorm[0] += alpha * norm[0];
        awnorm[1] += alpha * norm[1];
        awnorm[2] += alpha * norm[2];
      }
      vtkMath::Normalize(awnorm);
    }

    // sign(dist) = dot(grad, cell normal)
    if (ret == 0)
    {
      for (int i = 0; i < 3; i++)
      {
        g[i] = awnorm[i];
      }
    }
    ret *= (vtkMath::Dot(g, awnorm) < 0.) ? 1. : -1.;
  }
  return ret;
}

namespace
{
double Lerp(double a, double b, double t)
{
  return a * (1.0f - t) + b * t;
}
}

void vtkRobustIterativeClosestPointTransform::UpdateLandmarks(
  vtkPoints* a, vtkIdType nb_points, vtkPoints* closestp, vtkPoints* sourceLandmarks)
{
  vtkIdType cell_id;
  int sub_id;
  double dist2;
  double outPoint[3];

  double FarThreshold2 = this->ThresholdParameter * this->ThresholdParameter;

  closestp->Initialize();

  // TODO: Vector of pairs<dist, point>, sort them and add
#if 1
  for (vtkIdType i = 0; i < nb_points; i++)
  {
    this->Locator->FindClosestPoint(a->GetPoint(i), outPoint, cell_id, sub_id, dist2);
    if (dist2 < FarThreshold2)
    {
      closestp->InsertNextPoint(outPoint);
      sourceLandmarks->InsertNextPoint(a->GetPoint(i));
    }
  }
#else
  // Too expensive

  vtkNew<vtkPoints> tempLandMarks;
  vtkNew<vtkPoints> outPoints;
  std::vector<double> dists;
  dists.reserve(nb_points);

  double average_value = {};
  double average_valueSquared = {};

  vtkIdType i;
  for (i = 0; i < nb_points; i++)
  {
    double value = this->ClosestPoint(a->GetPoint(i), outPoint);

    outPoints->InsertNextPoint(outPoint);
    tempLandMarks->InsertNextPoint(a->GetPoint(i));
    dists.push_back(value);

    average_value = Lerp(average_value, value, 1.0 / double(i + 1));
    average_valueSquared = Lerp(average_valueSquared, value * value, 1.0f / double(i + 1));
  }
  double std_value = std::sqrt(std::abs(average_valueSquared - (average_value * average_value)));

  for (i = 0; i < nb_points; i++)
  {
    double dist = dists[i];
    if (std::fabs(dist) < std_value)
    {
      closestp->InsertNextPoint(outPoints->GetPoint(i));
      sourceLandmarks->InsertNextPoint(tempLandMarks->GetPoint(i));
    }
  }

  for (vtkIdType i = 0; i < nb_points; i++)
  {

    double dist = this->ClosestPoint(a->GetPoint(i), outPoint);
    dist2 = dist * dist;
    if (dist2 < FarThreshold2)
    {
      closestp->InsertNextPoint(outPoint);
      sourceLandmarks->InsertNextPoint(a->GetPoint(i));
    }
  }
#endif
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::InternalUpdate()
{
  // Check source, target

  if (this->Source == nullptr || !this->Source->GetNumberOfPoints())
  {
    vtkErrorMacro(<< "Can't execute with nullptr or empty input");
    return;
  }

  if (this->Target == nullptr || !this->Target->GetNumberOfPoints())
  {
    vtkErrorMacro(<< "Can't execute with nullptr or empty target");
    return;
  }

  // Create locator
  this->CreateDefaultLocator();
  this->Locator->SetDataSet(this->Target);
  this->Locator->SetNumberOfCellsPerBucket(1);

  // Extra stuff
  // this->Locator->SetTolerance(0.0);
  // this->Locator->SetNumberOfCellsPerBucket(10);
  // this->Locator->CacheCellBoundsOn();
  // this->Locator->AutomaticOn();

  this->Locator->BuildLocator();

  // Create two sets of points to handle iteration

  int step = 1;
  if (this->Source->GetNumberOfPoints() > this->MaximumNumberOfLandmarks)
  {
    step = this->Source->GetNumberOfPoints() / this->MaximumNumberOfLandmarks;
    vtkDebugMacro(<< "Landmarks step is now : " << step);
  }

  vtkIdType nb_points = this->Source->GetNumberOfPoints() / step;

  // Allocate some points.
  // - closestp is used so that the internal state of LandmarkTransform remains
  //   correct whenever the iteration process is stopped (hence its source
  //   and landmark points might be used in a vtkThinPlateSplineTransform).
  // - points2 could have been avoided, but do not ask me why
  //   InternalTransformPoint is not working correctly on my computer when
  //   in and out are the same pointer.

  vtkNew<vtkPoints> points1;
  points1->SetNumberOfPoints(nb_points);

  vtkNew<vtkPoints> closestp;
  closestp->SetNumberOfPoints(nb_points);

  vtkNew<vtkPoints> points2;
  points2->SetNumberOfPoints(nb_points);

  // Fill with initial positions (sample dataset using step)

  vtkNew<vtkTransform> accumulate;
  accumulate->PostMultiply();

  vtkIdType i;
  int j;
  double p1[3], p2[3];

  if (StartByMatchingCentroids)
  {
    double source_centroid[3] = { 0, 0, 0 };
    for (i = 0; i < this->Source->GetNumberOfPoints(); i++)
    {
      this->Source->GetPoint(i, p1);
      source_centroid[0] += p1[0];
      source_centroid[1] += p1[1];
      source_centroid[2] += p1[2];
    }
    source_centroid[0] /= this->Source->GetNumberOfPoints();
    source_centroid[1] /= this->Source->GetNumberOfPoints();
    source_centroid[2] /= this->Source->GetNumberOfPoints();

    double target_centroid[3] = { 0, 0, 0 };
    for (i = 0; i < this->Target->GetNumberOfPoints(); i++)
    {
      this->Target->GetPoint(i, p2);
      target_centroid[0] += p2[0];
      target_centroid[1] += p2[1];
      target_centroid[2] += p2[2];
    }
    target_centroid[0] /= this->Target->GetNumberOfPoints();
    target_centroid[1] /= this->Target->GetNumberOfPoints();
    target_centroid[2] /= this->Target->GetNumberOfPoints();

    accumulate->Translate(target_centroid[0] - source_centroid[0],
      target_centroid[1] - source_centroid[1], target_centroid[2] - source_centroid[2]);
    accumulate->Update();

    for (i = 0, j = 0; i < nb_points; i++, j += step)
    {
      double outPoint[3];
      accumulate->InternalTransformPoint(this->Source->GetPoint(j), outPoint);
      points1->SetPoint(i, outPoint);
    }
  }
  else
  {
    for (i = 0, j = 0; i < nb_points; i++, j += step)
    {
      points1->SetPoint(i, this->Source->GetPoint(j));
    }
  }

  // Go

  vtkIdType cell_id;
  int sub_id;
  double dist2, totaldist = 0.0;
  double outPoint[3];

  vtkPoints *temp, *a = points1.GetPointer(), *b = points2.GetPointer();

  this->NumberOfIterations = 0;

  double ThresholdParameter2 = this->ThresholdParameter * this->ThresholdParameter;
  do
  {
    // Fill points with the closest points to each vertex in input
    if (this->ThresholdMode != vtkRobustIterativeClosestPointTransform::ThresholdFar)
    {
      closestp->SetNumberOfPoints(
        nb_points); // It was set above, but at this point it should be removed from there
      for (i = 0; i < nb_points; i++)
      {
        this->Locator->FindClosestPoint(a->GetPoint(i), outPoint, cell_id, sub_id, dist2);
        closestp->SetPoint(i, outPoint);
      }
      this->LandmarkTransform->SetSourceLandmarks(a);
    }
    else
    {
      // use a ThresholdParameter
      vtkNew<vtkPoints> sourceLandmarks;
      this->UpdateLandmarks(a, nb_points, closestp, sourceLandmarks.GetPointer());
      vtkDebugMacro(<< "Number of source landmarks: " << sourceLandmarks->GetNumberOfPoints());
      this->LandmarkTransform->SetSourceLandmarks(sourceLandmarks);
    }
    // Build the landmark transform

    this->LandmarkTransform->SetTargetLandmarks(closestp);
    this->LandmarkTransform->Update();

    // Concatenate (can't use this->Concatenate directly)

    accumulate->Concatenate(this->LandmarkTransform->GetMatrix());

    this->NumberOfIterations++;
    vtkDebugMacro(<< "Iteration: " << this->NumberOfIterations);
    if (this->NumberOfIterations >= this->MaximumNumberOfIterations)
    {
      break;
    }

    // Move mesh and compute mean distance (running) if needed

    if (this->CheckMeanDistance)
    {
      totaldist = 0.0;
    }

    for (i = 0; i < nb_points; i++)
    {
      a->GetPoint(i, p1);
      this->LandmarkTransform->InternalTransformPoint(p1, p2);
      b->SetPoint(i, p2);
      if (this->CheckMeanDistance)
      {
        if (this->MeanDistanceMode == VTK_ICP_MODE_RMS)
        {
          totaldist += vtkMath::Distance2BetweenPoints(p1, p2);
        }
        else
        {
          totaldist += sqrt(vtkMath::Distance2BetweenPoints(p1, p2));
        }
      }
    }

    if (this->CheckMeanDistance)
    {
      if (this->MeanDistanceMode == VTK_ICP_MODE_RMS)
      {
        this->MeanDistance = sqrt(totaldist / (double)nb_points);
      }
      else
      {
        this->MeanDistance = totaldist / (double)nb_points;
      }
      vtkDebugMacro("Mean distance: " << this->MeanDistance);
      if (this->MeanDistance <= this->MaximumMeanDistance)
      {
        break;
      }
    }

    temp = a;
    a = b;
    b = temp;

  } while (1);

  // Now recover accumulated result
  this->Matrix->DeepCopy(accumulate->GetMatrix());
}

//----------------------------------------------------------------------------

void vtkRobustIterativeClosestPointTransform::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  if (this->Source)
  {
    os << indent << "Source: " << this->Source << "\n";
  }
  else
  {
    os << indent << "Source: (none)\n";
  }

  if (this->Target)
  {
    os << indent << "Target: " << this->Target << "\n";
  }
  else
  {
    os << indent << "Target: (none)\n";
  }

  if (this->Locator)
  {
    os << indent << "Locator: " << this->Locator << "\n";
  }
  else
  {
    os << indent << "Locator: (none)\n";
  }

  os << indent << "MaximumNumberOfIterations: " << this->MaximumNumberOfIterations << "\n";
  os << indent << "CheckMeanDistance: " << this->CheckMeanDistance << "\n";
  os << indent << "MeanDistanceMode: " << this->GetMeanDistanceModeAsString() << "\n";
  os << indent << "MaximumMeanDistance: " << this->MaximumMeanDistance << "\n";
  os << indent << "MaximumNumberOfLandmarks: " << this->MaximumNumberOfLandmarks << "\n";
  os << indent << "StartByMatchingCentroids: " << this->StartByMatchingCentroids << "\n";
  os << indent << "StoreLastMeanDistance: " << this->StoreLastMeanDistance << "\n";
  os << indent << "NumberOfIterations: " << this->NumberOfIterations << "\n";
  os << indent << "MeanDistance: " << this->MeanDistance << "\n";
  os << indent << "LastMeanDistance: " << this->LastMeanDistance << "\n";
  if (this->LandmarkTransform)
  {
    os << indent << "LandmarkTransform:\n";
    this->LandmarkTransform->PrintSelf(os, indent.GetNextIndent());
  }
  os << indent << "ThresholdParameter: " << this->ThresholdParameter << "\n";
}
