/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkTriangulate25D.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkTriangulate25D
 * @brief   Triangulate a 2.5 dimensional dataset.
 *

 * Triangulate25D is a filter that constructs a triangulation from
 * input points arranged in a structured grid. The triangulation can
 * be tuned by specifying a maximum edge-length and a depth range. The
 * triangulation is made filling out each quad with at most two
 * triangles and the two triangles formed will be ones with the
 * shortest edge-lengths. The input grid must be two-dimensional,
 * i.e. at least one of the dimensions must a singleton. The maximum
 * range works such that the resulting triangulation will span from
 * the lowest value of the singleton dimension and the extent will
 * never be more than the maximum depth range.
 *
 * In addition to geometric constraints, the points can be filtered
 * using an upper, a lower, and a range threshold. The threshold is
 * applied to the active points scalars. The threshold works much like
 * the vtkThreshold filter. The vtkThreshold filter cannot be used in
 * conjunction with this filter, since the output vtkTreshold is an
 * unstructured grid.
 *
 * The filter can be configured to run in-place and the triangulation
 * can be added to the point and cell data in arrays named
 * "Marks". The output on port 0 equals the input except that the
 * "Marks" are added. On port 1, a vtkPolyData output can be
 * requested.
 *
 * It is a work-in-progress.
 */

#ifndef vtkTriangulate25D_h
#define vtkTriangulate25D_h

#include "vtkFiltersCoreModule.h" // For export macro
#include "vtkPassInputTypeAlgorithm.h"

class vtkInformationVector;
class vtkInformation;
class vtkPolyData;
class vtkUnsignedShortArray;

class VTKSPSFILTERSCORE_EXPORT vtkTriangulate25D : public vtkPassInputTypeAlgorithm
{
public:
  vtkTypeMacro(vtkTriangulate25D, vtkPassInputTypeAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   *
   */
  static vtkTriangulate25D* New();

  ///@{
  /**
   * Specify whether to generate polydata on port 1.
   *
   */
  vtkSetMacro(GeneratePolyData, vtkTypeBool);
  vtkGetMacro(GeneratePolyData, vtkTypeBool);
  vtkBooleanMacro(GeneratePolyData, vtkTypeBool);
  ///@}

  ///@{
  /**
   * Specify the maximum depth range of the output.
   *
   */
  vtkSetClampMacro(MaxRange, double, 0.0, VTK_DOUBLE_MAX);
  vtkGetMacro(MaxRange, double);
  ///@}

  ///@{
  /**
   * Specify the maximum edge length allowed in the triangulation.
   *
   */
  vtkSetClampMacro(MaxEdgeLength, double, 0.0, VTK_DOUBLE_MAX);
  vtkGetMacro(MaxEdgeLength, double);
  ///@}

  ///@{
  /**
   * Control how the decision of valid / invalid is made with
   * multi-component data.  The choices are allow all to pass
   * (UseAllPass), to use the selected component (specified in the
   * SelectedComponent ivar), or to look at all components. When
   * looking at all components, the evaluation can pass if all the
   * components satisfy the rule (UseAll) or if any satisfy is
   * (UseAny). The default value is UseSelected.
   */

  enum ComponentModeType : int
  {
    UseAllPass = 0,
    UseSelected = 1,
    UseAll = 2,
    UseAny = 3,
    UseNorm = 4
  };

  vtkSetClampMacro(ComponentMode, int, UseAllPass, UseNorm);
  vtkGetMacro(ComponentMode, int);
  void SetComponentModeToUseSelected() { this->SetComponentMode(UseSelected); }
  void SetComponentModeToUseAll() { this->SetComponentMode(UseAll); }
  void SetComponentModeToUseAny() { this->SetComponentMode(UseAny); }
  void SetComponentModeToUseAllPass() { this->SetComponentMode(UseAllPass); }
  void SetComponentModeToUseNorm() { this->SetComponentMode(UseNorm); }
  const char* GetComponentModeAsString();
  ///@}

  ///@{
  /**
   * When the component mode is UseSelected, this ivar indicated the selected
   * component. The default value is 0.
   */
  vtkSetClampMacro(SelectedComponent, int, 0, VTK_INT_MAX);
  vtkGetMacro(SelectedComponent, int);
  ///@}

  int Lower(double s) const;
  int Upper(double s) const;
  int Between(double s) const;

  /**
   * Possible values for the threshold function:
   * - THRESHOLD_BETWEEN - Keep values between the lower and upper thresholds.
   * - THRESHOLD_LOWER - Keep values below the lower threshold.
   * - THRESHOLD_UPPER - Keep values above the upper threshold.
   */
  enum ThresholdType : int
  {
    ThresholdBetween = 0,
    ThresholdLower = 1,
    ThresholdUpper = 2,
  };

  ///@{
  /**
   * Get/Set the threshold method, defining which threshold bounds to use. The default method is
   * vtkThreshold::Between.
   */
  void SetThresholdFunction(int function);
  int GetThresholdFunction();
  ///@}

  ///@{
  /**
   * Set/get the upper and lower thresholds. The default values are set to +infinity and -infinity,
   * respectively.
   */
  vtkSetMacro(UpperThreshold, double);
  vtkSetMacro(LowerThreshold, double);
  vtkGetMacro(UpperThreshold, double);
  vtkGetMacro(LowerThreshold, double);
  ///@}

protected:
  vtkTriangulate25D();
  ~vtkTriangulate25D() override;

  vtkPolyData* GetPolyDataOutput();

  int FillInputPortInformation(int port, vtkInformation* info) override;

  int FillOutputPortInformation(int port, vtkInformation* info) override;

  int RequestDataObject(vtkInformation* request, vtkInformationVector** inputVector,
    vtkInformationVector* outputVector) override;

  int RequestInformation(vtkInformation* request, vtkInformationVector** inputVector,
    vtkInformationVector* outputVector) override;

  int RequestUpdateTime(vtkInformation* vtkNotUsed(request), vtkInformationVector** inputVector,
    vtkInformationVector* outputVector) override;

  int RequestUpdateTimeDependentInformation(vtkInformation* request,
    vtkInformationVector** inputVector, vtkInformationVector* outputVector) override;

  int RequestUpdateExtent(vtkInformation* vtkNotUsed(request), vtkInformationVector** inputVector,
    vtkInformationVector* outputVector) override;

  int RequestData(vtkInformation* request, vtkInformationVector** inputVector,
    vtkInformationVector* outputVector) override;

  int CheckAbort() { return 0; }
#ifndef __linux__
  int GetAbortOutput() { return 0; }
#endif
  double MaxEdgeLength;
  double MaxRange;
  vtkTypeBool GeneratePolyData = 1;
  int ComponentMode = vtkTriangulate25D::UseAllPass;
  double LowerThreshold;
  double UpperThreshold;
  int NumberOfComponents;
  int SelectedComponent = 0;

  int (vtkTriangulate25D::*ThresholdFunction)(double s) const = &vtkTriangulate25D::Between;

  double RangeLow;
  double RangeHigh;
  template <typename TScalarsArray>
  int EvaluateComponents(TScalarsArray& scalars, vtkIdType id);

  template <typename TPoints, typename TScalarArray>
  struct EvaluatePointsFunctor;
  struct EvaluatePointsWorker;

  template <typename TPoints>
  struct ComputeMarksFunctor;
  struct ComputeMarksWorker;

  template <typename TPoints, typename MarksT>
  struct ComputePolyDataFunctor;

private:
  vtkTriangulate25D(const vtkTriangulate25D&) = delete;
  void operator=(const vtkTriangulate25D&) = delete;
  ///@{
  /*
   * Used for indexing singleton direction
   */
  enum TwoDimensionType : int
  {
    YZ = 0,
    XZ = 1,
    XY = 2,
  };
  ///@}
  TwoDimensionType TwoDimType = XY;
};

#endif // vtkTriangulate25D_h
