#include "vtkTriangulate25D.h"

#include "vtkArrayDispatch.h"
#include "vtkCellArray.h"
#include "vtkCellArrayIterator.h"
#include "vtkCellData.h"
#include "vtkDataArray.h"
#include "vtkDataArrayRange.h"
#include "vtkDataSetAttributes.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkPointData.h"
#include "vtkPolyData.h"
#include "vtkRectilinearGrid.h"
#include "vtkSMPThreadLocal.h"
#include "vtkSMPThreadLocalObject.h"
#include "vtkSMPTools.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkStructuredGrid.h"
#include "vtkUnsignedCharArray.h"
#include "vtkUnsignedShortArray.h"

#include <array>

#ifndef SQUARE
#define SQUARE(z) ((z) * (z))
#endif

vtkStandardNewMacro(vtkTriangulate25D);

// TODO: Look into vtkImageInPlaceFilter and add support for in-place
//       operations on the grid
//       ArrayDispatch on the remaining functors

vtkTriangulate25D::vtkTriangulate25D()
{
  this->MaxEdgeLength = 10.0;
  this->MaxRange = 100.0;
  this->SetNumberOfOutputPorts(2);
  this->SetInputArrayToProcess(0, /* index */
    0,                            /* port */
    0,                            /* connection */
    vtkDataObject::FIELD_ASSOCIATION_POINTS, vtkDataSetAttributes::SCALARS);
  this->UpperThreshold = 255;
  this->LowerThreshold = 0;
  // Private variables used by functors
  this->NumberOfComponents = 3;
  this->RangeLow = 0.0;
  this->RangeHigh = 0.0;
}

vtkTriangulate25D::~vtkTriangulate25D() {}

//------------------------------------------------------------------------------
int vtkTriangulate25D::Lower(double s) const
{
  return (s <= this->LowerThreshold ? 1 : 0);
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::Upper(double s) const
{
  return (s >= this->UpperThreshold ? 1 : 0);
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::Between(double s) const
{
  return (s >= this->LowerThreshold ? (s <= this->UpperThreshold ? 1 : 0) : 0);
}

//------------------------------------------------------------------------------
template <typename TScalarsArray>
int vtkTriangulate25D::EvaluateComponents(TScalarsArray& scalars, vtkIdType id)
{
  // TODO: Avoid static_cast
  int keepPoint = 0;
  double sqrLength = 0.0;
  int c;
  switch (this->ComponentMode)
  {
    case vtkTriangulate25D::UseSelected:
      c = this->SelectedComponent < this->NumberOfComponents ? this->SelectedComponent : 0;
      keepPoint = (this->*(this->ThresholdFunction))(static_cast<double>(scalars[id][c]));
      break;
    case vtkTriangulate25D::UseAny:
      keepPoint = 0;
      for (c = 0; (!keepPoint) && (c < this->NumberOfComponents); c++)
      {
        keepPoint = (this->*(this->ThresholdFunction))(static_cast<double>(scalars[id][c]));
      }
      break;
    case vtkTriangulate25D::UseAll:
      keepPoint = 1;
      for (c = 0; keepPoint && (c < this->NumberOfComponents); c++)
      {
        keepPoint = (this->*(this->ThresholdFunction))(static_cast<double>(scalars[id][c]));
      }
      break;
    case vtkTriangulate25D::UseNorm:
      for (c = 0; (c < this->NumberOfComponents); c++)
      {
        sqrLength += SQUARE(static_cast<double>(scalars[id][c]));
      }
      keepPoint = (this->*(this->ThresholdFunction))(sqrt(sqrLength));
      break;
    default:
      keepPoint = 1;
  }
  return keepPoint;
}

namespace
{

//@{
// Marks used for points and cells. Their bit-masks do not overlap, so it is
// possible to store marks only on points
enum MarkType : unsigned short
{
  // Marks applied to points
  MarkPointValid = 1 << 0,
  MarkPointVertex = 1 << 1,
  MarkPointBoundary = 1 << 2,

  // Marks applied to quad cells
  MarkQuadUpper = 1 << 3,
  MarkQuadLower = 1 << 4,
  MarkQuadDiagonal = 1 << 5, // 1 if UR-LL, 0 if UL-LR
  MarkQuadDiagonalUpper = MarkQuadDiagonal | MarkQuadUpper,
  MarkQuadDiagonalLower = MarkQuadDiagonal | MarkQuadLower,
};
//@}
}

template <typename TPoints, typename TScalarArray>
struct vtkTriangulate25D::EvaluatePointsFunctor
{
  using TLS = vtkSMPThreadLocal<std::array<double, 2>>;
  TLS TLMinMaxDepth;

  vtkTriangulate25D* Self;
  vtkDataSet* Input;
  TPoints* Points;
  TScalarArray* ScalarsArray;
  vtkIdType NumberOfPoints;
  double RangeLow;
  double RangeHigh;
  vtkNew<vtkUnsignedCharArray> ValidityArray;

  EvaluatePointsFunctor(
    vtkTriangulate25D* self, vtkDataSet* input, TPoints* points, TScalarArray* scalarsArray)
    : Self(self)
    , Input(input)
    , Points(points)
    , ScalarsArray(scalarsArray)
    , NumberOfPoints(input->GetNumberOfPoints())
    , RangeLow(0.0)
    , RangeHigh(0.0)
  {
    this->ValidityArray->SetNumberOfComponents(1);
    this->ValidityArray->SetNumberOfValues(this->NumberOfPoints);
    auto validityArray = vtk::DataArrayValueRange<1>(this->ValidityArray);
    std::fill(std::begin(validityArray), std::end(validityArray), 0);
    if (this->NumberOfPoints > 0)
    {
      // ensure that internal structures are initialized.
      this->Input->GetPoint(0);
    }
  }

  void Initialize()
  {
    std::array<double, 2>& minmax = this->TLMinMaxDepth.Local();
    minmax[0] = VTK_DOUBLE_MAX;
    minmax[1] = VTK_DOUBLE_MIN;
  }

  void operator()(vtkIdType begin, vtkIdType end)
  {
    const bool isFirst = false; // vtkSMPTools::GetSingleThread();

    const auto& scalars = vtk::DataArrayTupleRange(this->ScalarsArray);
    const auto& points = vtk::DataArrayTupleRange<3>(this->Points);
    auto validity = vtk::DataArrayValueRange<1>(this->ValidityArray);

    vtkIdType checkAbortInterval = std::min((end - begin) / 10 + 1, (vtkIdType)1000);

    // TODO: API type of input and max
    double minDepth = VTK_DOUBLE_MAX;
    double maxDepth = VTK_DOUBLE_MIN;
    for (vtkIdType pointId = begin; pointId < end; ++pointId)
    {
      if (pointId % checkAbortInterval == 0)
      {
        if (isFirst)
        {
          this->Self->CheckAbort();
        }
        if (this->Self->GetAbortOutput())
        {
          break;
        }
      }

      int keepPoint = 0;
      keepPoint = this->Self->EvaluateComponents(scalars, pointId);
      validity[pointId] = keepPoint ? MarkPointValid : 0;
      if (keepPoint)
      {
        auto point = points[pointId];
        double depth = point[Self->TwoDimType];
        minDepth = std::min(minDepth, depth);
        maxDepth = std::max(maxDepth, depth);
      }

      if (isFirst)
      {
        this->Self->UpdateProgress(end * 1.0 / this->NumberOfPoints * 1.0 / 3);
      }
    }
    std::array<double, 2>& minmax = this->TLMinMaxDepth.Local();
    minmax[0] = std::min(minmax[0], minDepth);
    minmax[1] = std::max(minmax[1], maxDepth);
  }

  void Reduce()
  {
    using TLSIter = TLS::iterator;
    TLSIter end = this->TLMinMaxDepth.end();
    double minDepth = VTK_DOUBLE_MAX;
    double maxDepth = VTK_DOUBLE_MIN;
    for (TLSIter itr = this->TLMinMaxDepth.begin(); itr != end; ++itr)
    {
      std::array<double, 2>& minmax = *itr;
      minDepth = std::min(minDepth, minmax[0]);
      maxDepth = std::max(maxDepth, minmax[1]);
    }
    if (maxDepth > 0.0)
    {
      this->RangeLow = minDepth;
      this->RangeHigh = this->RangeLow + Self->MaxRange;
    }
    else
    {
      this->RangeHigh = maxDepth;
      this->RangeLow = this->RangeHigh - Self->MaxRange;
    }
  }
};

//------------------------------------------------------------------------------
struct vtkTriangulate25D::EvaluatePointsWorker
{
  vtkSmartPointer<vtkUnsignedCharArray> ValidMask;
  EvaluatePointsWorker() { ValidMask = nullptr; }

  // Dispatch 2 arrays
  template <typename TPoints, typename TScalarArray>
  void operator()(
    TPoints* points, TScalarArray* scalarsArray, vtkTriangulate25D* self, vtkDataSet* input)
  {
    EvaluatePointsFunctor<TPoints, TScalarArray> functor(self, input, points, scalarsArray);
    vtkSMPTools::For(0, input->GetNumberOfPoints(), functor);
    self->RangeLow = functor.RangeLow;
    self->RangeHigh = functor.RangeHigh;
    this->ValidMask = functor.ValidityArray;
  }
};

//------------------------------------------------------------------------------
template <typename TPoints>
struct vtkTriangulate25D::ComputeMarksFunctor
{
  vtkStructuredGrid* Input;
  TPoints* Points;
  vtkUnsignedCharArray* Weights;
  vtkTriangulate25D* Self;

  vtkSmartPointer<vtkUnsignedShortArray> CellMarks;      // Output
  vtkSmartPointer<vtkUnsignedShortArray> PointMarks;     // Output
  vtkSmartPointer<vtkUnsignedShortArray> TempPointMarks; // Written from multiple threads

  double MaxSqrEdge;
  std::array<std::array<vtkIdType, 4>, 3> ShiftLUT;
  std::array<int, 3> CellDims;
  vtkIdType DepthIndex;
  vtkIdType CyStride, CzStride;
  vtkIdType PyStride, PzStride;

  template <typename TupleRangeT1, typename TupleRangeT2, typename TupleRangeT3, typename T>
  inline bool CanCreateTriangle(
    const TupleRangeT1& p0, const TupleRangeT2& p1, const TupleRangeT3& p2, T MaxSqrEdge)
  {
    return (vtkMath::Distance2BetweenPoints(p0, p1) < MaxSqrEdge &&
      vtkMath::Distance2BetweenPoints(p0, p2) < MaxSqrEdge);
  }

  // Can we infer type of ArrayT from TGrid?
  ComputeMarksFunctor(vtkTriangulate25D* self, vtkStructuredGrid* input, TPoints* points,
    vtkUnsignedCharArray* weights)
    : Self(self)
    , Input(input)
    , Points(points)
    , Weights(weights)
  {
    int gridDims[3];
    this->Input->GetDimensions(gridDims);
    vtkIdType nPoints = this->Input->GetNumberOfPoints();
    vtkIdType nCells = this->Input->GetNumberOfCells();

    this->PointMarks = vtkSmartPointer<vtkUnsignedShortArray>::New();
    this->PointMarks->SetNumberOfComponents(1);
    this->PointMarks->SetNumberOfValues(nPoints);
    auto pointMarks = vtk::DataArrayValueRange<1>(this->PointMarks);
    std::fill(std::begin(pointMarks), std::end(pointMarks), 0);
    this->CellMarks = vtkSmartPointer<vtkUnsignedShortArray>::New();
    this->CellMarks->SetNumberOfComponents(1);
    this->CellMarks->SetNumberOfValues(nCells);
    auto cellMarks = vtk::DataArrayValueRange<1>(this->CellMarks);
    std::fill(std::begin(cellMarks), std::end(cellMarks), 0);

    this->TempPointMarks = vtkSmartPointer<vtkUnsignedShortArray>::New();
    this->TempPointMarks->SetNumberOfComponents(1);
    this->TempPointMarks->SetNumberOfValues(nPoints);

    this->MaxSqrEdge = SQUARE(Self->GetMaxEdgeLength());

    // Shift LUT for cell traversal
    const std::array<vtkIdType, 4> shiftLUTx = { 0, 1, 1, 0 };
    const std::array<vtkIdType, 4> shiftLUTy = { 0, 0, 1, 1 };
    const std::array<vtkIdType, 4> shiftLUTz = { 0, 0, 0, 0 };

    if (Self->TwoDimType == vtkTriangulate25D::XZ)
    {
      this->ShiftLUT[0] = shiftLUTx;
      this->ShiftLUT[1] = shiftLUTz;
      this->ShiftLUT[2] = shiftLUTy;
      this->DepthIndex = 1;
    }
    else if (Self->TwoDimType == vtkTriangulate25D::YZ)
    {
      this->ShiftLUT[0] = shiftLUTy;
      this->ShiftLUT[1] = shiftLUTz;
      this->ShiftLUT[2] = shiftLUTx;
      this->DepthIndex = 0;
    }
    else
    {
      this->ShiftLUT[0] = shiftLUTx;
      this->ShiftLUT[1] = shiftLUTy;
      this->ShiftLUT[2] = shiftLUTz;
      this->DepthIndex = 2;
    }

    // Compute strides
    this->CellDims = { gridDims[0] - 1, gridDims[1] - 1, gridDims[2] - 1 };
    this->CyStride = (this->CellDims[0] ? this->CellDims[0] : 1);
    this->CzStride =
      (this->CellDims[0] ? this->CellDims[0] : 1) * (this->CellDims[1] ? this->CellDims[1] : 1);
    this->PyStride = gridDims[0];
    this->PzStride = gridDims[0] * gridDims[1];
  }

  void operator()(vtkIdType beginCellId, vtkIdType endCellId)
  {
    vtkIdType theCellI, theCellJ, theCellK;

    const auto inRange = vtk::DataArrayTupleRange<3>(this->Points);

    // TODO: Replace with functor
    const auto weights = vtk::DataArrayValueRange<1>(this->Weights);

    auto pointMarks = vtk::DataArrayValueRange<1>(this->PointMarks);
    auto cellMarks = vtk::DataArrayValueRange<1>(this->CellMarks);
    auto tempPointMarks = vtk::DataArrayValueRange<1>(this->TempPointMarks);

    /*
    // TODO: Figure out how to use tuple references
    using PointReference = PointType::TupleReferenceType;
    PointReference vertices[4];
    */

    vtkIdType pointIndices[4]{ 0 };

    // Note: The extra information about whether points are MarkValidPoint or
    //       MarkVertexPoint is not needed for triangulation, but with this
    //       information, it is possible to see the faith of point only
    //       using the "Marks" array of the point data.
    for (vtkIdType cellId = beginCellId; cellId < endCellId; cellId++)
    {
      bool usable[4]{ false, false, false, false };
      cellMarks[cellId] = 0;
      theCellI = (this->CellDims[0] > 0 ? cellId % this->CellDims[0] : 0);
      theCellJ = (this->CellDims[1] > 0 ? (cellId / this->CyStride) % this->CellDims[1] : 0);
      theCellK = (this->CellDims[2] > 0 ? (cellId / this->CzStride) : 0);

      //    0----1
      //    |    |
      //    |    |
      //    3----2

      int nUsable = 0;
      for (size_t j = 0; j < 4; j++)
      {
        vtkIdType cellPointIndex = (theCellI + this->ShiftLUT[0][j]) +
          (theCellJ + this->ShiftLUT[1][j]) * this->PyStride +
          (theCellK + this->ShiftLUT[2][j]) * this->PzStride;
        pointIndices[j] = cellPointIndex;

        // std::cout << inRange[cellPointIndex][this->DepthIndex] << std::endl;
        // TODO: Assign vertex references
        if ((weights[cellPointIndex] > 0) &&
          (inRange[cellPointIndex][Self->TwoDimType] < Self->RangeHigh) &&
          (inRange[cellPointIndex][Self->TwoDimType] > Self->RangeLow))
        {
          // Mark point valid
          usable[j] = true;
          pointMarks[cellPointIndex] = MarkPointValid;
          nUsable++;
        }
      }
      const vtkIdType i0 = pointIndices[0];
      const vtkIdType i1 = pointIndices[1];
      const vtkIdType i2 = pointIndices[2];
      const vtkIdType i3 = pointIndices[3];

      // TODO: Use vertex references instead
      const auto p0 = inRange[i0];
      const auto p1 = inRange[i1];
      const auto p2 = inRange[i2];
      const auto p3 = inRange[i3];

      // We need at least 3 vertices to form a triangle
      if (nUsable > 2)
      {
        double sqrDist02 = usable[0] && usable[2] ? vtkMath::Distance2BetweenPoints(p0, p2) : 0.0;
        double sqrDist13 = usable[1] && usable[3] ? vtkMath::Distance2BetweenPoints(p1, p3) : 0.0;

        // Use distances between local vertices
        if (sqrDist02 > 0 && sqrDist02 < sqrDist13)
        {
          if (sqrDist02 < this->MaxSqrEdge)
          {
            if (usable[3] && CanCreateTriangle(p3, p0, p2, this->MaxSqrEdge))
            {
              cellMarks[cellId] |= MarkQuadLower;
              tempPointMarks[i3] = MarkPointVertex;
              tempPointMarks[i0] = MarkPointVertex;
              tempPointMarks[i2] = MarkPointVertex;
            }
            if (usable[1] && CanCreateTriangle(p1, p0, p2, this->MaxSqrEdge))
            {
              cellMarks[cellId] |= MarkQuadUpper;
              tempPointMarks[i1] = MarkPointVertex;
              tempPointMarks[i0] = MarkPointVertex;
              tempPointMarks[i2] = MarkPointVertex;
            }
            cellMarks[cellId] |= MarkQuadDiagonal;
          }
        }
        else if (sqrDist13 > 0 && sqrDist13 < this->MaxSqrEdge)
        {
          if (usable[2] && CanCreateTriangle(p2, p1, p3, this->MaxSqrEdge))
          {
            cellMarks[cellId] |= MarkQuadLower;
            tempPointMarks[i2] = MarkPointVertex;
            tempPointMarks[i1] = MarkPointVertex;
            tempPointMarks[i3] = MarkPointVertex;
          }
          if (usable[0] && CanCreateTriangle(p0, p1, p3, this->MaxSqrEdge))
          {
            cellMarks[cellId] |= MarkQuadUpper;
            tempPointMarks[i0] = MarkPointVertex;
            tempPointMarks[i1] = MarkPointVertex;
            tempPointMarks[i3] = MarkPointVertex;
          }
        }
      }
    }
  }

  void Reduce()
  {
    // Marking points as vertices (single thread)
    auto pointMarks = vtk::DataArrayValueRange<1>(this->PointMarks);
    auto tempPointMarks = vtk::DataArrayValueRange<1>(this->TempPointMarks);
    auto weights = vtk::DataArrayValueRange<1>(this->Weights);
    for (vtkIdType pointId = 0; pointId < pointMarks.size(); pointId++)
    {
      // Mark points as vertices
      pointMarks[pointId] |= tempPointMarks[pointId];
      // Transfer valid marks
      pointMarks[pointId] |= (unsigned short)weights[pointId];
    }
  }

  void Execute()
  {
    vtkSMPTools::For(0, static_cast<vtkIdType>(this->Input->GetNumberOfCells()), *this);
  }
};

//------------------------------------------------------------------------------
struct vtkTriangulate25D::ComputeMarksWorker
{
  vtkUnsignedShortArray* PointMarks;
  vtkUnsignedShortArray* CellMarks;
  ComputeMarksWorker()
    : PointMarks(nullptr)
    , CellMarks(nullptr)
  {
  }
  template <typename TPoints>
  void operator()(TPoints* points, vtkTriangulate25D* self, vtkStructuredGrid* input,
    vtkUnsignedCharArray* validMask)
  {
    ComputeMarksFunctor<TPoints> functor(self, input, points, validMask);
    vtkSMPTools::For(0, input->GetNumberOfPoints(), functor);
    this->PointMarks = functor.PointMarks;
    this->CellMarks = functor.CellMarks;
  }
};

//------------------------------------------------------------------------------
template <typename TPoints, typename MarksT>
struct vtkTriangulate25D::ComputePolyDataFunctor
{
  using TLS = vtkSMPThreadLocalObject<vtkCellArray>;
  TLS TLTriangles;

  vtkStructuredGrid* Input;
  TPoints* Points;
  MarksT* PointMarks;
  MarksT* CellMarks;
  vtkTriangulate25D* Self;
  int CyStride = 0;
  int CzStride = 0;
  int PyStride = 0;
  int PzStride = 0;
  std::array<std::array<vtkIdType, 4>, 3> ShiftLUT;
  std::array<int, 3> CellDims;

  ComputePolyDataFunctor(vtkTriangulate25D* self, vtkStructuredGrid* input, TPoints* points,
    MarksT* pointMarks, MarksT* cellMarks)
    : Input(input)
    , Points(points)
    , PointMarks(pointMarks)
    , CellMarks(cellMarks)
    , Self(self)
  {
    int gridDims[3];
    this->Input->GetDimensions(gridDims);

    // Shift LUT for cell traversal
    const std::array<vtkIdType, 4> shiftLUTx = { 0, 1, 1, 0 };
    const std::array<vtkIdType, 4> shiftLUTy = { 0, 0, 1, 1 };
    const std::array<vtkIdType, 4> shiftLUTz = { 0, 0, 0, 0 };

    if (Self->TwoDimType == vtkTriangulate25D::XZ)
    {
      ShiftLUT[0] = shiftLUTx;
      ShiftLUT[1] = shiftLUTz;
      ShiftLUT[2] = shiftLUTy;
    }
    else if (Self->TwoDimType == vtkTriangulate25D::YZ)
    {
      ShiftLUT[0] = shiftLUTy;
      ShiftLUT[1] = shiftLUTz;
      ShiftLUT[2] = shiftLUTx;
    }
    else
    {
      ShiftLUT[0] = shiftLUTx;
      ShiftLUT[1] = shiftLUTy;
      ShiftLUT[2] = shiftLUTz;
    }

    // Compute strides
    CellDims = { gridDims[0] - 1, gridDims[1] - 1, gridDims[2] - 1 };
    this->CyStride = (CellDims[0] ? CellDims[0] : 1);
    this->CzStride = (CellDims[0] ? CellDims[0] : 1) * (CellDims[1] ? CellDims[1] : 1);
    this->PyStride = gridDims[0];
    this->PzStride = gridDims[0] * gridDims[1];
  }

  inline void ComputeCellIndex(const std::array<int, 3>& cellDims, const vtkIdType& cellId,
    const vtkIdType& yStride, const vtkIdType& zStride, vtkIdType& cellI, vtkIdType& cellJ,
    vtkIdType& cellK)
  {
    cellI = (cellDims[0] > 0 ? cellId % cellDims[0] : 0);
    cellJ = (cellDims[1] > 0 ? (cellId / yStride) % cellDims[1] : 0);
    cellK = (cellDims[2] > 0 ? (cellId / zStride) : 0);
  }
  inline bool Boundary(unsigned short cellMark)
  {
    return ((cellMark & MarkQuadDiagonalUpper) == MarkQuadUpper) ||
      (((cellMark & MarkQuadDiagonalUpper) == MarkQuadDiagonalUpper) &&
        ((cellMark & MarkQuadDiagonalLower) == MarkQuadDiagonalLower));
  }
  void operator()(vtkIdType beginCellId, vtkIdType endCellId)
  {
    const auto inRange = vtk::DataArrayTupleRange<3>(this->Points);
    auto cellMarks = vtk::DataArrayValueRange<1>(this->CellMarks);
    auto pointMarks = vtk::DataArrayValueRange<1>(this->PointMarks);

    auto triangles = this->TLTriangles.Local();

    vtkIdType theCellI = 0;
    vtkIdType theCellJ = 0;
    vtkIdType theCellK = 0;

    vtkIdType pointIndices[4]{ 0 };

    for (vtkIdType cellId = beginCellId; cellId < endCellId; cellId++)
    {
      ComputeCellIndex(
        CellDims, cellId, this->CyStride, this->CzStride, theCellI, theCellJ, theCellK);

      //    0----1
      //    |    |
      //    |    |
      //    3----2

      for (size_t j = 0; j < 4; j++)
      {
        vtkIdType cellPointIndex = (theCellI + ShiftLUT[0][j]) +
          (theCellJ + ShiftLUT[1][j]) * this->PyStride +
          (theCellK + ShiftLUT[2][j]) * this->PzStride;
        pointIndices[j] = cellPointIndex;
      }
      const vtkIdType i0 = pointIndices[0];
      const vtkIdType i1 = pointIndices[1];
      const vtkIdType i2 = pointIndices[2];
      const vtkIdType i3 = pointIndices[3];

      // Update points if boundary
      vtkIdType cellIdLeft = cellId - 1;
      vtkIdType cellIdBelowLeft = cellId - 1 - this->CyStride;
      vtkIdType cellIdBelow = cellId - this->CyStride;

      vtkIdType i0x = i0 % this->PyStride;
      vtkIdType i0y = i0 / this->PyStride;

      // Test for boundary point. A point is not boundary if it is surrounded by triangles
      // Verify CellDims[1] is okay for xz-grid and yz-grid
      bool isNotBoundary = i0x > 0 && i0x < this->PyStride - 1 && i0y > 0 && i0y < CellDims[1];

      if (isNotBoundary)
      {
        isNotBoundary = Boundary(cellMarks[cellId]) && Boundary(cellMarks[cellIdLeft]) &&
          Boundary(cellMarks[cellIdBelowLeft]) && Boundary(cellMarks[cellIdBelow]);
      }

      if (!isNotBoundary)
      {
        pointMarks[i0] = (pointMarks[i0] | MarkPointBoundary);
      }

      vtkIdType nodes[3];

      if (((cellMarks[cellId]) & MarkQuadDiagonalLower) == MarkQuadLower)
      {
        nodes[0] = i1;
        nodes[1] = i3;
        nodes[2] = i2;
        triangles->InsertNextCell(3, nodes);
      }
      // UL
      if (((cellMarks[cellId]) & MarkQuadDiagonalUpper) == MarkQuadUpper)
      {
        nodes[0] = i0;
        nodes[1] = i3;
        nodes[2] = i1;
        triangles->InsertNextCell(3, nodes);
      }
      // LL
      if (((cellMarks[cellId]) & MarkQuadDiagonalLower) == MarkQuadDiagonalLower)
      {
        nodes[0] = i0;
        nodes[1] = i3;
        nodes[2] = i2;
        triangles->InsertNextCell(3, nodes);
      }
      // UR
      if (((cellMarks[cellId]) & MarkQuadDiagonalUpper) == MarkQuadDiagonalUpper)
      {
        nodes[0] = i0;
        nodes[1] = i2;
        nodes[2] = i1;
        triangles->InsertNextCell(3, nodes);
      }
    }
  }
  void Initialize() {}
  void Reduce()
  {
    // Merge triangles from different threads
    vtkPolyData* polyData = Self->GetPolyDataOutput();
    polyData->SetPoints(this->Input->GetPoints());
    polyData->GetPointData()->PassData(this->Input->GetPointData());

    using TLSIter = TLS::iterator;
    TLSIter end = this->TLTriangles.end();

    vtkNew<vtkCellArray> triangles;
    for (TLSIter itr = this->TLTriangles.begin(); itr != end; ++itr)
    {
      auto iter = vtk::TakeSmartPointer((*itr)->NewIterator());
      for (iter->GoToFirstCell(); !iter->IsDoneWithTraversal(); iter->GoToNextCell())
      {
        vtkIdType numCellPts;
        const vtkIdType* cellPts;
        iter->GetCurrentCell(numCellPts, cellPts);
        triangles->InsertNextCell(3, cellPts);
      }
    }
    polyData->SetPolys(triangles);
  }

  void Execute()
  {
    vtkSMPTools::For(0, static_cast<vtkIdType>(this->Input->GetNumberOfCells()), *this);
  }
};

/*
struct vtkTriangulate25D::ComputePolyDataWorker
{
  vtkUnsignedShortArray* CellMarks;
  vtkUnsignedShortArray* PointMarks;

  // Dispatch 2 arrays
  template <typename ArrayT, typename MarksT>
  void operator()(ArrayT* points, MarksT* pointMarks, MarksT* cellMarks, vtkTriangulate25D* self,
    vtkDataSet* input)
  {
    ComputePolyDataFunctor<vtkDataArray, vtkUnsignedShortArray> functor(
      input, points, pointMarks, cellMarks, self);
    vtkSMPTools::For(0, input->GetNumberOfPoints(), functor);
    this->PointMarks = functor.PointMarks;
    this->CellMarks = functor.CellMarks;
  }
};
*/

//------------------------------------------------------------------------------
void vtkTriangulate25D::SetThresholdFunction(int function)
{
  if (this->GetThresholdFunction() != function)
  {
    switch (function)
    {
      case vtkTriangulate25D::ThresholdBetween:
        this->ThresholdFunction = &vtkTriangulate25D::Between;
        break;
      case vtkTriangulate25D::ThresholdLower:
        this->ThresholdFunction = &vtkTriangulate25D::Lower;
        break;
      case vtkTriangulate25D::ThresholdUpper:
        this->ThresholdFunction = &vtkTriangulate25D::Upper;
        break;
    }

    this->Modified();
  }
}

int vtkTriangulate25D::GetThresholdFunction()
{
  if (this->ThresholdFunction == &vtkTriangulate25D::Between)
  {
    return vtkTriangulate25D::ThresholdBetween;
  }
  else if (this->ThresholdFunction == &vtkTriangulate25D::Lower)
  {
    return vtkTriangulate25D::ThresholdLower;
  }
  else if (this->ThresholdFunction == &vtkTriangulate25D::Upper)
  {
    return vtkTriangulate25D::ThresholdUpper;
  }

  // Added to avoid warning. Should never be reached.
  return -1;
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::FillInputPortInformation(int, vtkInformation* info)
{
  // This filter takes as input any structured data set, but currently
  // only vtkStructuredGrid and vtkRectilinearGrid. Consider narrowing down
  // the types at this point
  info->Set(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkDataSet");
  return 1;
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::FillOutputPortInformation(int port, vtkInformation* info)
{
  if (port == 0)
  {
    info->Set(vtkDataObject::DATA_TYPE_NAME(), "vtkDataSet");
  }
  else if (port == 1)
  {
    info->Set(vtkDataObject::DATA_TYPE_NAME(), "vtkPolyData");
  }
  return 1;
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::RequestUpdateExtent(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** vtkNotUsed(inputVector), vtkInformationVector* vtkNotUsed(outputVector))
{
  // At the moment we do not allow spatial streaming, such that a
  // subset can be processed.
  return 1;
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::RequestInformation(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** vtkNotUsed(inputVector), vtkInformationVector* vtkNotUsed(outputVector))
{
  return 1;
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::RequestUpdateTime(
  vtkInformation* request, vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  // Just forward time information
  return Superclass::RequestUpdateTime(request, inputVector, outputVector);
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::RequestUpdateTimeDependentInformation(
  vtkInformation* request, vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  // Just forward time information
  return Superclass::RequestUpdateTimeDependentInformation(request, inputVector, outputVector);
}

//------------------------------------------------------------------------------
void vtkTriangulate25D::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
  os << indent << "MaxEdgeLength: " << this->MaxEdgeLength << endl;
  os << indent << "MaxRange: " << this->MaxRange << endl;
  os << indent << "GeneratePolyData: " << this->GeneratePolyData << endl;
  os << indent << "Component Mode: " << this->GetComponentModeAsString() << endl;
  os << indent << "Selected Component: " << this->SelectedComponent << endl;
  if (this->ThresholdFunction == &vtkTriangulate25D::Upper)
  {
    os << indent << "Threshold By Upper\n";
  }

  else if (this->ThresholdFunction == &vtkTriangulate25D::Lower)
  {
    os << indent << "Threshold By Lower\n";
  }

  else if (this->ThresholdFunction == &vtkTriangulate25D::Between)
  {
    os << indent << "Threshold Between\n";
  }
}

//------------------------------------------------------------------------------
const char* vtkTriangulate25D::GetComponentModeAsString()
{
  if (this->ComponentMode == vtkTriangulate25D::UseSelected)
  {
    return "UseSelected";
  }
  else if (this->ComponentMode == vtkTriangulate25D::UseAny)
  {
    return "UseAny";
  }
  else if (this->ComponentMode == vtkTriangulate25D::UseAll)
  {
    return "UseAll";
  }
  else if (this->ComponentMode == vtkTriangulate25D::UseNorm)
  {
    return "UseNorm";
  }
  else
  {
    return "UseAllPass";
  }
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::RequestDataObject(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  if (this->GetNumberOfInputPorts() == 0 || this->GetNumberOfOutputPorts() == 0)
  {
    return 1;
  }

  // No information about input
  vtkInformation* inInfo = inputVector[0]->GetInformationObject(0);
  if (!inInfo)
  {
    return 0;
  }

  vtkDataObject* input = inInfo->Get(vtkDataObject::DATA_OBJECT());

  if (input)
  {
    vtkInformation* info = outputVector->GetInformationObject(0);
    vtkDataObject* output = info->Get(vtkDataObject::DATA_OBJECT());

    // Handle port 0
    if (!output || !output->IsA(input->GetClassName()))
    {
      vtkDataObject* newOutput = input->NewInstance();
      info->Set(vtkDataObject::DATA_OBJECT(), newOutput);
      newOutput->Delete();
    }

    // Handle port 1
    info = outputVector->GetInformationObject(1);
    if (info)
    {
      output = info->Get(vtkDataObject::DATA_OBJECT());
      if (output)
      {
        vtkPolyData* polyOutput = vtkPolyData::SafeDownCast(output);
        if (!polyOutput)
        {
          vtkErrorMacro(<< "Output on port 1 must be vtkPolyData");
          return 0;
        }
      }
    }
    return 1;
  }
  return 0;
}

//------------------------------------------------------------------------------
vtkPolyData* vtkTriangulate25D::GetPolyDataOutput()
{
  if (!this->GeneratePolyData)
  {
    return nullptr;
  }
  return vtkPolyData::SafeDownCast(this->GetExecutive()->GetOutputData(1));
}

//------------------------------------------------------------------------------
int vtkTriangulate25D::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  // input and output information objects
  vtkInformation* inInfo = inputVector[0]->GetInformationObject(0);
  vtkInformation* outInfo = outputVector->GetInformationObject(0);

  vtkStructuredGrid* inputSG =
    vtkStructuredGrid::SafeDownCast(inInfo->Get(vtkDataObject::DATA_OBJECT()));
  if (!inputSG)
  {
    vtkErrorMacro(<< "Only vtkStructuredGrid is supported");
    return 0;
  }
  else if (inputSG->GetNumberOfPoints() == 0)
  {
    vtkDebugMacro(<< "No input points");
    return 1;
  }

  // Read scalars for thresholding
  vtkDataArray* inScalars;

  inScalars = this->GetInputArrayToProcess(0, inputVector);

  // TODO: Consider filtering cells rather than points
  // int fieldAssociation = this->GetInputArrayAssociation(0, inputVector);

  int gridDims[3];
  inputSG->GetDimensions(gridDims);

  if (gridDims[0] <= 1)
  {
    this->TwoDimType = YZ;
  }
  else if (gridDims[1] <= 1)
  {
    this->TwoDimType = XZ;
  }
  else if (gridDims[2] <= 1)
  {
    this->TwoDimType = XY;
  }
  else
  {
    vtkErrorMacro(<< "Input must be two-dimensional");
    return 0;
  }
  this->NumberOfComponents = inScalars->GetNumberOfComponents();

  vtkStructuredGrid* outputSG =
    vtkStructuredGrid::SafeDownCast(outInfo->Get(vtkDataObject::DATA_OBJECT()));
  vtkPolyData* outputPolyData = nullptr;

  outputPolyData = this->GetPolyDataOutput();

  // Pass point and cell data to output
  outputSG->CopyStructure(inputSG);
  outputSG->GetPointData()->PassData(inputSG->GetPointData());
  outputSG->GetCellData()->PassData(inputSG->GetCellData());

  // Filter input and compute depth limits
  EvaluatePointsWorker pointWorker;
  if (!vtkArrayDispatch::Dispatch2::Execute(
        inputSG->GetPoints()->GetData(), inScalars, pointWorker, this, inputSG))
  {
    pointWorker(inputSG->GetPoints()->GetData(), inScalars, this, inputSG);
  }

  if (this->CheckAbort())
  {
    return 1;
  }

  vtkSmartPointer<vtkUnsignedShortArray> cellMarks;
  vtkSmartPointer<vtkUnsignedShortArray> pointMarks;

  auto marksFunctor = ComputeMarksFunctor<vtkDataArray>(
    this, inputSG, inputSG->GetPoints()->GetData(), pointWorker.ValidMask);
  marksFunctor.Execute();
  cellMarks = marksFunctor.CellMarks;
  pointMarks = marksFunctor.PointMarks;

  if (this->CheckAbort())
  {
    return 1;
  }

  using ComputePolyDataFunctorType = ComputePolyDataFunctor<vtkDataArray, vtkUnsignedShortArray>;

  // Issue with deleted function - when containing vtkSMPThreadLocal.....
  ComputePolyDataFunctorType polyDataFunctor(this, inputSG, inputSG->GetPoints()->GetData(),
    pointMarks.GetPointer(), cellMarks.GetPointer());

  polyDataFunctor.Execute();
  outputSG->GetPointData()->AddArray(polyDataFunctor.PointMarks);
  outputSG->GetPointData()->SetActiveScalars("Marks");

  outputSG->GetFieldData()->PassData(inputSG->GetFieldData());

  return 1;
}

/* Local variables: */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* c-basic-offset: 2 */
/* End: */
