// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#include "SMP/Common/vtkSMPToolsImpl.h"
#include "SMP/STDThread/vtkSMPToolsImpl.txx"

#include <cstdlib> // For std::getenv()
#include <thread>  // For std::thread::hardware_concurrency()

#ifdef __EMSCRIPTEN_PTHREAD__
#include <emscripten/threading.h>
#endif

namespace vtk
{
namespace detail
{
namespace smp
{
VTK_ABI_NAMESPACE_BEGIN
static int specifiedNumThreadsSTD; // Default initialized to zero

//------------------------------------------------------------------------------
int GetNumberOfThreadsSTDThread()
{
#ifdef __EMSCRIPTEN_PTHREAD__
  const int maxThreads = EM_ASM_INT(PThread.unusedWorkers);
#else
  const int maxThreads = std::thread::hardware_concurrency();
#endif
  return specifiedNumThreadsSTD ? specifiedNumThreadsSTD : maxThreads;
}

//------------------------------------------------------------------------------
template <>
void vtkSMPToolsImpl<BackendType::STDThread>::Initialize(int numThreads)
{
#ifdef __EMSCRIPTEN_PTHREAD__
  const int maxThreads = EM_ASM_INT(PThread.unusedWorkers);
#else
  const int maxThreads = std::thread::hardware_concurrency();
#endif
  if (numThreads == 0)
  {
    const char* vtkSmpNumThreads = std::getenv("VTK_SMP_MAX_THREADS");
    if (vtkSmpNumThreads)
    {
      numThreads = std::atoi(vtkSmpNumThreads);
    }
    else
    {
      specifiedNumThreadsSTD = 0;
    }
  }
  if (numThreads > 0)
  {
    numThreads = std::min(numThreads, maxThreads);
    specifiedNumThreadsSTD = numThreads;
  }
}

//------------------------------------------------------------------------------
template <>
int vtkSMPToolsImpl<BackendType::STDThread>::GetEstimatedNumberOfThreads()
{
#ifdef __EMSCRIPTEN_PTHREAD__
  const int maxThreadCount = EM_ASM_INT(PThread.unusedWorkers);
#else
  const int maxThreadCount = std::thread::hardware_concurrency();
#endif
  return specifiedNumThreadsSTD > 0 ? specifiedNumThreadsSTD : maxThreadCount;
}

//------------------------------------------------------------------------------
template <>
int vtkSMPToolsImpl<BackendType::STDThread>::GetEstimatedDefaultNumberOfThreads()
{
#ifdef __EMSCRIPTEN_PTHREAD__
  return EM_ASM_INT(PThread.unusedWorkers);
#else
  return std::thread::hardware_concurrency();
#endif
}

//------------------------------------------------------------------------------
template <>
bool vtkSMPToolsImpl<BackendType::STDThread>::GetSingleThread()
{
  return vtkSMPThreadPool::GetInstance().GetSingleThread();
}

//------------------------------------------------------------------------------
template <>
bool vtkSMPToolsImpl<BackendType::STDThread>::IsParallelScope()
{
  return vtkSMPThreadPool::GetInstance().IsParallelScope();
}

VTK_ABI_NAMESPACE_END
} // namespace smp
} // namespace detail
} // namespace vtk
