// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include <vtkActor.h>
#include <vtkCellData.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkRegressionTestImage.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkRenderer.h>
#include <vtkSTLReader.h>
#include <vtkSmartPointer.h>

#include <iostream>
#include <string>

int TestSTLReader(int argc, char* argv[])
{
  if (argc < 2)
  {
    std::cerr << "Required parameters: <filename>" << std::endl;
    return EXIT_FAILURE;
  }

  std::string inputFilename = argv[1];

  vtkNew<vtkSTLReader> reader;
  reader->SetFileName(inputFilename.c_str());
  reader->Update();

  // Check if header and solids match for ASCII STL
  if (reader->GetBinaryHeader() == nullptr) // check if ASCII
  {
    reader->ScalarTagsOn();
    reader->Update();

    double range[2];
    reader->GetOutput()->GetCellData()->GetScalars("STLSolidLabeling")->GetRange(range);
    int nSolids = static_cast<int>(range[1]) + 1;

    int nHeaders = 1; // At least one solid even when it does not have associated name

    std::string header(reader->GetHeader());
    if (!header.empty())
    {
      nHeaders += std::count(header.begin(), header.end(), '\n');
    }

    if (nSolids != nHeaders)
    {
      std::cerr << "Number of Solid Names in Header does not match with the number of solids"
                << std::endl;
      return EXIT_FAILURE;
    }

    reader->ScalarTagsOff();
    reader->Update();
  }

  // Visualize
  vtkSmartPointer<vtkPolyDataMapper> mapper = vtkSmartPointer<vtkPolyDataMapper>::New();
  mapper->SetInputConnection(reader->GetOutputPort());

  vtkSmartPointer<vtkActor> actor = vtkSmartPointer<vtkActor>::New();
  actor->SetMapper(mapper);

  vtkSmartPointer<vtkRenderer> renderer = vtkSmartPointer<vtkRenderer>::New();
  vtkSmartPointer<vtkRenderWindow> renderWindow = vtkSmartPointer<vtkRenderWindow>::New();
  renderWindow->AddRenderer(renderer);
  vtkSmartPointer<vtkRenderWindowInteractor> renderWindowInteractor =
    vtkSmartPointer<vtkRenderWindowInteractor>::New();
  renderWindowInteractor->SetRenderWindow(renderWindow);

  renderer->AddActor(actor);
  renderer->SetBackground(.3, .6, .3); // Background color green

  renderWindow->Render();

  int retVal = vtkRegressionTestImage(renderWindow);
  if (retVal == vtkRegressionTester::DO_INTERACTOR)
  {
    renderWindowInteractor->Start();
  }

  return !retVal;
}
