// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkCellData.h"
#include "vtkCubeSource.h"
#include "vtkDelaunay3D.h"
#include "vtkGenericDataObjectReader.h"
#include "vtkGenericDataObjectWriter.h"
#include "vtkIntArray.h"
#include "vtkPointData.h"
#include "vtkPolyData.h"
#include "vtkRectilinearGrid.h"
#include "vtkStructuredGrid.h"
#include "vtkTable.h"
#include "vtkUnstructuredGrid.h"
#include "vtkVariant.h"

#include <iostream>

void InitializeData(vtkPolyData* Data)
{
  vtkCubeSource* const source = vtkCubeSource::New();
  source->Update();

  Data->ShallowCopy(source->GetOutput());
  source->Delete();
}

bool CompareData(vtkPolyData* Output, vtkPolyData* Input)
{
  if (Input->GetNumberOfPoints() != Output->GetNumberOfPoints())
    return false;
  if (Input->GetNumberOfPolys() != Output->GetNumberOfPolys())
    return false;

  return true;
}

void InitializeData(vtkRectilinearGrid* Data)
{
  Data->SetDimensions(2, 3, 4);
}

bool CompareData(vtkRectilinearGrid* Output, vtkRectilinearGrid* Input)
{
  return memcmp(Input->GetDimensions(), Output->GetDimensions(), 3 * sizeof(int)) == 0;
}

void InitializeData(vtkStructuredGrid* Data)
{
  Data->SetDimensions(2, 3, 4);
}

bool CompareData(vtkStructuredGrid* Output, vtkStructuredGrid* Input)
{
  int inputDims[3];
  Input->GetDimensions(inputDims);

  int outputDims[3];
  Output->GetDimensions(outputDims);

  return memcmp(inputDims, outputDims, 3 * sizeof(int)) == 0;
}

void InitializeData(vtkTable* Data)
{
  vtkIntArray* const column1 = vtkIntArray::New();
  Data->AddColumn(column1);
  column1->Delete();
  column1->SetName("column1");

  vtkIntArray* const column2 = vtkIntArray::New();
  Data->AddColumn(column2);
  column2->Delete();
  column2->SetName("column2");

  Data->InsertNextBlankRow();
  Data->InsertNextBlankRow();
  Data->InsertNextBlankRow();

  Data->SetValue(0, 0, 1);
  Data->SetValue(0, 1, 2);
  Data->SetValue(1, 0, 3);
  Data->SetValue(1, 1, 4);
  Data->SetValue(2, 0, 5);
  Data->SetValue(2, 1, 6);
}

bool CompareData(vtkTable* Output, vtkTable* Input)
{
  if (Input->GetNumberOfColumns() != Output->GetNumberOfColumns())
    return false;
  if (Input->GetNumberOfRows() != Output->GetNumberOfRows())
    return false;

  for (int column = 0; column != Input->GetNumberOfColumns(); ++column)
  {
    for (int row = 0; row != Input->GetNumberOfRows(); ++row)
    {
      if (Input->GetValue(row, column).ToDouble() != Output->GetValue(row, column).ToDouble())
      {
        return false;
      }
    }
  }

  return true;
}

void InitializeData(vtkUnstructuredGrid* Data)
{
  vtkCubeSource* const source = vtkCubeSource::New();
  vtkDelaunay3D* const delaunay = vtkDelaunay3D::New();
  delaunay->AddInputConnection(source->GetOutputPort());
  delaunay->Update();

  Data->ShallowCopy(delaunay->GetOutput());

  delaunay->Delete();
  source->Delete();
}

bool CompareData(vtkUnstructuredGrid* Output, vtkUnstructuredGrid* Input)
{
  if (Input->GetNumberOfPoints() != Output->GetNumberOfPoints())
    return false;
  if (Input->GetNumberOfCells() != Output->GetNumberOfCells())
    return false;

  return true;
}

template <typename DataT>
bool TestDataObjectSerialization()
{
  DataT* const output_data = DataT::New();
  InitializeData(output_data);

  const char* const filename = output_data->GetClassName();

  vtkGenericDataObjectWriter* const writer = vtkGenericDataObjectWriter::New();
  writer->SetInputData(output_data);
  writer->SetFileName(filename);
  writer->Write();
  writer->Delete();

  vtkGenericDataObjectReader* const reader = vtkGenericDataObjectReader::New();
  reader->SetFileName(filename);
  reader->Update();

  vtkDataObject* obj = reader->GetOutput();
  DataT* const input_data = DataT::SafeDownCast(obj);
  if (!input_data)
  {
    reader->Delete();
    output_data->Delete();
    return false;
  }

  const bool result = CompareData(output_data, input_data);

  reader->Delete();
  output_data->Delete();

  return result;
}

int TestDataObjectIO(int /*argc*/, char* /*argv*/[])
{
  int result = 0;

  if (!TestDataObjectSerialization<vtkPolyData>())
  {
    std::cerr << "Error: failure serializing vtkPolyData" << std::endl;
    result = 1;
  }
  if (!TestDataObjectSerialization<vtkRectilinearGrid>())
  {
    std::cerr << "Error: failure serializing vtkRectilinearGrid" << std::endl;
    result = 1;
  }
  if (!TestDataObjectSerialization<vtkStructuredGrid>())
  {
    std::cerr << "Error: failure serializing vtkStructuredGrid" << std::endl;
    result = 1;
  }
  if (!TestDataObjectSerialization<vtkTable>())
  {
    std::cerr << "Error: failure serializing vtkTable" << std::endl;
    result = 1;
  }
  if (!TestDataObjectSerialization<vtkUnstructuredGrid>())
  {
    std::cerr << "Error: failure serializaing vtkUnstructuredGrid" << std::endl;
    result = 1;
  }

  return result;
}
