// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
// .NAME TestFFMPEGWriter - Tests vtkFFMPEGWriter.
// .SECTION Description
// Creates a scene and uses FFMPEGWriter to generate a movie file. Test passes
// if the file exists and has non zero length.

#include "vtkFFMPEGWriter.h"
#include "vtkImageCast.h"
#include "vtkImageData.h"
#include "vtkImageMandelbrotSource.h"
#include "vtkImageMapToColors.h"
#include "vtkLookupTable.h"
#include "vtksys/SystemTools.hxx"

#include <iostream>

int TestFFMPEGWriter(int vtkNotUsed(argc), char* vtkNotUsed(argv)[])
{
  int err = 0;
  int cc = 0;
  int exists = 0;
  unsigned long length = 0;
  vtkImageMandelbrotSource* Fractal0 = vtkImageMandelbrotSource::New();
  Fractal0->SetWholeExtent(0, 247, 0, 247, 0, 0);
  Fractal0->SetProjectionAxes(0, 1, 2);
  Fractal0->SetOriginCX(-1.75, -1.25, 0, 0);
  Fractal0->SetSizeCX(2.5, 2.5, 2, 1.5);
  Fractal0->SetMaximumNumberOfIterations(100);

  vtkImageCast* cast = vtkImageCast::New();
  cast->SetInputConnection(Fractal0->GetOutputPort());
  cast->SetOutputScalarTypeToUnsignedChar();

  vtkLookupTable* table = vtkLookupTable::New();
  table->SetTableRange(0, 100);
  table->SetNumberOfColors(100);
  table->Build();
  table->SetTableValue(99, 0, 0, 0);

  vtkImageMapToColors* colorize = vtkImageMapToColors::New();
  colorize->SetOutputFormatToRGB();
  colorize->SetLookupTable(table);
  colorize->SetInputConnection(cast->GetOutputPort());

  vtkFFMPEGWriter* w = vtkFFMPEGWriter::New();
  w->SetInputConnection(colorize->GetOutputPort());
  w->SetFileName("TestFFMPEGWriter.avi");
  std::cout << "Writing file TestFFMPEGWriter.avi..." << std::endl;
  w->SetBitRate(1024 * 1024 * 30);
  w->SetBitRateTolerance(1024 * 1024 * 3);
  w->Start();
  for (cc = 2; cc < 99; cc++)
  {
    std::cout << ".";
    Fractal0->SetMaximumNumberOfIterations(cc);
    table->SetTableRange(0, cc);
    table->SetNumberOfColors(cc);
    table->ForceBuild();
    table->SetTableValue(cc - 1, 0, 0, 0);
    w->Write();
  }
  w->End();
  std::cout << std::endl;
  std::cout << "Done writing file TestFFMPEGWriter.avi..." << std::endl;
  w->Delete();

  exists = (int)vtksys::SystemTools::FileExists("TestFFMPEGWriter.avi");
  length = vtksys::SystemTools::FileLength("TestFFMPEGWriter.avi");
  std::cout << "TestFFMPEGWriter.avi file exists: " << exists << std::endl;
  std::cout << "TestFFMPEGWriter.avi file length: " << length << std::endl;
  if (!exists)
  {
    err = 1;
    std::cerr << "ERROR: 1 - Test failing because TestFFMPEGWriter.avi file doesn't exist..."
              << std::endl;
  }
  else
  {
    vtksys::SystemTools::RemoveFile("TestFFMPEGWriter.avi");
  }
  if (0 == length)
  {
    err = 2;
    std::cerr << "ERROR: 2 - Test failing because TestFFMPEGWriter.avi file has zero length..."
              << std::endl;
  }

  colorize->Delete();
  table->Delete();
  cast->Delete();
  Fractal0->Delete();

  // err == 0 means test passes...
  //
  return err;
}
