import asyncio

from parat.services import ParaT, PipelineBuilder, PropertyManager
from paraview.modules.vtkRemotingPythonAsyncCore import (
    vtkPythonObservableWrapperUtilities,
)
import vtk
import sys


async def observeViewOutput(view):

    frameCounter = -1
    maxNumFrames = 999

    async for package in vtkPythonObservableWrapperUtilities.GetIterator(
        view.GetViewOutputObservable()
    ):

        if package:
            chunk = package.GetData()
            frameCounter += 1
            filename_suffix = f"{frameCounter}".zfill(len(str(maxNumFrames)))
            filename = f"frame-{filename_suffix}.bin"
            print(f"Write chunk {filename}")

            with open(filename, "wb") as f:
                # prepare the bytes string from chunk's values
                as_bytes = b"".join(
                    [
                        chunk.GetValue(i).to_bytes(1, sys.byteorder)
                        for i in range(chunk.GetNumberOfValues())
                    ]
                )
                f.write(as_bytes)
        else:
            continue

    print("Done")


async def main():
    App = ParaT()
    session = await App.initialize()

    builder = PipelineBuilder(session)

    sphere = await builder.CreateProxy(
        "sources", "SphereSource", ThetaResolution=80, Radius=2
    )
    shrink = await builder.CreateProxy(
        "filters", "ShrinkFilter", Input=sphere, ShrinkFactor=0.3
    )

    pmanager = PropertyManager()
    pmanager.SetValues(sphere, Radius=3, Center=[1, 2, 3], EndPhi=90)

    await asyncio.sleep(0.1)

    view = await builder.CreateProxy("views", "RenderView")
    chunkTask = asyncio.create_task(observeViewOutput(view))
    # Use VP9
    pmanager.SetValues(view, CodecType=0)
    # Turn on hardware acceleration
    pmanager.SetValues(view, UseHardwareAcceleration=False)
    # Use the highest quality for encoding
    pmanager.SetValues(view, Quality=100)
    # Enable client side decode.
    pmanager.SetValues(view, DecompressPackets=True)
    # Enable output stream.
    # When codec type is lossless, interaction will slow down due to I/O bottleneck in chunkTask.
    pmanager.SetValues(view, StreamOutput=True)

    representation = await builder.CreateRepresentation(
        shrink, 0, view, "GeometryRepresentation"
    )

    # before we 'update' the view, let's install an interactor on the view
    # so that the viewport dimensions, cameras of the client and server's render windows sync up.
    iren = vtk.vtkRenderWindowInteractor()
    iren.SetRenderWindow(view.GetRenderWindow())

    pmanager.Push(view)
    await pmanager.Update(view)
    if "-I" in sys.argv:
        while not iren.GetDone():
            iren.ProcessEvents()
            await asyncio.sleep(0.01)
            if iren.GetKeyCode() == "s":
                pmanager.SetVisibility(representation, view, True)
            if iren.GetKeyCode() == "h":
                pmanager.SetVisibility(representation, view, False)
    else:
        await asyncio.sleep(0.1)

    status = await builder.DeleteProxy(shrink)
    assert status == True

    # the view is destructed here.
    # on its way out, the output stream subscribers are unsubbed here,
    # the webm writer + decoder get finalized here.
    await App.finalize()


asyncio.run(main())
