/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkPTSPipelineViewer.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class vtkPTSPipelineViewer
 * @brief
 *
 *
 */

#ifndef vtkPTSPipelineViewer_h
#define vtkPTSPipelineViewer_h

#include "vtkObject.h"
#include "vtkRemotingMicroservicesModule.h" // for exports
#include "vtkSmartPointer.h"                // for vtkSmartPointer

#include "vtkPythonObservableWrapper.h" // for VTK_REMOTING_MAKE_PYTHON_OBSERVABLE
#include "vtkSMProxy.h" // due to vtkPythonObservableWrapper limitations full definition is required

#include <memory> // for std::unique_ptr

#include "vtkType.h"
#include "vtk_rxcpp.h" // for rxcpp
// clang-format off
// ideally, we include rx-lite.hpp here.
#include VTK_REMOTING_RXCPP(rx.hpp)
// clang-format on

class vtkClientSession;

class vtkPipelineElement;


/**
 * @brief Adaptor for vtkSMProxy for representing elements in the pipeline browser.
 */
class VTKREMOTINGMICROSERVICES_EXPORT vtkPipelineElement : public vtkObject
{
public:
  static vtkPipelineElement* New();
  vtkTypeMacro(vtkPipelineElement, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  std::vector<vtkSmartPointer<vtkSMProxy>> GetRepresentations() { return { Proxy }; }
  std::vector<char> GetVisibility() { return { true }; };

  vtkTypeInt32 GetID() { return this->Proxy->GetGlobalID(); }
  std::string GetName() { return ProxyName; }
  void SetName(const std::string& name) { ProxyName = name; };
  std::vector<vtkTypeInt32> GetParentIDs()
  {
    std::vector<vtkTypeInt32> ids;
    for (int i = 0; i < this->Proxy->GetNumberOfProducers(); i++)
    {
      ids.push_back(this->Proxy->GetProducerProxy(i)->GetGlobalID());
    }
    return ids;
  }

  void SetProxy(vtkSMProxy* proxy) { Proxy = proxy; };

private:
  vtkSMProxy* Proxy;
  // this is the name shown on the ui.
  std::string ProxyName;

protected:
  vtkPipelineElement();
  ~vtkPipelineElement() override;

private:
  vtkPipelineElement(const vtkPipelineElement&) = delete;
  void operator=(const vtkPipelineElement&) = delete;
};

class VTKREMOTINGMICROSERVICES_EXPORT vtkPTSPipelineViewer : public vtkObject
{
public:
  static vtkPTSPipelineViewer* New();
  vtkTypeMacro(vtkPTSPipelineViewer, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  rxcpp::observable<std::vector<vtkSmartPointer<vtkPipelineElement>>> GetObservable() const;
  VTK_REMOTING_MAKE_PYTHON_OBSERVABLE(
    std::vector<vtkSmartPointer<vtkObject>>, GetObservable() const);

  ///@{
  /**
   * Get/set the session. Changing the session will cause the current/selection to
   * change and hence trigger `on_next` call on the subscribed observables, if any.
   */
  void SetSession(vtkClientSession* session);
  vtkClientSession* GetSession() const;
  ///@}

  void UnsubscribeAll();

protected:
  vtkPTSPipelineViewer();
  ~vtkPTSPipelineViewer() override;

private:
  vtkPTSPipelineViewer(const vtkPTSPipelineViewer&) = delete;
  void operator=(const vtkPTSPipelineViewer&) = delete;

  class vtkInternals;
  std::unique_ptr<vtkInternals> Internals;
};

#endif
