/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkPTSDefinitionManager.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class vtkPTSDefinitionManager
 * @brief A class to track definition of a proxy
 *
 *
 */

#ifndef vtkPTSDefinitionManager_h
#define vtkPTSDefinitionManager_h

#include "vtkObject.h"
#include "vtkPVXMLElement.h" // due to vtkPythonObservableWrapper limitations full definition is required
#include "vtkPropertyWidgetDecorator.h"
#include "vtkProxyAdapter.h" // due to vtkPythonObservableWrapper limitations full definition is required
#include "vtkPythonObservableWrapper.h"     // for VTK_REMOTING_MAKE_PYTHON_OBSERVABLE
#include "vtkRemotingMicroservicesModule.h" // for exports
#include "vtkSmartPointer.h"                // for vtkSmartPointer

#include <memory> // for std::unique_ptr

#include "vtk_rxcpp.h" // for rxcpp
// clang-format off
// ideally, we include rx-lite.hpp here.
#include VTK_REMOTING_RXCPP(rx.hpp)
// clang-format on

class vtkClientSession;

class VTKREMOTINGMICROSERVICES_EXPORT vtkPTSDefinitionManager : public vtkObject
{
public:
  static vtkPTSDefinitionManager* New();
  vtkTypeMacro(vtkPTSDefinitionManager, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  ///@{
  /**
   * Returns an observable for the definition of the proxy
   *
   * The return value is a cold observable i.e. it does not start emitting
   * values until something subscribes to it.
   *
   * @section Triggers Triggers
   *
   * * on_next: @TODO every time the session is updated
   *
   * * on_error: if session is not set
   *
   * * on_completed: when this instance is destroyed.
   */
  rxcpp::observable<vtkSmartPointer<vtkProxyAdapter>> GetDefinition(vtkSMProxy* smproxy) const;
  VTK_REMOTING_MAKE_PYTHON_OBSERVABLE(
    vtkSmartPointer<vtkObject>, GetDefinition(vtkSMProxy* smproxy) const);

  rxcpp::observable<vtkSmartPointer<vtkProxyAdapter>> GetDefinition(
    const std::string& group, const std::string& name) const;
  VTK_REMOTING_MAKE_PYTHON_OBSERVABLE(vtkSmartPointer<vtkObject>,
    GetDefinition(const std::string& group, const std::string& name) const);
  ///@}

  ///@{
  /**
   * Returns an observable for the xml layout of a proxy
   *
   * The return value is a cold observable i.e. it does not start emitting
   * values until something subscribes to it.
   *
   * @section Triggers Triggers
   *
   * * on_next: @TODO every time the session is updated
   *
   * * on_error: if session is not set
   *
   * * on_completed: when this instance is destroyed.
   */
  rxcpp::observable<vtkSmartPointer<vtkPVXMLElement>> GetLayout(vtkSMProxy* smproxy) const;
  VTK_REMOTING_MAKE_PYTHON_OBSERVABLE(
    vtkSmartPointer<vtkObject>, GetLayout(vtkSMProxy* smproxy) const);

  rxcpp::observable<vtkSmartPointer<vtkPVXMLElement>> GetLayout(
    const std::string& group, const std::string& name) const;

  VTK_REMOTING_MAKE_PYTHON_OBSERVABLE(
    vtkSmartPointer<vtkObject>, GetLayout(const std::string& group, const std::string& name) const);
  ///@}

  std::vector<vtkSmartPointer<vtkPropertyWidgetDecorator>> GetPropertyDecorators(
    vtkSMProxy* proxy, const std::string& propertyName) const;

  std::vector<vtkSmartPointer<vtkPropertyWidgetDecorator>> GetPropertyGroupDecorators(
    vtkSMProxy* proxy, const std::string& propertyGroup) const;

  ///@{
  /**
   * Get/set the session.
   */
  void SetSession(vtkClientSession* session);
  vtkClientSession* GetSession() const;
  ///@}

protected:
  vtkPTSDefinitionManager();
  ~vtkPTSDefinitionManager() override;

private:
  vtkPTSDefinitionManager(const vtkPTSDefinitionManager&) = delete;
  void operator=(const vtkPTSDefinitionManager&) = delete;

  class vtkInternals;
  std::unique_ptr<vtkInternals> Internals;
};

#endif
