/*=========================================================================

  Program:   ParaView
  Module:    vtkPythonAsyncCoreUtilities.txx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#ifndef vtkPythonAsyncCoreUtilities_txx
#define vtkPythonAsyncCoreUtilities_txx

/* A set of utilities for setting a value of a asyncio.future from C++ */

#include "vtkPython.h" // include this first

#include "vtkPythonUtil.h"
#include "vtkSMSourceProxy.h"
#include "vtkSmartPointer.h"

#include <typeinfo> // operator typeid

// Check if any errors occurred on the python interpreter.
// If there is print the erro message and return true.
inline bool CheckAndClearError()
{
  if (PyObject* exception = PyErr_Occurred())
  {
    PyErr_Print();
    PyErr_Clear();
    return true;
  }
  return false;
}

//----------------------------------------------------------------------------
// Create an exception of type `AsyncCore.Internal` and contents equal to `message`
// and set it as exception for `future` .
inline void SetFutureException(PyObject* future, const std::string& message)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  PyObject* pyException = PyErr_NewException("AsyncCore.Internal", PyExc_Exception, nullptr);
  if (pyException == nullptr)
  {
    CheckAndClearError();
    throw std::runtime_error("error creating exception object");
  }
  PyErr_SetString(pyException, message.c_str());

  if (PyObject_CallMethod(future, "set_exception", "O", pyException) == nullptr)
  {
    CheckAndClearError();
    const std::string errorMessage =
      std::string("error setting future's exception : ") + std::string(" | ") + message;
    throw std::runtime_error(errorMessage.c_str());
  }
}

//----------------------------------------------------------------------------
template <typename T>
void SetFutureValue(PyObject* future, T value)
{
  const std::string message = std::string("Undefined template specialization for this type") +
    std::string(typeid(value).name());
  SetFutureException(future, message);
}

//============================================================================
// Template specializations
//----------------------------------------------------------------------------
template <>
inline void SetFutureValue(PyObject* future, bool value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  if (PyObject_CallMethod(future, "set_result", "O", PyBool_FromLong(value)) == nullptr)
  {
    SetFutureException(future, "Error setting future's value");
  }
}

//----------------------------------------------------------------------------
template <>
inline void SetFutureValue(PyObject* future, vtkTypeUInt32 value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  if (PyObject_CallMethod(future, "set_result", "I", value) == nullptr)
  {
    SetFutureException(future, "Error setting future's value");
  }
}

//----------------------------------------------------------------------------
template <>
inline void SetFutureValue(PyObject* future, const std::string& value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  if (PyObject_CallMethod(future, "set_result", "s", value.c_str()) == nullptr)
  {
    SetFutureException(future, "Error setting future's value");
  }
}

//----------------------------------------------------------------------------
template <typename T>
inline void SetFutureValue(PyObject* future, vtkSmartPointer<T> value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  PyObject* pyValue = vtkPythonUtil::GetObjectFromPointer(value);
  if (PyObject_CallMethod(future, "set_result", "O", pyValue) == nullptr)
  {
    SetFutureException(future, "Error setting future's value");
  }
  Py_DECREF(pyValue);
}

//----------------------------------------------------------------------------
template <typename T,
  typename SFINAE = typename std::enable_if<std::is_base_of<vtkObject, T>::value>::type>
inline void SetFutureValue(PyObject* future, T* value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  PyObject* pyValue = vtkPythonUtil::GetObjectFromPointer(value);
  if (PyObject_CallMethod(future, "set_result", "O", pyValue) == nullptr)
  {
    SetFutureException(future, "Error setting future's value");
  }
  // pyValue is passed to the future so we can erase it now.
  Py_DECREF(pyValue);
  //  vtkPythonUtil::GetObjectFromPointer transfers the ownership to pyValue so we can reduce
  //  reference counter
  value->Delete();
}

//============================================================================
// Push an element to a asyncio.Queue
//----------------------------------------------------------------------------
template <typename T>
void PushToQueue(PyObject* queue, T value)
{
  (void)queue;
  (void)value;
  throw std::runtime_error("error pushing to iterator's queue");
}

//----------------------------------------------------------------------------
template <>
inline void PushToQueue(PyObject* queue, int value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  if (PyObject_CallMethod(queue, "put_nowait", "i", value) == nullptr)
  {
    CheckAndClearError();
    throw std::runtime_error("error pushing to iterator's queue");
  }
}

//----------------------------------------------------------------------------
template <typename T>
void PushToQueue(PyObject* queue, vtkSmartPointer<T> value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  PyObject* pyValue = vtkPythonUtil::GetObjectFromPointer(value);
  if (PyObject_CallMethod(queue, "put_nowait", "O", pyValue) == nullptr)
  {
    CheckAndClearError();
    throw std::runtime_error("error pushing to iterator's queue");
  }
  // FIXME is this required ?
  // Py_DECREF(pyValue);
}

#endif
