r""" Utilities for rxcpp  Observables in python"""

import asyncio
from paraview.servermanager import Proxy as pythonProxy
from paraview.modules.vtkRemotingServerManager import vtkSMProxy
from paraview.modules.vtkRemotingPythonAsyncCore import vtkPythonContextHelper as Helper


class ProxyContextManager:
    """ProxyContextManager is the helper  that allows to expose a method of a
    proxy that returns an observable to a structure easily manageable from
    python"""

    def __init__(self, proxy):
        _proxy = None
        if isinstance(proxy, pythonProxy):
            _proxy = proxy.SMProxy
        elif isinstance(proxy, vtkSMProxy):
            _proxy = proxy
        else:
            raise TypeError("argument is not of type vtkSMProxy")
        self._helper = Helper()
        self._helper.SetProxy(_proxy)
        self._iterator = None

    def __enter__(self):
        self._iterator = self._helper.NewIterator()
        return self

    def __exit__(self, type, value, traceback):
        self._helper.Unsubscribe()
        del self._iterator

    def values(self):
        return self._iterator


# TODO should become internal class of ProxyContextManager)
class QueueIterator:
    """An Iterator for an asyncio.Queue object

    Allows a consumer to iterate the elements of a queue using an `async for ` clause.
    Producer may push elements through the queue object supplied to the constructor.
    `done` should be set once the producer stops pushing elements. In that case the
    iterator will consume any elemnts in the queue and exit.

    The producer may use `queue.join()` to monitor when the queue is empty
    """

    def __init__(self, queue, done):
        if not isinstance(queue, asyncio.Queue):
            raise TypeError("first argument is not of type asyncio.Queue")

        if not isinstance(done, asyncio.Future):
            raise TypeError("second argument is not of type asyncio.Future")

        self._Queue = queue
        self._future = done

    def get_queue(self):
        return self._Queue

    def get_future(self):
        return self._future

    def __aiter__(self):
        return self

    async def __anext__(self):
        while True:
            if self._future.cancelled():
                raise StopAsyncIteration
            if self._Queue.qsize() == 0 and self._future.done():
                raise StopAsyncIteration
            if self._Queue.qsize() > 0:
                f = self._Queue.get_nowait()
                self._Queue.task_done()
                return f
            else:
                # Wait until an element is pushed or future is set
                task = asyncio.create_task(self._Queue.get())
                done, _ = await asyncio.wait(
                    {task, self._future},
                    return_when=asyncio.FIRST_COMPLETED,
                )
                if task in done:
                    self._Queue.task_done()
                    return task.result()
                else:
                    task.cancel()
