// SPDX-FileCopyrightText: Copyright (c) Copyright 2021 NVIDIA Corporation
// SPDX-License-Identifier: BSD-3-Clause

#ifndef vtknvindex_sparse_volume_importer_h
#define vtknvindex_sparse_volume_importer_h

#include <mi/dice.h>
#include <nv/index/idistributed_data_import_callback.h>
#include <nv/index/isparse_volume_subset.h>

#include "vtknvindex_utilities.h"

// Fragmented job to import in parallel all brick storage pieces
class vtknvindex_import_bricks
  : public mi::neuraylib::Fragmented_job<0x6538523c, 0xba06, 0x4227, 0xbe, 0xb4, 0x2, 0xdf, 0x9,
      0xb6, 0x69, 0xa4>
{
public:
  vtknvindex_import_bricks(
    const nv::index::ISparse_volume_subset_data_descriptor* subset_data_descriptor,
    nv::index::ISparse_volume_subset* volume_subset, const mi::Uint8* source_buffer,
    mi::Size vol_fmt_size, const std::string& source_scalar_type, mi::Sint32 border_size,
    mi::Sint32 ghost_levels, const vtknvindex::util::Bbox3i& source_bbox,
    const vtknvindex_volume_neighbor_data* neighbor_data);

  void execute_fragment(mi::neuraylib::IDice_transaction* dice_transaction, mi::Size index,
    mi::Size count, const mi::neuraylib::IJob_execution_context* context) override;

  mi::Size get_nb_fragments() const;

private:
  const nv::index::ISparse_volume_subset_data_descriptor* m_subset_data_descriptor;
  nv::index::ISparse_volume_subset* m_volume_subset;
  const mi::Uint8* m_source_buffer;
  mi::Size m_vol_fmt_size;
  std::string m_source_scalar_type;
  mi::Sint32 m_border_size;
  mi::Sint32 m_ghost_levels;
  mi::Size m_nb_fragments;
  mi::Size m_nb_bricks;
  vtknvindex::util::Bbox3i m_source_bbox;

  const vtknvindex_volume_neighbor_data* m_neighbor_data;
};

// The class vtknvindex_sparse_volume_importer represents a distributed data importer for NVIDIA
// IndeX to load subsets of a sparse volume dataset based shared memory.
class vtknvindex_sparse_volume_importer
  : public nv::index::Distributed_discrete_data_import_callback<0x5c35b7ce, 0x496c, 0x4342, 0xa3,
      0x5f, 0x9d, 0x85, 0x4, 0xa1, 0x69, 0x7e>
{
public:
  vtknvindex_sparse_volume_importer();

  vtknvindex_sparse_volume_importer(const mi::math::Vector_struct<mi::Uint32, 3>& volume_size,
    mi::Sint32 border_size, mi::Sint32 ghost_levels, const std::string& scalar_type,
    mi::Sint32 scalar_components);

  // Estimates the volume data size inside the bounding box (in bytes).
  mi::Size estimate(const mi::math::Bbox_struct<mi::Sint32, 3>& bounding_box,
    mi::neuraylib::IDice_transaction* dice_transaction) const override;
  using Self::estimate; // handle overloaded method

  // NVIDIA IndeX triggers this callback if time varying data shall be imported.
  nv::index::IDistributed_data_subset* create(
    const mi::math::Bbox_struct<mi::Sint32, 3>& bounding_box, mi::Uint32 time_step,
    nv::index::IData_subset_factory* factory,
    mi::neuraylib::IDice_transaction* dice_transaction) const override;

  // Create internal storage of ParaView's subset inside bounding box.
  nv::index::IDistributed_data_subset* create(
    const mi::math::Bbox_struct<mi::Sint32, 3>& bounding_box,
    nv::index::IData_subset_factory* factory,
    mi::neuraylib::IDice_transaction* dice_transaction) const override;
  using Self::create; // handle overloaded method

  // The cluster properties triggered by ParaView.
  void set_cluster_properties(vtknvindex_cluster_properties* host_properties);

  // DiCE methods.
  void serialize(mi::neuraylib::ISerializer* serializer) const override;
  void deserialize(mi::neuraylib::IDeserializer* deserializer) override;
  mi::base::Uuid subset_id() const override;

private:
  mi::Sint32 m_border_size;                      // IndeX subcube border size.
  mi::Sint32 m_ghost_levels;                     // VTK ghost levels.
  mi::math::Vector<mi::Uint32, 3> m_volume_size; // Volume size.
  std::string m_scalar_type;                     // Volume scalar type as string.
  mi::Sint32 m_scalar_components;
  vtknvindex_cluster_properties* m_cluster_properties; // Cluster properties.
};

#endif // vtknvindex_sparse_volume_importer_h
