import asyncio

from parat.services import ParaT, PipelineBuilder, DefinitionManager

HAS_YAML = False
try:
    import yaml

    HAS_YAML = True
except ImportError:
    pass

import os


def write_definition(definition, filename):
    if HAS_YAML:
        base_path = os.getcwd()
        with open(base_path + "/" + filename, "w") as test:
            test.write(yaml.dump(definition.as_dict()))


async def main():
    App = ParaT()
    session = await App.initialize()

    builder = PipelineBuilder(session)

    sphere = await builder.CreateSource(
        "sources", "SphereSource", ThetaResolution=80, Radius=2
    )
    wavelet = await builder.CreateSource("sources", "RTAnalyticSource")

    shrink = await builder.CreateFilter(
        "filters", "ShrinkFilter", Input=sphere, ShrinkFactor=0.3
    )

    definitionManager = DefinitionManager(session)

    shrinkDef = await definitionManager.GetDefinition(shrink)
    sphereDef = await definitionManager.GetDefinition(sphere)

    print(dir(sphereDef))
    print(type(sphereDef))
    print(sphereDef.as_dict())

    write_definition(sphereDef, "sphere.yaml")
    write_definition(shrinkDef, "shrinkDef.yaml")

    definition = await definitionManager.GetDefinition("sources", "RTAnalyticSource")

    write_definition(definition, "wavelet.yaml")

    definition = await definitionManager.GetDefinition("filters", "Contour")

    write_definition(definition, "contour.yaml")

    definition = await definitionManager.GetDefinition(
        "representations", "GeometryRepresentation"
    )

    write_definition(definition, "geometry.yaml")

    result = DefinitionManager.vtkPVXMLElement_to_string(
        await definitionManager.GetLayout(sphere)
    )
    print(result)
    result = DefinitionManager.vtkPVXMLElement_to_string(
        await definitionManager.GetLayout(wavelet)
    )
    print(result)

    result = DefinitionManager.vtkPVXMLElement_to_string(
        await definitionManager.GetLayout("sources", "RTAnalyticSource")
    )
    print(result)
    result = DefinitionManager.vtkPVXMLElement_to_string(
        await definitionManager.GetLayout("representations", "GeometryRepresentation")
    )
    print(result)

    #  ===================== Decorators =====================

    point2cell = await builder.CreateFilter(
        "filters", "PointDataToCellData", Input=sphere
    )

    definition = await definitionManager.GetDefinition(point2cell)
    write_definition(definition, "point2cell.yaml")

    for prop in definition.GetProperties():
        if len(prop.GetDecorators()) != 0:
            print(prop.GetDecorators())
            decorators = definitionManager.GetPropertyDecorators(
                point2cell, prop.GetName()
            )
            for decorator in decorators:
                print(
                    decorator.canShowWidget(False)
                )  # should we show widget (show_advanced = False) ?
                print(decorator.enableWidget())  # should we enable the widget ?

    await asyncio.sleep(0.1)


asyncio.run(main())
