/*=========================================================================

  Program:   ParaView
  Module:    vtkPythonObservableWrapperUtilities.txx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPythonObservableWrapperUtilities_txx
#define vtkPythonObservableWrapperUtilities_txx
#include "vtkPythonObservableWrapperUtilities.h"

#include "vtkLogger.h"
#include "vtkPythonAsyncCoreUtilities.txx"
#include "vtkPythonRunLoop.h"
#include "vtkSmartPyObject.h"

template <typename T>
PyObject* vtkPythonObservableWrapperUtilities::GetFutureInternal(
  vtkPythonObservableWrapper<T> wrapper)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  PyObject* future = vtkPythonRunLoop::GetInstance()->CreateFuture();
  rxcpp::observable<T> observable = wrapper.GetObservable();
  observable.subscribe(

    // on_next
    [future](T value) {
      try
      {
        SetFutureValue(future, value);
      }
      catch (const std::exception& e)
      {
        vtkLogF(ERROR, "%s", e.what());
        throw e;
      }
    },
    // on_error
    [future](std::exception_ptr eptr) {
      try
      {
        std::rethrow_exception(eptr);
      }
      catch (const std::exception& e)
      {
        SetFutureException(future, e.what());
      }
    }
    // on_completed ????
  );

  return future;
}

//----------------------------------------------------------------------------
template <typename T>
PyObject* vtkPythonObservableWrapperUtilities::GetIteratorInternal(
  vtkPythonObservableWrapper<T> wrapper)
{
  PyObject* iterator = vtkPythonRunLoop::GetInstance()->CreateAsyncIterator();
  rxcpp::observable<T> observable = wrapper.GetObservable();

  // Increase the reference count in case the iterator is discarded in the python side before the
  // end of the iterable
  Py_INCREF(iterator);
  observable.subscribe(

    // on_next
    [iterator](T value) {
      vtkPythonScopeGilEnsurer gilEnsurer;
      PyObject* queue = PyObject_CallMethod(iterator, "get_queue", "");
      if (queue == nullptr)
      {
        CheckAndClearError();
        throw std::runtime_error("error getting iterator's queue");
      }
      PushToQueue(queue, value);
    },

    // on_error
    [iterator](std::exception_ptr eptr) {
      vtkPythonScopeGilEnsurer gilEnsurer;
      PyObject* future = PyObject_CallMethod(iterator, "get_future", "");
      if (future == nullptr)
      {
        CheckAndClearError();
        throw std::runtime_error("error getting iterator's future");
      }
      try
      {
        std::rethrow_exception(eptr);
      }
      catch (const std::exception& e)
      {
        SetFutureException(future, e.what());
      }
    },

    // on_completed
    [iterator]() {
      vtkPythonScopeGilEnsurer gilEnsurer;
      PyObject* future = PyObject_CallMethod(iterator, "get_future", "");
      if (future == nullptr)
      {
        CheckAndClearError();
        throw std::runtime_error("error getting iterator's future");
      }

      SetFutureValue(future, true);
      Py_DECREF(iterator);
    });
  return iterator;
}

#endif
