/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmQTWrapCPPCommand.h"

#include "cmCustomCommandLines.h"
#include "cmMakefile.h"
#include "cmSourceFile.h"
#include "cmSystemTools.h"

#include <iterator>
#include <utility>

class cmExecutionStatus;

// cmQTWrapCPPCommand
bool cmQTWrapCPPCommand::InitialPass(std::vector<std::string> const& args,
                                     cmExecutionStatus&)
{
  if (args.size() < 3) {
    this->SetError("called with incorrect number of arguments");
    return false;
  }

  // Set default values:
  std::string moc_exe =
    this->Makefile->GetSafeDefinition("QT_MOC_EXECUTABLE");
  std::string sourceList = args[1];
  std::string prefix = "moc_";
  std::string extension = "cxx";
  std::vector<std::string> sources;

  // Parse arguments:
  bool isExtension = false;
  bool isPrefix = false;
  bool isExecutable = false;

  for (auto i = (std::begin(args) + 2); i != std::end(args); ++i) {
    if (isExtension) {
      extension = *i;
      isExtension = false;
    } else if (isPrefix) {
      prefix = *i;
      isPrefix = false;
    } else if (isExecutable) {
      moc_exe = *i;
      isExecutable = false;
    } else if (*i == "EXTENSION") {
      isExtension = true;
    } else if (*i == "PREFIX") {
      isPrefix = true;
    } else if (*i == "EXECUTABLE") {
      isExecutable = true;
    } else {
      sources.emplace_back(std::move(*i));
    }
  }

  // Validate:
  if (isExtension) {
    this->SetError("EXTENSION not specified.");
    return false;
  }
  if (isPrefix) {
    this->SetError("PREFIX not specified.");
    return false;
  }
  if (isExecutable) {
    this->SetError("EXECUTABLE not specified.");
    return false;
  }
  if (sources.empty()) {
    this->SetError("No sources found.");
    return false;
  }

  // Validate input:
  if (moc_exe.empty()) {
    this->SetError("No moc binary set. Please use the EXECUTABLE "
                   "option or set QT_MOC_EXECUTABLE.");
    return false;
  }

  // Get the variable holding the list of sources.
  std::string sourceListValue = this->Makefile->GetSafeDefinition(sourceList);

  // Create a rule for all sources listed.
  for (const std::string &j : sources) {
    cmSourceFile* curr = this->Makefile->GetSource(j);
    // if we should wrap the class
    if (!(curr && curr->GetPropertyAsBool("WRAP_EXCLUDE"))) {
      // Compute the name of the file to generate.
      std::string srcName = cmSystemTools::GetFilenameWithoutLastExtension(j);
      std::string newName = this->Makefile->GetCurrentBinaryDirectory();
      newName += '/' + prefix + srcName + "." + extension;
      cmSourceFile* sf = this->Makefile->GetOrCreateSource(newName, true);
      if (curr) {
        sf->SetProperty("ABSTRACT", curr->GetProperty("ABSTRACT"));
      }

      // Compute the name of the header from which to generate the file.
      std::string hname;
      if (cmSystemTools::FileIsFullPath(j)) {
        hname = j;
      } else {
        if (curr && curr->GetPropertyAsBool("GENERATED")) {
          hname = this->Makefile->GetCurrentBinaryDirectory();
        } else {
          hname = this->Makefile->GetCurrentSourceDirectory();
        }
        hname += "/";
        hname += j;
      }

      // Append the generated source file to the list.
      if (!sourceListValue.empty()) {
        sourceListValue += ";";
      }
      sourceListValue += newName;

      // Create the custom command to generate the file.
      cmCustomCommandLine commandLine;
      commandLine.push_back(moc_exe);
      commandLine.push_back("-o");
      commandLine.push_back(newName);
      commandLine.push_back(hname);

      cmCustomCommandLines commandLines;
      commandLines.push_back(std::move(commandLine));

      std::vector<std::string> depends;
      depends.push_back(moc_exe);
      depends.push_back(hname);

      std::string no_main_dependency;
      const char* no_working_dir = nullptr;
      this->Makefile->AddCustomCommandToOutput(
        newName, depends, no_main_dependency, commandLines, "Qt Wrapped File",
        no_working_dir);
    }
  }

  // Store the final list of source files.
  this->Makefile->AddDefinition(sourceList, sourceListValue.c_str());
  return true;
}
