// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

// Description
// This test verifies volume rendering of VTK Unstructured Grid with ANARI

#include "vtkColorTransferFunction.h"
#include "vtkDataSetTriangleFilter.h"
#include "vtkImageCast.h"
#include "vtkInteractorStyleTrackballCamera.h"
#include "vtkNew.h"
#include "vtkPiecewiseFunction.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkRenderer.h"
#include "vtkStructuredPointsReader.h"
#include "vtkTesting.h"
#include "vtkThreshold.h"
#include "vtkUnstructuredGridVolumeRayCastMapper.h"
#include "vtkVolume.h"
#include "vtkVolumeProperty.h"
#include "vtkTestUtilities.h"

#include "vtkAnariPass.h"
#include "vtkAnariRendererNode.h"
#include "vtkAnariTestInteractor.h"

namespace
{

static const char* TestAnariUnstructuredVolumeMapperLog = "# StreamVersion 1\n"
                                                           "EnterEvent 299 0 0 0 0 0 0\n"
                                                           "MouseMoveEvent 299 0 0 0 0 0 0\n"
                                                           "MouseMoveEvent 298 2 0 0 0 0 0\n"
                                                           "MouseMoveEvent 297 4 0 0 0 0 0\n"
                                                           "MouseMoveEvent 297 6 0 0 0 0 0\n"
                                                           "MouseMoveEvent 296 8 0 0 0 0 0\n"
                                                           "LeaveEvent 399 -8 0 0 0 0 0\n";

} // end anon namespace

int TestAnariUnstructuredVolumeMapper(int argc, char* argv[])
{
  bool useDebugDevice = false;

  for (int i = 0; i < argc; i++)
  {
    if (!strcmp(argv[i], "-trace"))
    {
      useDebugDevice = true;
    }
  }

  // Create the reader for the data
  // This is the data the will be volume rendered
  vtkNew<vtkStructuredPointsReader> reader;
  const char* file1 = vtkTestUtilities::ExpandDataFileName(argc, argv, "Data/ironProt.vtk");
  reader->SetFileName(file1);
  reader->Update();

  // currently ANARI only supports float
  vtkNew<vtkImageCast> toFloat;
  toFloat->SetInputConnection(reader->GetOutputPort());
  toFloat->SetOutputScalarTypeToFloat();

  // convert from vtkImageData to vtkUnstructuredGrid, remove
  // any cells where all values are below 80
  vtkNew<vtkThreshold> thresh;
  thresh->SetThresholdFunction(vtkThreshold::THRESHOLD_UPPER);
  thresh->SetUpperThreshold(80.0);
  thresh->AllScalarsOff();
  thresh->SetInputConnection(toFloat->GetOutputPort());

  // make sure we have only tetrahedra
  vtkNew<vtkDataSetTriangleFilter> trifilter;
  trifilter->SetInputConnection(thresh->GetOutputPort());

  // Create transfer mapping scalar value to opacity
  vtkNew<vtkPiecewiseFunction> opacityTransferFunction;
  opacityTransferFunction->AddPoint(80, 0.0);
  opacityTransferFunction->AddPoint(120, 0.2);
  opacityTransferFunction->AddPoint(255, 0.2);

  // Create transfer mapping scalar value to color
  vtkNew<vtkColorTransferFunction> colorTransferFunction;
  colorTransferFunction->AddRGBPoint(80.0, 0.0, 0.0, 0.0);
  colorTransferFunction->AddRGBPoint(120.0, 0.0, 0.0, 1.0);
  colorTransferFunction->AddRGBPoint(160.0, 1.0, 0.0, 0.0);
  colorTransferFunction->AddRGBPoint(200.0, 0.0, 1.0, 0.0);
  colorTransferFunction->AddRGBPoint(255.0, 0.0, 1.0, 1.0);

  // The property describes how the data will look
  vtkNew<vtkVolumeProperty> volumeProperty;
  volumeProperty->SetColor(colorTransferFunction);
  volumeProperty->SetScalarOpacity(opacityTransferFunction);
  volumeProperty->ShadeOff();
  volumeProperty->SetInterpolationTypeToLinear();

  // The mapper / ray cast function know how to render the data
  vtkNew<vtkUnstructuredGridVolumeRayCastMapper> volumeMapper;
  volumeMapper->SetInputConnection(trifilter->GetOutputPort());

  // The volume holds the mapper and the property and
  // can be used to position/orient the volume
  vtkNew<vtkVolume> volume;
  volume->SetMapper(volumeMapper);
  volume->SetProperty(volumeProperty);

  vtkNew<vtkRenderer> ren1;
  ren1->AddVolume(volume);

  // Create the renderwindow, interactor and renderer
  vtkNew<vtkRenderWindow> renderWindow;
  renderWindow->SetMultiSamples(0);
  renderWindow->SetSize(401, 399); // NPOT size
  vtkNew<vtkRenderWindowInteractor> iren;
  iren->SetRenderWindow(renderWindow);
  vtkNew<vtkInteractorStyleTrackballCamera> style;
  iren->SetInteractorStyle(style);
  ren1->SetBackground(0.3, 0.3, 0.4);
  renderWindow->AddRenderer(ren1);

  ren1->ResetCamera();

  // Attach ANARI render pass
  vtkNew<vtkAnariPass> anariPass;
  ren1->SetPass(anariPass);

  if (useDebugDevice)
  {
    vtkAnariRendererNode::SetUseDebugDevice(1, ren1);
    vtkNew<vtkTesting> testing;

    std::string traceDir = testing->GetTempDirectory();
    traceDir += "/anari-trace";
    traceDir += "/TestAnariUnstructuredVolumeMapper";
    vtkAnariRendererNode::SetDebugDeviceDirectory(traceDir.c_str(), ren1);
  }

  vtkAnariRendererNode::SetLibraryName("environment", ren1);
  vtkAnariRendererNode::SetSamplesPerPixel(6, ren1);
  vtkAnariRendererNode::SetLightFalloff(.5, ren1);
  vtkAnariRendererNode::SetUseDenoiser(1, ren1);
  vtkAnariRendererNode::SetCompositeOnGL(1, ren1);
  vtkAnariRendererNode::SetAccumulationCount(30, ren1);

  renderWindow->Render();
  volumeMapper->DebugOn();

  auto anariRendererNode = anariPass->GetSceneGraph();
  auto extensions = anariRendererNode->GetAnariDeviceExtensions();

  if (true) // TODO: extensions.ANARI_KHR_SPATIAL_FIELD_UNSTRUCTURED
  {
    iren->Initialize();
    return vtkTesting::InteractorEventLoop(argc, argv, iren, TestAnariUnstructuredVolumeMapperLog);
  }

  std::cout << "Required feature KHR_SPATIAL_FIELD_UNSTRUCTURED not supported." << std::endl;
  return VTK_SKIP_RETURN_CODE;
}
