// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
// This test checks if ANARI based AMR Volume rendering works

#include "vtkAMRGaussianPulseSource.h"
#include "vtkAMRVolumeMapper.h"
#include "vtkCamera.h"
#include "vtkColorTransferFunction.h"
#include "vtkNew.h"
#include "vtkOverlappingAMR.h"
#include "vtkPiecewiseFunction.h"
#include "vtkRegressionTestImage.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkRenderer.h"
#include "vtkTestUtilities.h"
#include "vtkVolumeProperty.h"

#include "vtkAnariPass.h"
#include "vtkAnariRendererNode.h"
#include "vtkAnariTestInteractor.h"

int TestAnariAMRVolumeMapper(int argc, char* argv[])
{
  bool useDebugDevice = false;

  for (int i = 0; i < argc; i++)
  {
    if (!strcmp(argv[i], "-trace"))
    {
      useDebugDevice = true;
    }
  }

  double scalarRange[2] = { 4.849e-23, 0.4145 };
  double dRange = scalarRange[1] - scalarRange[0];
  vtkNew<vtkAMRVolumeMapper> volumeMapper;

  vtkNew<vtkAMRGaussianPulseSource> amrSource;
  amrSource->SetXPulseOrigin(0);
  amrSource->SetYPulseOrigin(0);
  amrSource->SetZPulseOrigin(0);
  amrSource->SetXPulseWidth(.5);
  amrSource->SetYPulseWidth(.5);
  amrSource->SetZPulseWidth(.5);
  amrSource->SetPulseAmplitude(0.5);
  amrSource->SetDimension(3);
  amrSource->SetRootSpacing(0.5);
  amrSource->SetRefinementRatio(2);
  amrSource->Update();
  volumeMapper->SetInputConnection(amrSource->GetOutputPort());

  volumeMapper->SelectScalarArray("Gaussian-Pulse");
  volumeMapper->SetScalarMode(VTK_SCALAR_MODE_USE_POINT_FIELD_DATA);

  vtkNew<vtkRenderWindow> renWin;
  renWin->SetMultiSamples(0);
  vtkNew<vtkRenderer> ren;
  renWin->AddRenderer(ren);
  ren->SetBackground(0.2, 0.2, 0.5);
  renWin->SetSize(400, 400);

  vtkNew<vtkRenderWindowInteractor> iren;
  iren->SetRenderWindow(renWin);

  vtkNew<vtkPiecewiseFunction> scalarOpacity;
  scalarOpacity->AddPoint(scalarRange[0], 0.5);
  scalarOpacity->AddPoint(scalarRange[0] + dRange / 4.0, 0.0);
  scalarOpacity->AddPoint(scalarRange[1] - dRange / 2.0, 0.0);
  scalarOpacity->AddPoint(scalarRange[1], 1.0);

  vtkNew<vtkVolumeProperty> volumeProperty;
  volumeProperty->ShadeOff();
  volumeProperty->SetInterpolationType(VTK_LINEAR_INTERPOLATION);
  volumeProperty->SetScalarOpacity(scalarOpacity);

  vtkColorTransferFunction* colorTransferFunction = volumeProperty->GetRGBTransferFunction(0);
  colorTransferFunction->RemoveAllPoints();
  colorTransferFunction->AddRGBPoint(scalarRange[0], 0.0, 0.0, 1.0);
  colorTransferFunction->AddRGBPoint(scalarRange[0] + dRange / 2.0, 1.0, 1.0, 1.0);
  colorTransferFunction->AddRGBPoint(scalarRange[1], 1.0, 0.0, 0.0);

  vtkNew<vtkVolume> volume;
  volume->SetMapper(volumeMapper);
  volume->SetProperty(volumeProperty);

  // Attach ANARI render pass
  vtkNew<vtkAnariPass> anariPass;
  ren->SetPass(anariPass);

  if (useDebugDevice)
  {
    vtkAnariRendererNode::SetUseDebugDevice(1, ren);
    vtkNew<vtkTesting> testing;

    std::string traceDir = testing->GetTempDirectory();
    traceDir += "/anari-trace";
    traceDir += "/TestAnariAMRVolumeMapper";
    vtkAnariRendererNode::SetDebugDeviceDirectory(traceDir.c_str(), ren);
  }

  vtkAnariRendererNode::SetLibraryName("environment", ren);
  vtkAnariRendererNode::SetSamplesPerPixel(6, ren);
  vtkAnariRendererNode::SetLightFalloff(.5, ren);
  vtkAnariRendererNode::SetUseDenoiser(1, ren);
  vtkAnariRendererNode::SetCompositeOnGL(1, ren);
  vtkAnariRendererNode::SetAccumulationCount(30, ren);

  ren->AddViewProp(volume);
  renWin->Render();
  ren->ResetCamera();

  auto anariRendererNode = anariPass->GetSceneGraph();
  auto extensions = anariRendererNode->GetAnariDeviceExtensions();

  if (true) // TODO: extensions.ANARI_KHR_SPATIAL_FIELD_AMR
  {
    iren->Initialize();
    iren->SetDesiredUpdateRate(30.0);
    int retVal = vtkRegressionTestImage(renWin);

    if (retVal == vtkRegressionTester::DO_INTERACTOR)
    {
      iren->Start();
    }

    volumeMapper->SetInputConnection(nullptr);
    return !retVal;
  }

  std::cout << "Required feature KHR_SPATIAL_FIELD_AMR not supported." << std::endl;
  return VTK_SKIP_RETURN_CODE;
}
