/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmComputeTargetOrder.h"

#include "cmComputeComponentGraph.h"
#include "cmGeneratorTarget.h"
#include "cmGlobalGenerator.h"
#include "cmLocalGenerator.h"
#include "cmMakefile.h"
#include "cmRange.h"
#include "cmSystemTools.h"
#include "cmTargetDepend.h"

#include <sstream>
#include <stdio.h>

/*

This class is meant to analyze inter-target dependencies of targets
during the generation step.  The goal is to produce a list of the correct
build order to satisfy the target dependencies.

Cyclic dependencies may not be allowed.

  - Collect all targets and form the original dependency graph
  - Run Tarjan's algorithm to extract the strongly connected components
  - The original dependencies imply a DAG on the components.
    Use the implied DAG to construct a build order list of targets.

*/

cmComputeTargetOrder::cmComputeTargetOrder(
  std::vector<cmGeneratorTarget const*>* targets, bool no_cycles,
  bool missing_deps_error)
{
  this->DebugMode = false;
  this->NoCycles = no_cycles;
  this->MissingDepsError = missing_deps_error;
  this->TargetList = targets;
  this->StatusOK = true;
}

cmComputeTargetOrder::~cmComputeTargetOrder() = default;

bool cmComputeTargetOrder::Compute()
{
  // Build the original graph.
  this->CollectTargets();
  this->CollectDepends();
  if (this->DebugMode) {
    this->DisplayGraph(this->InitialGraph, "initial");
  }

  // Identify components.
  cmComputeComponentGraph ccg(this->InitialGraph);
  if (this->DebugMode) {
    this->DisplayComponents(ccg);
  }
  this->CheckComponents(ccg);

  return this->StatusOK;
}

void cmComputeTargetOrder::CollectTargets()
{
  // Collect all targets
  for (cmGeneratorTarget const* ti : *this->TargetList) {
    int index = static_cast<int>(this->Targets.size());
    this->TargetIndex[ti] = index;
    this->Targets.push_back(ti);
  }
}

void cmComputeTargetOrder::CollectDepends()
{
  // Allocate the dependency graph adjacency lists.
  this->InitialGraph.resize(this->Targets.size());

  // Compute each dependency list.
  for (unsigned int i = 0; i < this->Targets.size(); ++i) {
    this->CollectTargetDepends(i);
  }
}

void cmComputeTargetOrder::CollectTargetDepends(int depender_index)
{
  // Get the depender.
  cmGeneratorTarget const* depender = this->Targets[depender_index];

  cmTargetDependSet deps =
    depender->GetGlobalGenerator()->GetTargetDirectDepends(depender);

  for (auto& dep : deps) {
    this->AddTargetDepend(depender_index, dep);
  }
}

void cmComputeTargetOrder::AddTargetDepend(int depender_index,
                                           cmGeneratorTarget const* dependee)
{
  // Lookup the index for this target.  All targets should have
  // been provided by user.
  std::map<cmGeneratorTarget const*, int>::const_iterator tii =
    this->TargetIndex.find(dependee);

  if (tii != this->TargetIndex.end()) {
    int dependee_index = tii->second;

    // Add this entry to the dependency graph.
    this->InitialGraph[depender_index].emplace_back(dependee_index, true,
                                                    dependee->GetBacktrace());
  } else if (this->MissingDepsError) {
    cmGeneratorTarget const* depender = this->Targets[depender_index];
    std::ostringstream w;
    w << "The inter-target dependency graph of targets "
         "for target ["
      << depender->GetName() << "] depends on target [" << dependee->GetName()
      << "] which is not part of target set.\n";
    depender->GetLocalGenerator()->IssueMessage(MessageType::AUTHOR_WARNING,
                                                w.str());
    this->StatusOK = false;
  }
}

void cmComputeTargetOrder::DisplayGraph(Graph const& graph,
                                        const std::string& name)
{
  fprintf(stderr, "The %s target dependency graph is:\n", name.c_str());
  int n = static_cast<int>(graph.size());
  for (int depender_index = 0; depender_index < n; ++depender_index) {
    EdgeList const& nl = graph[depender_index];
    cmGeneratorTarget const* depender = this->Targets[depender_index];
    fprintf(stderr, "target %d is [%s]\n", depender_index,
            depender->GetName().c_str());
    for (cmGraphEdge const& ni : nl) {
      int dependee_index = ni;
      cmGeneratorTarget const* dependee = this->Targets[dependee_index];
      fprintf(stderr, "  depends on target %d [%s] (%s)\n", dependee_index,
              dependee->GetName().c_str(), ni.IsStrong() ? "strong" : "weak");
    }
  }
  fprintf(stderr, "\n");
}

void cmComputeTargetOrder::DisplayComponents(
  cmComputeComponentGraph const& ccg)
{
  fprintf(stderr, "The strongly connected components are:\n");
  std::vector<NodeList> const& components = ccg.GetComponents();
  int n = static_cast<int>(components.size());
  for (int c = 0; c < n; ++c) {
    NodeList const& nl = components[c];
    fprintf(stderr, "Component (%d):\n", c);
    for (int i : nl) {
      fprintf(stderr, "  contains target %d [%s]\n", i,
              this->Targets[i]->GetName().c_str());
    }
  }
  fprintf(stderr, "\n");
}

void cmComputeTargetOrder::CheckComponents(cmComputeComponentGraph const& ccg)
{
  // Check for cycles.
  std::vector<NodeList> const& components = ccg.GetComponents();
  int nc = static_cast<int>(components.size());
  for (int c = 0; c < nc; ++c) {
    // Get the current component.
    NodeList const& nl = components[c];

    // Complain if no cycles are allowed.
    if (nl.size() > 1) {
      this->ComplainAboutBadComponent(ccg, c);
      if (this->NoCycles) {
        this->StatusOK = false;
      }
    }

    // Add targets to build list
    for (int i : nl) {
      this->TargetsBuildOrder.push_back(this->Targets[i]);
    }
  }
}

void cmComputeTargetOrder::ComplainAboutBadComponent(
  cmComputeComponentGraph const& ccg, int c)
{
  // Construct the error message.
  std::ostringstream e;
  e << "The inter-target dependency graph of targets "
       "contains the following strongly connected component (cycle):\n";
  std::vector<NodeList> const& components = ccg.GetComponents();
  std::vector<int> const& cmap = ccg.GetComponentMap();
  NodeList const& cl = components[c];
  for (int i : cl) {
    // Get the depender.
    cmGeneratorTarget const* depender = this->Targets[i];

    // Describe the depender.
    e << "  \"" << depender->GetName() << "\n";

    // List its dependencies that are inside the component.
    EdgeList const& nl = this->InitialGraph[i];
    for (cmGraphEdge const& ni : nl) {
      int j = ni;
      if (cmap[j] == c) {
        cmGeneratorTarget const* dependee = this->Targets[j];
        e << "    depends on \"" << dependee->GetName() << "\"\n";
      }
    }
  }
  this->Targets[0]->GetLocalGenerator()->IssueMessage(
    MessageType::AUTHOR_WARNING, e.str());
}
