/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmComputeCustomCommandOrder.h"

#include "cmComputeComponentGraph.h"
#include "cmCustomCommand.h"
#include "cmGeneratorTarget.h"
#include "cmLocalGenerator.h"
#include "cmMakefile.h"
#include "cmRange.h"
#include "cmSourceFile.h"
#include "cmSystemTools.h"

#include <sstream>
#include <stdio.h>

/*

This class is meant to analyze inter-source dependencies of custom commands
during the generation step.  The goal is to produce a list of the correct
build order to satisfy the file dependencies.

Cyclic dependencies may not be allowed.

  - Collect all sources and form the original dependency graph
  - Run Tarjan's algorithm to extract the strongly connected components
  - The original dependencies imply a DAG on the components.
    Use the implied DAG to construct a build order list of custom commands.

*/

cmComputeCustomCommandOrder::cmComputeCustomCommandOrder(
  cmGeneratorTarget* gt, std::string config, bool no_cycles,
  bool missing_deps_error)
{
  this->DebugMode = false;
  this->NoCycles = no_cycles;
  this->MissingDepsError = missing_deps_error;
  this->GeneratorTarget = gt;
  this->Config = config;
  this->StatusOK = true;
}

cmComputeCustomCommandOrder::~cmComputeCustomCommandOrder() = default;

bool cmComputeCustomCommandOrder::Compute()
{
  // Build the original graph.
  this->CollectSources();
  this->CollectDepends();
  if (this->DebugMode) {
    this->DisplayGraph(this->InitialGraph, "initial");
  }

  // Identify components.
  cmComputeComponentGraph ccg(this->InitialGraph);
  if (this->DebugMode) {
    this->DisplayComponents(ccg);
  }
  this->CheckComponents(ccg);

  return this->StatusOK;
}

void cmComputeCustomCommandOrder::CollectSources()
{
  // Collect all sources for this target
  std::vector<cmSourceFile const*> customCommands;
  this->GeneratorTarget->GetCustomCommands(customCommands, this->Config);

  for (cmSourceFile const* si : customCommands) {
    int index = static_cast<int>(this->Sources.size());
    this->SourceIndex[si] = index;
    this->Sources.push_back(si);
  }
}

void cmComputeCustomCommandOrder::CollectDepends()
{
  // Allocate the dependency graph adjacency lists.
  this->InitialGraph.resize(this->Sources.size());

  // Compute each dependency list.
  for (unsigned int i = 0; i < this->Sources.size(); ++i) {
    this->CollectSourceDepends(i);
  }
}

void cmComputeCustomCommandOrder::CollectSourceDepends(int depender_index)
{
  // Get the depender.
  cmSourceFile const* depender = this->Sources[depender_index];

  // Loop over all dependencies.
  const cmCustomCommand* cc = depender->GetCustomCommand();

  for (auto& dep : cc->GetDepends()) {
    this->AddSourceDepend(depender_index, dep);
  }
}

void cmComputeCustomCommandOrder::AddSourceDepend(
  int depender_index, std::string const& dependee_name)
{
  // Get the dependee.
  cmSourceFile const* dependee = this->GeneratorTarget->GetLocalGenerator()
                                   ->GetMakefile()
                                   ->GetSourceFileWithOutput(dependee_name);

  if (dependee) {
    // Lookup the index for this source.  All sources should be known by
    // this point.
    std::map<cmSourceFile const*, int>::const_iterator tii =
      this->SourceIndex.find(dependee);
    assert(tii != this->SourceIndex.end());
    int dependee_index = tii->second;

    // Add this entry to the dependency graph.
    this->InitialGraph[depender_index].emplace_back(
      dependee_index, true, dependee->GetCustomCommand()->GetBacktrace());
  } else {
    dependee =
      this->GeneratorTarget->GetLocalGenerator()->GetMakefile()->GetSource(
        dependee_name);
    if (!dependee && this->MissingDepsError) {
      cmSourceFile const* depender = this->Sources[depender_index];
      std::ostringstream w;
      w << "The inter-source dependency graph for custom commands and targets "
           "for target ["
        << this->GeneratorTarget->GetName() << "] and rule ["
        << depender->GetFullPath() << "] depends on unknown source ["
        << dependee_name << "]\n";
      this->GeneratorTarget->GetLocalGenerator()->IssueMessage(
        MessageType::AUTHOR_WARNING, w.str());
      this->StatusOK = false;
    }
  }
}

void cmComputeCustomCommandOrder::DisplayGraph(Graph const& graph,
                                               const std::string& name)
{
  fprintf(stderr, "The %s source dependency graph is:\n", name.c_str());
  int n = static_cast<int>(graph.size());
  for (int depender_index = 0; depender_index < n; ++depender_index) {
    EdgeList const& nl = graph[depender_index];
    cmSourceFile const* depender = this->Sources[depender_index];
    fprintf(stderr, "source %d is [%s]\n", depender_index,
            depender->GetFullPath().c_str());
    for (cmGraphEdge const& ni : nl) {
      int dependee_index = ni;
      cmSourceFile const* dependee = this->Sources[dependee_index];
      fprintf(stderr, "  depends on source %d [%s] (%s)\n", dependee_index,
              dependee->GetFullPath().c_str(),
              ni.IsStrong() ? "strong" : "weak");
    }
  }
  fprintf(stderr, "\n");
}

void cmComputeCustomCommandOrder::DisplayComponents(
  cmComputeComponentGraph const& ccg)
{
  fprintf(stderr, "The strongly connected components are:\n");
  std::vector<NodeList> const& components = ccg.GetComponents();
  int n = static_cast<int>(components.size());
  for (int c = 0; c < n; ++c) {
    NodeList const& nl = components[c];
    fprintf(stderr, "Component (%d):\n", c);
    for (int i : nl) {
      fprintf(stderr, "  contains source %d [%s]\n", i,
              this->Sources[i]->GetFullPath().c_str());
    }
  }
  fprintf(stderr, "\n");
}

void cmComputeCustomCommandOrder::CheckComponents(
  cmComputeComponentGraph const& ccg)
{
  // Check for cycles.
  std::vector<NodeList> const& components = ccg.GetComponents();
  int nc = static_cast<int>(components.size());
  for (int c = 0; c < nc; ++c) {
    // Get the current component.
    NodeList const& nl = components[c];

    // Complain if no cycles are allowed.
    if (nl.size() > 1) {
      this->ComplainAboutBadComponent(ccg, c);
      if (this->NoCycles) {
        this->StatusOK = false;
      }
    }

    // Add sources to build list
    for (int i : nl) {
      this->SourcesBuildOrder.push_back(this->Sources[i]);
    }
  }
}

void cmComputeCustomCommandOrder::ComplainAboutBadComponent(
  cmComputeComponentGraph const& ccg, int c)
{
  // Construct the error message.
  std::ostringstream e;
  e << "The inter-source dependency graph for custom commands and targets "
       "contains the following strongly connected component (cycle) for "
       "target ["
    << this->GeneratorTarget->GetName() << "]:\n";
  std::vector<NodeList> const& components = ccg.GetComponents();
  std::vector<int> const& cmap = ccg.GetComponentMap();
  NodeList const& cl = components[c];
  for (int i : cl) {
    // Get the depender.
    cmSourceFile const* depender = this->Sources[i];

    // Describe the depender.
    e << "  \"" << depender->GetFullPath() << "\n";

    // List its dependencies that are inside the component.
    EdgeList const& nl = this->InitialGraph[i];
    for (cmGraphEdge const& ni : nl) {
      int j = ni;
      if (cmap[j] == c) {
        cmSourceFile const* dependee = this->Sources[j];
        e << "    depends on \"" << dependee->GetFullPath() << "\"\n";
      }
    }
  }
  this->GeneratorTarget->GetLocalGenerator()->IssueMessage(
    MessageType::AUTHOR_WARNING, e.str());
}
