# Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
# file Copyright.txt or https://cmake.org/licensing for details.

#[=======================================================================[.rst:
FindImageMagick
---------------

Find ImageMagick binary suite.

.. versionadded:: 3.9
  Added support for ImageMagick 7.

This module will search for a set of ImageMagick tools specified as
components in the :command:`find_package` call. The following is a
non-exhaustive list of possible components, as future versions of
ImageMagick might have additional components not listed here.

The ImageMagick 7 executable:

::

  magick


The ImageMagick 6 individual operator executables:

::

  animate
  compare
  composite
  conjure
  convert
  display
  identify
  import
  mogrify
  montage
  stream

The following ImageMagick APIs:

::

  Magick++
  MagickWand
  MagickCore


For each API component found, an IMPORTED target Magick::<component> is created.

If no component is specified in the :command:`find_package` call, then it only
searches for the ImageMagick executable directory.

Note that the standard :command:`find_package` features are supported (i.e.,
``QUIET``, ``REQUIRED``, etc.).


Legacy support
^^^^^^^^^^^^^^

For backwards compatibility, this code defines a set of legacy result
variables, though the IMPORTED targets are preferred.

The following variables are always defined:

::

  ImageMagick_FOUND                  - TRUE if all components are found.
  ImageMagick_EXECUTABLE_DIR         - Full path to executables directory.
  ImageMagick_VERSION_STRING         - The version of ImageMagick found.


If any API components are requested, additional variables are defined:

::

  ImageMagick_INCLUDE_DIRS             - Full paths to all include dirs.
  ImageMagick_LIBRARIES                - Full paths to all libraries.


For each executable component, the following variables are defined:

::

  ImageMagick_<component>_FOUND      - TRUE if <component> is found.
  ImageMagick_<component>_EXECUTABLE - Full path to <component> executable.


For each API component, the following variables are defined:

::

  ImageMagick_<component>_FOUND        - TRUE if <component> is found.
  ImageMagick_<component>_INCLUDE_DIRS - Full path to <component> include dirs.
  ImageMagick_<component>_LIBRARIES    - Full path to <component> libraries.


Example Usages:

::

  find_package(ImageMagick)
  find_package(ImageMagick COMPONENTS convert)
  find_package(ImageMagick COMPONENTS convert mogrify display)
  find_package(ImageMagick COMPONENTS Magick++)
  find_package(ImageMagick COMPONENTS Magick++ magick)


#]=======================================================================]

find_package(PkgConfig QUIET)

#---------------------------------------------------------------------
# Helper functions
#---------------------------------------------------------------------
function(FIND_IMAGEMAGICK_API component header)
  if(subcomponent STREQUAL "" OR subcomponent STREQUAL ${component})
    set(c ${component})
  else()
    # Subcomponent lookup
    set(c "${component}${subcomponent}")
  endif()
  set(ImageMagick_${component}_FOUND FALSE PARENT_SCOPE)

  pkg_check_modules(PC_${component} QUIET IMPORTED_TARGET GLOBAL ${component})

  find_path(ImageMagick_${component}_INCLUDE_DIR
    NAMES ${header}
    HINTS
      ${PC_${component}_INCLUDEDIR}
      ${PC_${component}_INCLUDE_DIRS}
    PATHS
      ${ImageMagick_INCLUDE_DIRS}
      "[HKEY_LOCAL_MACHINE\\SOFTWARE\\ImageMagick\\Current;BinPath]/include"
    PATH_SUFFIXES
      ImageMagick ImageMagick-6 ImageMagick-7
    DOC "Path to the ImageMagick arch-independent include dir."
    NO_DEFAULT_PATH
    )
  find_path(ImageMagick_${component}_ARCH_INCLUDE_DIR
    NAMES magick/magick-baseconfig.h
    HINTS
      ${PC_${component}_INCLUDEDIR}
      ${PC_${component}_INCLUDE_DIRS}
    PATHS
      ${ImageMagick_INCLUDE_DIRS}
      "[HKEY_LOCAL_MACHINE\\SOFTWARE\\ImageMagick\\Current;BinPath]/include"
    PATH_SUFFIXES
      ImageMagick ImageMagick-6 ImageMagick-7
    DOC "Path to the ImageMagick arch-specific include dir."
    NO_DEFAULT_PATH
    )
  find_library(ImageMagick_${component}_LIBRARY
    NAMES ${ARGN}
    HINTS
      ${PC_${component}_LIBDIR}
      ${PC_${component}_LIB_DIRS}
    PATHS
      "[HKEY_LOCAL_MACHINE\\SOFTWARE\\ImageMagick\\Current;BinPath]/lib"
    DOC "Path to the ImageMagick ${component} library."
    NO_DEFAULT_PATH
    )

  # old version have only indep dir
  if(ImageMagick_${component}_INCLUDE_DIR AND ImageMagick_${component}_LIBRARY)
    set(ImageMagick_${component}_FOUND TRUE PARENT_SCOPE)

    # Keep track of found components, for target creation later
    list(APPEND ImageMagick_API_COMPONENTS ${component})
    set(ImageMagick_API_COMPONENTS ${ImageMagick_API_COMPONENTS} PARENT_SCOPE)

    # Construct per-component include directories.
    set(ImageMagick_${component}_INCLUDE_DIRS
      ${ImageMagick_${component}_INCLUDE_DIR}
      )
    if(ImageMagick_${component}_ARCH_INCLUDE_DIR)
      list(APPEND ImageMagick_${component}_INCLUDE_DIRS
        ${ImageMagick_${component}_ARCH_INCLUDE_DIR})
    endif()
    list(REMOVE_DUPLICATES ImageMagick_${component}_INCLUDE_DIRS)
    set(ImageMagick_${component}_INCLUDE_DIRS
      ${ImageMagick_${component}_INCLUDE_DIRS} PARENT_SCOPE)

    # Add the per-component include directories to the full include dirs.
    list(APPEND ImageMagick_INCLUDE_DIRS ${ImageMagick_${component}_INCLUDE_DIRS})
    list(REMOVE_DUPLICATES ImageMagick_INCLUDE_DIRS)
    set(ImageMagick_INCLUDE_DIRS ${ImageMagick_INCLUDE_DIRS} PARENT_SCOPE)

    list(APPEND ImageMagick_LIBRARIES
      ${ImageMagick_${component}_LIBRARY}
      )
    set(ImageMagick_LIBRARIES ${ImageMagick_LIBRARIES} PARENT_SCOPE)
  endif()

  # Create a target for this component
  if(NOT TARGET ImageMagick::${component})
    add_library(ImageMagick::${component} UNKNOWN IMPORTED GLOBAL)
    if(TARGET PkgConfig::${component})
      target_link_libraries(ImageMagick::${component} PkgConfig::${component})
    else()
      set_property(TARGET ImageMagick::${component} APPEND PROPERTY
        INTERFACE_INCLUDE_DIRECTORIES "${ImageMagick_${component}_INCLUDE_DIRS}")

      set_property(TARGET ImageMagick::${component} APPEND PROPERTY
        INTERFACE_COMPILE_DEFINITIONS "${ImageMagick_${component}_DEFINITIONS}")

      set_property(TARGET ImageMagick::${component} APPEND PROPERTY
        IMPORTED_LOCATION "${ImageMagick_${component}_LIBRARIES}")
    endif()
  endif()

endfunction()

function(FIND_IMAGEMAGICK_EXE component)
  set(_IMAGEMAGICK_EXECUTABLE
    ${ImageMagick_EXECUTABLE_DIR}/${component}${CMAKE_EXECUTABLE_SUFFIX})
  if(EXISTS ${_IMAGEMAGICK_EXECUTABLE})
    set(ImageMagick_${component}_EXECUTABLE
      ${_IMAGEMAGICK_EXECUTABLE}
       PARENT_SCOPE
       )
    set(ImageMagick_${component}_FOUND TRUE PARENT_SCOPE)
  else()
    set(ImageMagick_${component}_FOUND FALSE PARENT_SCOPE)
  endif()
endfunction()

# ---------------------------------------------------------------------
# Create a matrix of lookups for each library component & configuration
# ---------------------------------------------------------------------
set(Magick++_HEADERNAMES "Magic++.h")
set(MagickWand_HEADERNAMES "wand/MagickWand.h;MagickWand/MagickWand.h")
set(MagickCore_HEADERNAMES "magick/MagickCore.h;MagickCore/MagickCore.h")

set(Magick++_LIBNAMES_BASE
  Magick++
  CORE_RL_Magick++_
)

set(MagickWand_LIBNAMES_BASE
  Wand MagickWand
  CORE_RL_wand_ CORE_RL_MagickWand_
)

set(MagickCore_LIBNAMES_BASE
  Magick MagickCore
  CORE_RL_magick_ CORE_RL_MagickCore_
)

foreach(n Magick++ MagickWand MagickCore)   # name
  list(APPEND ${n}_LIBNAMES_BASE ${n}-Q8 ${n}-Q16 ${n}-Q16HDRI ${n}-Q8HDRI)
  foreach (v 6 7)             # version
    list(APPEND ${n}_LIBNAMES_${v} ${n}-${v})
    foreach(d 8 16 32 64)     # depth (QUANTUM_DEPTH)
      foreach(h "" HDRI)      # HDRI build
        list(APPEND ${n}_LIBNAMES_${v} ${n}-${v}.Q${d}${h})
      endforeach()
    endforeach()
  endforeach()
  list(APPEND ${n}_LIBNAMES_ALL ${${n}_LIBNAMES_BASE} ${${n}_LIBNAMES_6} ${${n}_LIBNAMES_7})
endforeach()

# XXX: Debugging
include(CMakePrintHelpers)
foreach(n Magick++ MagickWand MagickCore)
  message(STATUS "\nVariables for ${n}:")
  foreach(v ${n}_LIBNAMES_BASE ${n}_LIBNAMES_6 ${n}_LIBNAMES_7 ${n}_LIBNAMES_ALL)
    cmake_print_variables(${v})
  endforeach()
endforeach()

#---------------------------------------------------------------------
# Start Actual Work
#---------------------------------------------------------------------
# Try to find a ImageMagick installation binary path.
find_path(ImageMagick_EXECUTABLE_DIR
  NAMES magick${CMAKE_EXECUTABLE_SUFFIX} mogrify${CMAKE_EXECUTABLE_SUFFIX}
  PATHS
    "[HKEY_LOCAL_MACHINE\\SOFTWARE\\ImageMagick\\Current;BinPath]"
  DOC "Path to the ImageMagick binary directory."
  NO_DEFAULT_PATH
  )
find_path(ImageMagick_EXECUTABLE_DIR
  NAMES magick${CMAKE_EXECUTABLE_SUFFIX} mogrify${CMAKE_EXECUTABLE_SUFFIX}
  )

# Find each component. Search for all tools in same dir
# <ImageMagick_EXECUTABLE_DIR>; otherwise they should be found
# independently and not in a cohesive module such as this one.
unset(ImageMagick_REQUIRED_VARS)
unset(ImageMagick_DEFAULT_EXECUTABLES)
unset(ImageMagick_API_COMPONENTS)
foreach(component ${ImageMagick_FIND_COMPONENTS}
  # DEPRECATED: forced components for backward compatibility
  magick convert mogrify import montage composite
  )

  if(component MATCHES "^(Magick[+][+]|MagickWand|MagickCore)(-?.*)$")
    if(CMAKE_MATCH_2 STREQUAL "")
      set(_lib ${CMAKE_MATCH_1})
      # Unqualified component name (e.g. 'MagickCore')
      FIND_IMAGEMAGICK_API(${_lib}
        ${${_lib}_HEADERNAMES}
        ${${_lib}_LIBNAMES_ALL}
      )
    else()
      set(_lib ${component})
      # Specific versioned and/or configured component
      FIND_IMAGEMAGICK_API(${_lib}
        ${${_lib}_HEADERNAMES}
        ${component})
    endif()
    list(APPEND ImageMagick_REQUIRED_VARS ImageMagick_${component}_LIBRARY)
  else()
    if(ImageMagick_EXECUTABLE_DIR)
      FIND_IMAGEMAGICK_EXE(${component})
    endif()

    if(ImageMagick_FIND_COMPONENTS)
      list(FIND ImageMagick_FIND_COMPONENTS ${component} is_requested)
      if(is_requested GREATER -1)
        list(APPEND ImageMagick_REQUIRED_VARS ImageMagick_${component}_EXECUTABLE)
      endif()
    elseif(ImageMagick_${component}_EXECUTABLE)
      # if no components were requested explicitly put all (default) executables
      # in the list
      list(APPEND ImageMagick_DEFAULT_EXECUTABLES ImageMagick_${component}_EXECUTABLE)
    endif()
  endif()
endforeach()

if(NOT ImageMagick_FIND_COMPONENTS AND NOT ImageMagick_DEFAULT_EXECUTABLES)
  # No components were requested, and none of the default components were
  # found. Just insert mogrify into the list of the default components to
  # find so FPHSA below has something to check
  list(APPEND ImageMagick_REQUIRED_VARS ImageMagick_mogrify_EXECUTABLE)
elseif(ImageMagick_DEFAULT_EXECUTABLES)
  list(APPEND ImageMagick_REQUIRED_VARS ${ImageMagick_DEFAULT_EXECUTABLES})
endif()

set(ImageMagick_INCLUDE_DIRS ${ImageMagick_INCLUDE_DIRS})
set(ImageMagick_LIBRARIES ${ImageMagick_LIBRARIES})

foreach(_exe IN mogrify magick)
  if(ImageMagick_${_exe}_EXECUTABLE)
    execute_process(
      COMMAND ${ImageMagick_mogrify_EXECUTABLE} -version
      OUTPUT_VARIABLE imagemagick_version
      ERROR_QUIET
      OUTPUT_STRIP_TRAILING_WHITESPACE
    )
    if(imagemagick_version MATCHES "^Version: ImageMagick ([-0-9\\.]+)")
      set(ImageMagick_VERSION_STRING "${CMAKE_MATCH_1}")
    endif()
    unset(imagemagick_version)
  endif()
endforeach()

#---------------------------------------------------------------------
# Create IMPORTED targets for APIs
#---------------------------------------------------------------------
foreach(component ${ImageMagick_API_COMPONENTS})
  if(NOT TARGET Magick::${component})
    add_library(Magick::${component} INTERFACE IMPORTED)
    if(ImageMagick_${component}_LIBRARY)
      set_property(TARGET Magick::${component} PROPERTY
      IMPORTED_LOCATION ${ImageMagick_${component}_LIBRARY})
    endif()
    if(ImageMagick_${component}_LIBRARIES)
      set_property(TARGET Magick::${component} PROPERTY
      INTERFACE_LINK_LIBRARIES ${ImageMagick_${component}_LIBRARIES})
    endif()
  endif()
endforeach()
unset(ImageMagick_API_COMPONENTS)

#---------------------------------------------------------------------
# Standard Package Output
#---------------------------------------------------------------------
include(${CMAKE_CURRENT_LIST_DIR}/FindPackageHandleStandardArgs.cmake)
FIND_PACKAGE_HANDLE_STANDARD_ARGS(ImageMagick
                                  REQUIRED_VARS ${ImageMagick_REQUIRED_VARS}
                                  VERSION_VAR ImageMagick_VERSION_STRING
  )
# Maintain consistency with all other variables.
set(ImageMagick_FOUND ${IMAGEMAGICK_FOUND})

#---------------------------------------------------------------------
# DEPRECATED: Setting variables for backward compatibility.
#---------------------------------------------------------------------
set(IMAGEMAGICK_BINARY_PATH          ${ImageMagick_EXECUTABLE_DIR}
    CACHE PATH "Path to the ImageMagick binary directory.")
set(IMAGEMAGICK_CONVERT_EXECUTABLE   ${ImageMagick_convert_EXECUTABLE}
    CACHE FILEPATH "Path to ImageMagick's convert executable.")
set(IMAGEMAGICK_MOGRIFY_EXECUTABLE   ${ImageMagick_mogrify_EXECUTABLE}
    CACHE FILEPATH "Path to ImageMagick's mogrify executable.")
set(IMAGEMAGICK_IMPORT_EXECUTABLE    ${ImageMagick_import_EXECUTABLE}
    CACHE FILEPATH "Path to ImageMagick's import executable.")
set(IMAGEMAGICK_MONTAGE_EXECUTABLE   ${ImageMagick_montage_EXECUTABLE}
    CACHE FILEPATH "Path to ImageMagick's montage executable.")
set(IMAGEMAGICK_COMPOSITE_EXECUTABLE ${ImageMagick_composite_EXECUTABLE}
    CACHE FILEPATH "Path to ImageMagick's composite executable.")
mark_as_advanced(
  IMAGEMAGICK_BINARY_PATH
  IMAGEMAGICK_CONVERT_EXECUTABLE
  IMAGEMAGICK_MOGRIFY_EXECUTABLE
  IMAGEMAGICK_IMPORT_EXECUTABLE
  IMAGEMAGICK_MONTAGE_EXECUTABLE
  IMAGEMAGICK_COMPOSITE_EXECUTABLE
)
