/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkPExtractVOI.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPExtractVOI.h"

// VTK includes
#include "vtkImageData.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMPIController.h"
#include "vtkMPIUtilities.h"
#include "vtkMultiProcessController.h"
#include "vtkObjectFactory.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkStructuredImplicitConnectivity.h"

#include <cassert>
#include <sstream>

// #define DEBUG

vtkStandardNewMacro(vtkPExtractVOI);

//------------------------------------------------------------------------------
vtkPExtractVOI::vtkPExtractVOI()
{
  this->Controller = vtkMPIController::SafeDownCast(
          vtkMultiProcessController::GetGlobalController());
}

//------------------------------------------------------------------------------
vtkPExtractVOI::~vtkPExtractVOI()
{

}

//------------------------------------------------------------------------------
void vtkPExtractVOI::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
}

//------------------------------------------------------------------------------
int vtkPExtractVOI::RequestData(
        vtkInformation* request,
        vtkInformationVector** inputVector,
        vtkInformationVector* outputVector)
{
    // STEP 0: Execute superclass
    int rc = this->Superclass::RequestData(request,inputVector,outputVector);
    if( rc < 0 )
      {
      return( rc );
      }

    // STEP 1: Get output information
    vtkInformation* outInfo = outputVector->GetInformationObject(0);
    assert("pre: output invformation is NULL!" && (outInfo != NULL) );

    // STEP 2: Get the whole extent
    int wholeExtent[6];
    outInfo->Get(vtkStreamingDemandDrivenPipeline::WHOLE_EXTENT(),wholeExtent);

#ifdef DEBUG
    vtkMPIUtilities::SynchronizedPrintf(
        this->Controller,"wholeExtent=[%d,%d,%d,%d,%d,%d]\n",
        wholeExtent[0],wholeExtent[1],wholeExtent[2],
        wholeExtent[3],wholeExtent[4],wholeExtent[5]);
#endif

    // STEP 3: Get the structured grid instance in this process
    vtkImageData* grid = vtkImageData::SafeDownCast(
            outInfo->Get(vtkDataObject::DATA_OBJECT()));

#ifdef DEBUG
    int myExt[6];
    grid->GetExtent(myExt);

    vtkMPIUtilities::SynchronizedPrintf(
        this->Controller,"ext=[%d,%d,%d,%d,%d,%d]\n",
        myExt[0],myExt[1],myExt[2],myExt[3],myExt[4],myExt[5]);
#endif

    // STEP 4: Detect & resolve gaps if any
    vtkStructuredImplicitConnectivity* gridConnectivity =
        vtkStructuredImplicitConnectivity::New();
    gridConnectivity->SetWholeExtent(wholeExtent);

    // Register the grid, grid ID is the same as the process ID
    gridConnectivity->RegisterGrid(
      this->Controller->GetLocalProcessId(),
      grid->GetExtent(),
      NULL,
      grid->GetPointData()
      );

    // Establish neighbor connectivity & detect any gaps
    gridConnectivity->EstablishConnectivity();

    // Check if there are any gaps, if any close them now
    if( gridConnectivity->HasImplicitConnectivity() )
      {
#ifdef DEBUG
      vtkMPIUtilities::Printf(this->Controller, "Closing Gap...\n");
#endif
      // there are gaps, grow the grid to the right
      gridConnectivity->ExchangeData();

      gridConnectivity->GetOutputImageData(
        this->Controller->GetLocalProcessId(),grid);
      }

#ifdef DEBUG
    grid->GetExtent(myExt);
    vtkMPIUtilities::SynchronizedPrintf(
        this->Controller,"ext=[%d,%d,%d,%d,%d,%d]\n",
        myExt[0],myExt[1],myExt[2],myExt[3],myExt[4],myExt[5]);
#endif

    gridConnectivity->Delete();
    return( rc );
}

//------------------------------------------------------------------------------
int vtkPExtractVOI::RequestInformation(
        vtkInformation* request,
        vtkInformationVector** inputVector,
        vtkInformationVector* outputVector)
{
    int rc = this->Superclass::RequestInformation(
                request,inputVector,outputVector);
    return( rc );
}

//------------------------------------------------------------------------------
int vtkPExtractVOI::RequestUpdateExtent(
        vtkInformation* request,
        vtkInformationVector** inputVector,
        vtkInformationVector* outputVector)
{
    int rc = this->Superclass::RequestUpdateExtent(
                request,inputVector,outputVector);
    return( rc );
}
