/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile$
  Language:  C++
  Date:      $Date$
  Version:   $Revision$

  Copyright (c) 2002 Insight Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#include "cmBorlandMakefileGenerator.h"
#include "cmMakefile.h"
#include "cmStandardIncludes.h"
#include "cmSystemTools.h"
#include "cmSourceFile.h"
#include "cmMakeDepend.h"
#include "cmCacheManager.h"
#include "cmGeneratedFileStream.h"
#include "windows.h"


cmBorlandMakefileGenerator::cmBorlandMakefileGenerator()
{
  this->SetLibraryPathOption("-L");
  this->SetLibraryLinkOption("");
}

cmBorlandMakefileGenerator::~cmBorlandMakefileGenerator()
{
}


void cmBorlandMakefileGenerator::ComputeSystemInfo()
{
  // now load the settings
  if(!m_Makefile->GetDefinition("CMAKE_ROOT"))
    {
    cmSystemTools::Error(
      "CMAKE_ROOT has not been defined, bad GUI or driver program");
    return;
    }
  std::string outdir = m_Makefile->GetCurrentOutputDirectory();
  if(outdir.find('-') != std::string::npos)
    {
    std::string message = "The Borland command line tools do not support path names that have - in them.  Please re-name your output directory and use _ instead of -.";
    message += "\nYour path currently is: ";
    message += outdir;
    cmSystemTools::Error(message.c_str());
    }
  
  std::string fpath = 
    m_Makefile->GetDefinition("CMAKE_ROOT");
  fpath += "/Templates/CMakeBorlandWindowsSystemConfig.cmake";
  m_Makefile->ReadListFile(NULL,fpath.c_str());
}


  
void cmBorlandMakefileGenerator::OutputMakeVariables(std::ostream& fout)
{
  fout << "# NMake Makefile generated by cmake\n";
  const char* variables = 
    "# general varibles used in the makefile\n"
    "\n"
    "CMAKE_STANDARD_WINDOWS_LIBRARIES = @CMAKE_STANDARD_WINDOWS_LIBRARIES@\n"
    "CMAKE_C_FLAGS        = @CMAKE_C_FLAGS@ @BUILD_FLAGS@\n"
    "CMAKE_OBJECT_FILE_SUFFIX            = @CMAKE_OBJECT_FILE_SUFFIX@\n"
    "CMAKE_EXECUTABLE_SUFFIX             = @CMAKE_EXECUTABLE_SUFFIX@\n"
    "CMAKE_STATICLIB_SUFFIX              = @CMAKE_STATICLIB_SUFFIX@\n"
    "CMAKE_SHLIB_SUFFIX                  = @CMAKE_SHLIB_SUFFIX@\n" 
    "CMAKE_SHLIB_CFLAGS                  = -tWR\n" 
    "CMAKE_LINKER_FLAGS                  = @CMAKE_LINKER_FLAGS@ @LINKER_BUILD_FLAGS@\n"
    "CMAKE_CXX_FLAGS      = -P @CMAKE_CXX_FLAGS@ @BUILD_FLAGS@\n"
    "!IF \"$(OS)\" == \"Windows_NT\"\n"
    "NULL=\n"
    "!ELSE \n"
    "NULL=nul\n"
    "!ENDIF \n"
    "RM = del\n";
  std::string buildType = "CMAKE_CXX_FLAGS_";
  buildType +=  m_Makefile->GetDefinition("CMAKE_BUILD_TYPE");
  buildType = cmSystemTools::UpperCase(buildType);
  m_Makefile->AddDefinition("BUILD_FLAGS",
                            m_Makefile->GetDefinition(
                              buildType.c_str()));
  
  buildType = "CMAKE_LINKER_FLAGS_";
  buildType +=  m_Makefile->GetDefinition("CMAKE_BUILD_TYPE");
  buildType = cmSystemTools::UpperCase(buildType);
  m_Makefile->AddDefinition("LINKER_BUILD_FLAGS",
                            m_Makefile->GetDefinition(
                              buildType.c_str()));

  std::string replaceVars = variables;
  m_Makefile->ExpandVariablesInString(replaceVars);
  
  std::string ccompiler = m_Makefile->GetDefinition("CMAKE_C_COMPILER");
  cmSystemTools::ConvertToWindowsSlashes(ccompiler);
  fout << "CMAKE_C_COMPILER  = " << cmSystemTools::EscapeSpaces(ccompiler.c_str())
       << "\n";
  std::string cxxcompiler = m_Makefile->GetDefinition("CMAKE_CXX_COMPILER");
  cmSystemTools::ConvertToWindowsSlashes(cxxcompiler);
  fout << "CMAKE_CXX_COMPILER  = " << cmSystemTools::EscapeSpaces(cxxcompiler.c_str())
       << "\n";

  
  std::string cmakecommand = m_Makefile->GetDefinition("CMAKE_COMMAND");
  cmSystemTools::ConvertToWindowsSlashes(cmakecommand);
  fout << "CMAKE_COMMAND = " << cmSystemTools::EscapeSpaces(cmakecommand.c_str()) << "\n";

  fout << replaceVars.c_str();
  fout << "CMAKE_CURRENT_SOURCE = " 
       << ShortPath(m_Makefile->GetStartDirectory() )
       << "\n";
  fout << "CMAKE_CURRENT_BINARY = " 
       << ShortPath(m_Makefile->GetStartOutputDirectory())
       << "\n";
  fout << "CMAKE_SOURCE_DIR = " 
       << ShortPath(m_Makefile->GetHomeDirectory()) << "\n";
  fout << "CMAKE_BINARY_DIR = " 
       << ShortPath(m_Makefile->GetHomeOutputDirectory() )
       << "\n";
  // Output Include paths
  fout << "INCLUDE_FLAGS = ";
  std::vector<std::string>& includes = m_Makefile->GetIncludeDirectories();
  std::vector<std::string>::iterator i;
  fout << "-I" << cmSystemTools::EscapeSpaces(m_Makefile->GetStartDirectory()) << " ";
  for(i = includes.begin(); i != includes.end(); ++i)
    {
    std::string include = *i;
    // Don't output a -I for the standard include path "/usr/include".
    // This can cause problems with certain standard library
    // implementations because the wrong headers may be found first.
    if(include != "/usr/include")
      {
      fout << "-I" << cmSystemTools::EscapeSpaces(i->c_str()).c_str() << " ";
      }
    } 
  fout << m_Makefile->GetDefineFlags();
  fout << "\n\n";
}



void 
cmBorlandMakefileGenerator::
OutputBuildObjectFromSource(std::ostream& fout,
                            const char* shortName,
                            const cmSourceFile& source,
                            const char* extraCompileFlags,
                            bool shared)
{
  // force shared flag if building shared libraries
  if(cmSystemTools::IsOn(m_Makefile->GetDefinition("BUILD_SHARED_LIBS")))
    {
    shared = true;
    }
  // Header files shouldn't have build rules.
  if(source.IsAHeaderFileOnly())
    return;

  std::string comment = "Build ";
  std::string objectFile = std::string(shortName) + 
    this->GetOutputExtension(source.GetSourceExtension().c_str());
  cmSystemTools::ConvertToWindowsSlashes(objectFile);
  comment += objectFile + "  From ";
  comment += source.GetFullPath();
  std::string compileCommand;
  std::string ext = source.GetSourceExtension();
  if(ext == "c" )
    {
    compileCommand = "$(CMAKE_C_COMPILER) $(CMAKE_C_FLAGS) ";
    compileCommand += extraCompileFlags;
    if(shared)
      {
      compileCommand += "$(CMAKE_SHLIB_CFLAGS) ";
      }
    compileCommand += " -o";
    compileCommand += objectFile;
    compileCommand += " $(INCLUDE_FLAGS) -c ";
    compileCommand += 
      cmSystemTools::EscapeSpaces(source.GetFullPath().c_str());
    }
  else if (ext == "rc")
    {
    compileCommand = "$(RC) -o\"";
    compileCommand += objectFile;
    compileCommand += "\" ";
    compileCommand += 
      cmSystemTools::EscapeSpaces(source.GetFullPath().c_str());
    }
  else if (ext == "def")
    {
    // no rule to output for this one
    return;
    }
  // assume c++ if not c rc or def
  else
    {
    compileCommand = "$(CMAKE_CXX_COMPILER) $(CMAKE_CXX_FLAGS) ";
    compileCommand += extraCompileFlags;
    if(shared)
      {
      compileCommand += "$(CMAKE_SHLIB_CFLAGS) ";
      }
    compileCommand += " -o";
    compileCommand += objectFile;
    compileCommand += " $(INCLUDE_FLAGS) -c ";
    compileCommand += 
      cmSystemTools::EscapeSpaces(source.GetFullPath().c_str());
    }
  this->OutputMakeRule(fout,
                       comment.c_str(),
                       objectFile.c_str(),
                       cmSystemTools::EscapeSpaces(
                         source.GetFullPath().c_str()).c_str(),
                       compileCommand.c_str());
}

void cmBorlandMakefileGenerator::OutputSharedLibraryRule(std::ostream& fout, 
                                                       const char* name,
                                                       const cmTarget &t)
{
  std::string target = m_LibraryOutputPath + name;
  std::string libpath = target + ".lib";
  target += ".dll";
  cmSystemTools::ConvertToWindowsSlashes(libpath);
  cmSystemTools::ConvertToWindowsSlashes(target);
  target = cmSystemTools::EscapeSpaces(target.c_str());
  libpath = cmSystemTools::EscapeSpaces(libpath.c_str());
  std::string depend = "$(";
  depend += name;
  depend += "_SRC_OBJS) $(" + std::string(name) + "_DEPEND_LIBS)";
  std::string command = "$(CMAKE_CXX_COMPILER) -tWD $(CMAKE_SHLIB_CFLAGS) $(CMAKE_LINKER_FLAGS) @&&|\n";
  // must be executable name
  command += "-e";
  command += target;
  command += " ";
  std::strstream linklibs;
  this->OutputLinkLibraries(linklibs, name, t);
  linklibs << std::ends;
  // then the linker options -L and libraries (any other order will fail!)
  command += linklibs.str();
  delete [] linklibs.str();
  // then list of object files
  command += " $(" + std::string(name) + "_SRC_OBJS) ";
  std::string command2 = "implib -w ";
  command2 += libpath + " " + target;
  const std::vector<cmSourceFile>& sources = t.GetSourceFiles();
  for(std::vector<cmSourceFile>::const_iterator i = sources.begin();
      i != sources.end(); ++i)
    {
    if(i->GetSourceExtension() == "def")
      {
      command += "";
      command += i->GetFullPath();
      }
    }
  command += "\n|\n";
    
  std::string customCommands = this->CreateTargetRules(t, name);
  const char* cc = 0;
  if(customCommands.size() > 0)
    {
    cc = customCommands.c_str();
    }
  this->OutputMakeRule(fout, "rules for a shared library",
                       target.c_str(),
                       depend.c_str(),
                       command.c_str(),
                       command2.c_str(),
                       cc);
}

void cmBorlandMakefileGenerator::OutputModuleLibraryRule(std::ostream& fout, 
                                                       const char* name, 
                                                       const cmTarget &target)
{
  this->OutputSharedLibraryRule(fout, name, target);
}

void cmBorlandMakefileGenerator::OutputStaticLibraryRule(std::ostream& fout, 
                                                       const char* name,
                                                       const cmTarget &t)
{
  std::string target = m_LibraryOutputPath + std::string(name) + ".lib";
  cmSystemTools::ConvertToWindowsSlashes(target);
  target = cmSystemTools::EscapeSpaces(target.c_str());
  std::string depend = "$(";
  depend += std::string(name) + "_SRC_OBJS)";
  std::string command = "tlib  @&&|\n\t /p512 /a ";
  command += target;
  command += " ";
  std::string deleteCommand = "if exist ";
  deleteCommand +=  target;
  deleteCommand += " del ";
  deleteCommand += target;

  command += " $(";
  command += std::string(name) + "_SRC_OBJS)";
  command += "\n|\n";
  std::string comment = "rule to build static library: ";
  comment += name;
  std::string customCommands = this->CreateTargetRules(t, name);
  const char* cc = 0;
  if(customCommands.size() > 0)
    {
    cc = customCommands.c_str();
    }
  this->OutputMakeRule(fout,
                       comment.c_str(),
                       target.c_str(),
                       depend.c_str(),
                       deleteCommand.c_str(),
                       command.c_str(), cc);
}

void cmBorlandMakefileGenerator::OutputExecutableRule(std::ostream& fout,
                                                    const char* name,
                                                    const cmTarget &t)
{
  std::string target = m_ExecutableOutputPath + name + m_ExecutableExtension;
  cmSystemTools::ConvertToWindowsSlashes(target);
  target = cmSystemTools::EscapeSpaces(target.c_str());
  std::string depend = "$(";
  depend += std::string(name) + "_SRC_OBJS) $(" + std::string(name) + "_DEPEND_LIBS)";
  std::string command = 
    "$(CMAKE_CXX_COMPILER) ";
  command += " $(CMAKE_LINKER_FLAGS) -e" + target;
  if(cmSystemTools::IsOn(m_Makefile->GetDefinition("BUILD_SHARED_LIBS")))
    {
    command += " $(CMAKE_SHLIB_CFLAGS) ";
    }
  if(t.GetType() == cmTarget::WIN32_EXECUTABLE)
    {
    command +=  " -tWM ";
    }
  else
    {
    command += " -tWC ";
    }
  std::strstream linklibs;
  this->OutputLinkLibraries(linklibs, 0, t);
  linklibs << std::ends;
  command += linklibs.str();
  delete [] linklibs.str();
  command += " $(" + std::string(name) + "_SRC_OBJS) ";
  
  std::string comment = "rule to build executable: ";
  comment += name;
  std::string customCommands = this->CreateTargetRules(t, name);
  const char* cc = 0;
  if(customCommands.size() > 0)
    {
    cc = customCommands.c_str();
    }
  this->OutputMakeRule(fout, 
                       comment.c_str(),
                       target.c_str(),
                       depend.c_str(),
                       command.c_str(), cc);
}

  


std::string cmBorlandMakefileGenerator::GetOutputExtension(const char* s)
{
  std::string sourceExtension = s;
  if(sourceExtension == "def")
    {
    return "";
    }
  if(sourceExtension == "ico" || sourceExtension == "rc2")
    {
    return "";
    }
  if(sourceExtension == "rc")
    {
    return ".res";
    }
  return ".obj";
}



bool cmBorlandMakefileGenerator::SamePath(const char* path1, const char* path2)
{
  // first check to see if they are the same anyway
  if (strcmp(path1, path2) == 0)
    {
    return true;
    }
  // next short path and lower case both of them for the compare
  return 
    cmSystemTools::LowerCase(ShortPath(path1)) ==
    cmSystemTools::LowerCase(ShortPath(path2));
}


