#include <Block.h>
#include <limits>

std::vector<std::string> DomainBlock::nameMap;
std::map<int, DomainBlock *> DomainBlock::leafMap;

DomainBlock::DomainBlock()
{
    gid = -1;
    dom = -1;
    sub = -1;
    parent = NULL;
    skipSharedFaceStats = false;
    for(int i=0; i<6; i++)
        bbox[i]=0.0f;
    setNm(std::string(""));
    leafBlockType = NONE;
}

DomainBlock::DomainBlock(int d, float *b, std::string n)
{
    gid = -1;
    dom = d;
    sub = 0;
    parent = NULL;
    skipSharedFaceStats = false;
    setBBox(b);
    setNm(n);
    leafBlockType = NONE;
}

DomainBlock::DomainBlock(int d, double *b, std::string n)
{
    gid = -1;
    dom = d;
    sub = 0;
    parent = NULL;
    skipSharedFaceStats = false;
    setBBox(b);
    setNm(n);
    leafBlockType = NONE;
}

DomainBlock::~DomainBlock()
{
    for(int i=0; i<children.size(); i++)
        delete children[i];
    children.clear();
}

void
DomainBlock::setNm(const std::string &n)
{
    nm = "";
    if (n.empty() && dom >= 0)
    {
        char t[32];
        sprintf(t,"%d", dom);
        nm = t;
    }
    else
        nm = n;
}

DomainBlock *
DomainBlock::AddChild(float *bb, const char *n)
{
    char t[32];
    sprintf(t, "%s:%s", nm.c_str(), n);
    DomainBlock *blk = new DomainBlock(dom, bb, t);
    blk->parent = this;
    children.push_back(blk);
    return blk;
}

void
DomainBlock::GetLeaves(std::vector<DomainBlock *> &v)
{
    if (children.empty())
        v.push_back(this);
    else
        for (int i = 0; i < children.size(); i++)
            children[i]->GetLeaves(v);
}

void
DomainBlock::GetBBox(float *bb)
{
    for (int i = 0; i < 6; i++)
        bb[i] = bbox[i];
}

void
DomainBlock::GetExtents(float *ext)
{
    ext[0] = bbox[1]-bbox[0];
    ext[1] = bbox[3]-bbox[2];
    ext[2] = bbox[5]-bbox[4];
}

void
DomainBlock::GetExtents(avtVector &ext)
{
    ext[0] = bbox[1]-bbox[0];
    ext[1] = bbox[3]-bbox[2];
    ext[2] = bbox[5]-bbox[4];
}

int
DomainBlock::NumLeafs() const
{
    if (children.empty())
        return 1;
    
    int n = 0;
    for (int i = 0; i < children.size(); i++)
        n += children[i]->NumLeafs();
    return n;
}

DomainBlock *
DomainBlock::GetLeafFromIndex(int idx)
{
    std::vector<DomainBlock*> leaves;
    GetLeaves(leaves);
    return leaves[idx];
}

bool
DomainBlock::InBBox(float *p) const
{
    if (p[0] >= bbox[0] && p[0] <= bbox[1])
        if (p[1] >= bbox[2] && p[1] <= bbox[3])
            if (p[2] >= bbox[4] && p[2] <= bbox[5])
                return true;
    return false;
}
bool
DomainBlock::InBBox(const avtVector &p) const
{
    float pt[3] = {p.x,p.y,p.z};
    return InBBox(pt);
}

bool
DomainBlock::InBBox(const avtVector &p, const double &tol) const
{
    double bb[6] = {bbox[0]-tol, bbox[1]+tol,
                    bbox[2]-tol, bbox[3]+tol,
                    bbox[4]-tol, bbox[5]+tol};
    if (p.x >= bb[0] && p.x <= bb[1])
        if (p.y >= bb[2] && p.y <= bb[3])
            if (p.z >= bb[4] && p.z <= bb[5])
                return true;
    return false;                   
}

bool
DomainBlock::Inside(float *p, int &id) const
{
    if (!InBBox(p))
        return false;
    
    //hit a node.
    int sz = children.size();
    if (sz == 0)
    {
        id = sub;
        return true;
    }
    
    for (int i = 0; i < sz; i++)
        if (children[i]->Inside(p, id))
            return true;
    
    return false;
}

DomainBlock *
DomainBlock::GetLeaf(const avtVector &p)
{
    std::vector<DomainBlock*> leaves;
    GetLeaves(leaves);
    for (int i = 0; i < leaves.size(); i++)
        if (leaves[i]->InBBox(p))
            return leaves[i];

    double tol = 1e-4;
    for (int i = 0; i < leaves.size(); i++)
        if (leaves[i]->InBBox(p, tol))
            return leaves[i];
    
    return NULL;
}

void
DomainBlock::dump(ostream &s, bool rawNums) const
{
    s<<"["<<nm<<"] ";
    //dumpBBox(s);
    s<<"[";
    for (int i = 0; i < data.size(); i++)
    {
        if ( i > 0)
            s<<" ";
        data[i].dump(s, rawNums);
    }
    s<<"]";
}

void
DomainBlock::dumpBBox(ostream& s) const
{
    s<<"{("<<bbox[0]<<","<<bbox[1]<<") ";
    s<<"("<<bbox[2]<<","<<bbox[3]<<") ";
    s<<"("<<bbox[4]<<","<<bbox[5]<<")} ";
}

void
DomainBlock::dump(ostream &s, int lvl, bool rawNums, std::string indent) const
{
    s<<indent;
    dump(s, rawNums);
    s<<endl;
    if (lvl > 0)
    {
        for (int i=0; i<children.size(); i++)
            children[i]->dump(s, lvl-1, rawNums, indent+" ");
    }
}

static void
inflate(float *x, float xi)
{
    x[0] -= xi;
    x[1] += xi;
    x[2] -= xi;
    x[3] += xi;
    x[4] -= xi;
    x[5] += xi;
}

static void
mkCorners(std::vector<avtVector> &pts, float *bb)
{
    for (int i = 0; i < 2; i++)
        for (int j = 2; j < 4; j++)
            for (int k = 4; k < 6; k++)
                pts.push_back(avtVector(bb[i], bb[j], bb[k]));
}

static bool
ptInside(const avtVector &p, float *bb)
{
    return (p.x >= bb[0] && p.x <= bb[1] &&
            p.y >= bb[2] && p.y <= bb[3] &&
            p.z >= bb[4] && p.z <= bb[5]);
}

bool
DomainBlock::BlocksShareFace(DomainBlock *blk)
{
    if (blk == NULL)
        return false;

    if (leafBlockType == INTERNAL || blk->leafBlockType == INTERNAL)
        return false;
    if (children.size() > 0 || blk->children.size() > 0)
        return false;
    if (dom == blk->dom)
        return false;

    if (1)
    {
        float bb0[6], bb1[6], bb0i[6], bb1i[6];
        GetBBox(bb0);
        blk->GetBBox(bb1);
        memcpy(bb0i, bb0, 6*sizeof(float));
        memcpy(bb1i, bb1, 6*sizeof(float));
        float eps = std::numeric_limits<float>::epsilon() * 10.0f;
        inflate(bb0i, eps);
        inflate(bb1i, eps);

        std::vector<avtVector> pts0, pts1;
        mkCorners(pts0, bb0);
        mkCorners(pts1, bb1);
        
        for (int i = 0; i < pts0.size(); i++)
            if (ptInside(pts0[i], bb1i))
                return true;
        
        for (int i = 0; i < pts1.size(); i++)
            if (ptInside(pts1[i], bb0i))
                return true;
        
        return false;
    }
    else
    {
        LeafBlockType b0 = leafBlockType, b1 = blk->leafBlockType;

        //cout<<"Blocks Share face? ("<<nm<<","<<blk->nm<<")"<<endl;
        
        if (b0 == X_MIN && b1 == X_MAX) return true;
        if (b0 == Y_MIN && b1 == Y_MAX) return true;
        if (b0 == Z_MIN && b1 == Z_MAX) return true;
        
        if (b0 == X_MAX && b1 == X_MIN) return true;
        if (b0 == Y_MAX && b1 == Y_MIN) return true;
        if (b0 == Z_MAX && b1 == Z_MIN) return true;
        
        return false;
    }
}

bool
DomainBlock::AddBlockData(DomainBlock *dstBlk, int numICs, int numIters, int totalNumICs)
{
    if (dstBlk == NULL)
        return false;

    /*
    if (!skipSharedFaceStats)
    {
        if (BlocksShareFace(dstBlk))
        {
            cout<<"ERROR: "<<nm<<" --> "<<dstBlk->nm<<endl;
            cout<<"  ";
            dumpBBox(cout);
            cout<<"  ";
            dstBlk->dumpBBox(cout);
            cout<<endl;
        }
    }
    */

    if (skipSharedFaceStats && BlocksShareFace(dstBlk))
        return false;
    
    bool foundIt = false;
    for (int i = 0; i < data.size(); i++)
        if (data[i].blk == dstBlk)
        {
            data[i].numICs += numICs;
            data[i].numIters += numIters;
            data[i].totalNumICs += totalNumICs;
            foundIt = true;
        }
    
    if (!foundIt)
    {
        blockData b(dstBlk, numICs, numIters, totalNumICs);
        data.push_back(b);
    }

    if (parent && dstBlk->parent)
    {
        //Arg... Interior block is off by a level. No children, so skip a level.
        if (nm[nm.size()-1] == 'I')
            parent->AddBlockData(dstBlk->parent->parent, numICs, numIters, totalNumICs);
        else
            parent->AddBlockData(dstBlk->parent, numICs, numIters, totalNumICs);
    }
    return true;
}

void
DomainBlock::UnifyData()
{
    expIt = 0;
    double totNumIters = 0;
    int totICs = 0;
    for (int i = 0; i < data.size(); i++)
    {
        if(!i)
        {
            minIt = data[i].numIters;
            maxIt = data[i].numIters;
        }
        else
        {
            if(data[i].numIters < minIt)
                minIt = data[i].numIters;
            else if(data[i].numIters > maxIt)
                maxIt = data[i].numIters;
        }
        totICs += data[i].numICs;
        totNumIters += data[i].numIters;
    }
    for (int i = 0; i < data.size(); i++)
        data[i].totalNumICs = totICs;

    expIt = totNumIters/totICs;
    //cout<<minIt<<" "<<maxIt<<" "<<expIt<<endl;
    for (int i = 0; i < data.size(); i++)
        data[i].computeAvg();
    sort(data.begin(), data.end(), blockData::rcmp);
    
    for (int i = 0; i < children.size(); i++)
        children[i]->UnifyData();
}

int
DomainBlock::SetIDs(int id)
{
    sub = -1;

    int sz = children.size();
    if (sz == 0)
    {
        sub = id;
        id++;
    }
    else
    {
        for (int i = 0; i < sz; i++)
            id = children[i]->SetIDs(id);
    }
    
    return id;
}

void
DomainBlock::DoSubdivideUniform(int nx, int ny, int nz)
{
    //Nothing to subdivide....
    if (nx == 1 && ny == 1 && nz == 1)
        return;
    
    float b[6];
    char n[64];
    
    float dx = (bbox[1]-bbox[0]) / (float)nx;
    float dy = (bbox[3]-bbox[2]) / (float)ny;
    float dz = (bbox[5]-bbox[4]) / (float)nz;

    for (int i = 0; i < nx; i++)
        for (int j = 0; j < ny; j++)
            for (int k = 0; k < nz; k++)
            {
                b[0] = bbox[0] + i*dx;
                b[1] = b[0] + dx;
                b[2] = bbox[2] + j*dy;
                b[3] = b[2] + dy;
                b[4] = bbox[4] + k*dz;
                b[5] = b[4] + dz;

                sprintf(n, "%d%d%d", i,j,k);
                DomainBlock *blk = AddChild(b, n);
                
                //Propagate the block type to the children.
                blk->leafBlockType = leafBlockType;
            }
}

void
DomainBlock::DoSubdivideFaces(int nx, int ny, int nz, float pct)
{
    float px = pct*(bbox[1]-bbox[0]);
    float py = pct*(bbox[3]-bbox[2]);
    float pz = pct*(bbox[5]-bbox[4]);
    DomainBlock *blk;

    //Make the interior.
    float b[6] = {bbox[0]+px, bbox[1]-px,
                  bbox[2]+py, bbox[3]-py,
                  bbox[4]+pz, bbox[5]-pz};
    blk = AddChild(b, "I");
    blk->leafBlockType = INTERNAL;

    //Make the Xmin:
    float bbx[6] = {bbox[0], bbox[0]+px,
                    bbox[2], bbox[3],
                    bbox[4], bbox[5]};
    blk = AddChild(bbx, "x");
    blk->leafBlockType = X_MIN;
    blk->SubdivideUniform(1, ny, nz, blk->leafBlockType);

    //Make the Xmax:
    float bbX[6] = {bbox[1]-px, bbox[1],
                    bbox[2], bbox[3],
                    bbox[4], bbox[5]};
    blk = AddChild(bbX, "X");
    blk->leafBlockType = X_MAX;
    blk->SubdivideUniform(1, ny, nz, blk->leafBlockType);

    //Make the Ymin:
    float bby[6] = {bbox[0]+px, bbox[1]-px,
                    bbox[2], bbox[2]+py,
                    bbox[4], bbox[5]};
    blk = AddChild(bby, "y");
    blk->leafBlockType = Y_MIN;
    blk->SubdivideUniform(nx, 1, nz, blk->leafBlockType);

    //Make the Ymax:
    float bbY[6] = {bbox[0]+px, bbox[1]-px,
                    bbox[3]-py, bbox[3],
                    bbox[4], bbox[5]};
    blk = AddChild(bbY, "Y");
    blk->leafBlockType = Y_MAX;
    blk->SubdivideUniform(nx, 1, nz, blk->leafBlockType);

    //Make the Zmin:
    float bbz[6] = {bbox[0]+px, bbox[1]-px,
                    bbox[2]+py, bbox[3]-py,
                    bbox[4], bbox[4]+pz};
    blk = AddChild(bbz, "z");
    blk->leafBlockType = Z_MIN;
    blk->SubdivideUniform(nx, ny, 1, blk->leafBlockType);

    //Make the Zmax:
    float bbZ[6] = {bbox[0]+px, bbox[1]-px,
                    bbox[2]+py, bbox[3]-py,
                    bbox[5]-pz, bbox[5]};
    blk = AddChild(bbZ, "Z");
    blk->leafBlockType = Z_MAX;
    blk->SubdivideUniform(nx, ny, 1, blk->leafBlockType);
}

void
DomainBlock::SubdivideUniform(int nx, int ny, int nz, LeafBlockType bt)
{
    leafBlockType = bt;
    DoSubdivideUniform(nx, ny, nz);
    SetIDs(0);
}

void
DomainBlock::SubdivideFaces(int nx, int ny, int nz, float pct)
{
    leafBlockType = NON_UNIFORM;
    DoSubdivideFaces(nx, ny, nz, pct);
    SetIDs(0);
}

/*
inline ostream&
operator<<(ostream &out, const DomainBlock &b)
{
    out<<"["<<b.dom;
    if (b.sub != -1)
        out<<"."<<b.sub;
    if (b.nm != "")
        out<<":"<<b.nm;
    out<<"] ";

    out<<"{("<<b.bbox[0]<<","<<b.bbox[1]<<") ";
    out<<"("<<b.bbox[2]<<","<<b.bbox[3]<<") ";
    out<<"("<<b.bbox[4]<<","<<b.bbox[5]<<")} ";
    //out<<b.data;
    return out;
}
*/

void
DomainBlock::Dump(DomainBlock *b, ostream &s, int lvl, bool rawNums)
{
    b->dump(s,lvl, rawNums);
}

void
DomainBlock::Dump(std::vector<DomainBlock*> &v, ostream &s, int lvl, bool rawNums)
{
    for (int i = 0; i < v.size(); i++)
        DomainBlock::Dump(v[i], s, lvl, rawNums);
}

int
DomainBlock::TotalNumLeaves(std::vector<DomainBlock*> &v)
{
    int N = 0;
    for (int i = 0; i < v.size(); i++)
        N += v[i]->NumLeafs();
    return N;
}

void
DomainBlock::CreateBlockInfo(std::vector<DomainBlock*> &v, int nDom, avtIntervalTree *it,
                             bool subdivUniform, int nX, int nY, int nZ, double pct,
                             bool _skipSharedFaceStats)
{
    double bbox[6];

    v.resize(nDom);

    int totLeafs = 0;
    for (int i = 0; i < nDom; i++)
    {
        it->GetElementExtents(i, bbox);
        v[i] = new DomainBlock(i,bbox);
        v[i]->skipSharedFaceStats = _skipSharedFaceStats;
        if (subdivUniform)
            v[i]->SubdivideUniform(nX, nY, nZ);
        else
            v[i]->SubdivideFaces(nX, nY, nZ, pct);

        totLeafs += v[i]->NumLeafs();
    }
    
    //Set the GIDs, and the name map.
    nameMap.resize(totLeafs);
    int id = 0;
    for (int i = 0; i < v.size(); i++)
    {
        std::vector<DomainBlock*> leaves;
        v[i]->GetLeaves(leaves);
        v[i]->skipSharedFaceStats = _skipSharedFaceStats;
        for (int j = 0; j < leaves.size(); j++)
        {
            leaves[j]->gid = id;
            leaves[j]->skipSharedFaceStats = _skipSharedFaceStats;
            id++;
            nameMap[leaves[j]->gid] = leaves[j]->nm;
            leafMap[leaves[j]->gid] = leaves[j];
        }
    }
}

DomainBlock *
DomainBlock::GetBlockFromGID(std::vector<DomainBlock*> &v, int gid)
{
    std::map<int,DomainBlock*>::iterator it = leafMap.find(gid);
    if (it == leafMap.end())
        return NULL;
    
    return it->second;
    
    /*
    for (int i = 0; i < v.size(); i++)
    {
        std::vector<DomainBlock*> leaves;
        v[i]->GetLeaves(leaves);
        int n = leaves.size();
        if ( n > 0 && gid >= leaves[0]->gid && gid <= leaves[n-1]->gid)
            for (int j = 0; j < n; j++)
                if (leaves[j]->gid == gid)
                    return leaves[j];
    }
    return NULL;
    */
}

DomainBlock *
DomainBlock::GetLeaf_FIX_THIS(std::vector<DomainBlock*> &v, const avtVector &p)
{
    DomainBlock *blk = NULL;
    for (int i = 0; !blk && i < v.size(); i++)
        blk = v[i]->GetLeaf(p);
    
    return blk;
}


