/*****************************************************************************
*
* Copyright (c) 2000 - 2014, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                              avtPODICAlgorithm.C                          //
// ************************************************************************* //

#include <avtPODICAlgorithm.h>
#include <TimingsManager.h>
#include <VisItStreamUtil.h>

using namespace std;

#ifdef PARALLEL

int avtPODICAlgorithm::TERMINATE = 1;

// ****************************************************************************
// Method:  avtPODICAlgorithm::avtPODICAlgorithm
//
// Purpose: constructor
//
// Programmer:  Dave Pugmire
// Creation:    March 21, 2012
//
// ****************************************************************************


avtPODICAlgorithm::avtPODICAlgorithm(avtPICSFilter *picsFilter, int count, bool async, bool useT, bool useAdv)
    : avtParICAlgorithm(picsFilter)
{
    maxCount = count;
    numTerminated = 0;
    asyncComm = async;
    useTimers = useT;
    useAdvTimer = useAdv;
}

// ****************************************************************************
// Method:  avtPODICAlgorithm::~avtPODICAlgorithm
//
// Purpose: destructor
//
// Programmer:  Dave Pugmire
// Creation:    March 21, 2012
//
// ****************************************************************************

avtPODICAlgorithm::~avtPODICAlgorithm()
{
}

// ****************************************************************************
// Method:  avtPODICAlgorithm::Initialize
//
// Purpose: Initialize
//
// Programmer:  Dave Pugmire
// Creation:    March 21, 2012
//
// ****************************************************************************

void
avtPODICAlgorithm::Initialize(vector<avtIntegralCurve *> &seeds)
{
    if (useTimers) runSW.start();
    if (useTimers) initSW.start();
    
    int numRecvs = 64;
    if (numRecvs > nProcs)
        numRecvs = nProcs-1;

    int msgSz = 2;
    int numMsgs = 0;
    if (asyncComm)
        numMsgs = numRecvs;
    
    avtParICAlgorithm::InitializeBuffers(seeds, msgSz, numMsgs, numRecvs);
    if (useTimers) initSW.stop();
    if (useTimers) runSW.stop();
    
    AddIntegralCurves(seeds);
}

// ****************************************************************************
// Method:  avtPODICAlgorithm::AddIntegralCurves
//
// Purpose: Add ICs.
//
// Programmer:  Dave Pugmire
// Creation:    March 21, 2012
//
// ****************************************************************************

void
avtPODICAlgorithm::AddIntegralCurves(vector<avtIntegralCurve*> &ics)
{
    int timerHandle = visitTimer->StartTimer();

    set<int> myBlocks;
    std::list<BlockEntry>::const_iterator it;
    for (it = picsFilter->blockQueue.begin(); it != picsFilter->blockQueue.end(); it++)
        myBlocks.insert((*it).id.domain);
    
    //Get the ICs that I own.
    for (size_t i = 0; i < ics.size(); i++)
    {
        avtIntegralCurve *ic = ics[i];
        //picsFilter->FindCandidateBlocks(ic);
        if (myBlocks.find(ic->blockList.front().domain) != myBlocks.end())
        {
            ic->originatingRank = rank;
            activeICs.push_back(ic);
        }
        else
            delete ic;
        /*
        
        if (DomainLoaded(ic->blockList.front()))
        {
            ic->originatingRank = rank;
            activeICs.push_back(ic);
            
#ifdef USE_IC_STATE_TRACKING
            ic->InitTrk();
#endif
        }
        else
            delete ic;
        */
    }

    /*
    if (DebugStream::Level1())
    {
        debug1<<"My ICcount= "<<activeICs.size()<<endl;
        debug1<<"I own: [";
        for (int i = 0; i < numDomains; i++)
        {
            BlockIDType d(i,0);
            if (OwnDomain(d)) 
            {
                debug1<<i<<" ";
            }
        }
        debug1<<"]\n";
    }
    */

    totalNumICs = activeICs.size();
    SumIntAcrossAllProcessors(totalNumICs);
    visitTimer->StopTimer(timerHandle, "AddIntegralCurves");
}

// ****************************************************************************
// Method:  avtPODICAlgorithm::PreRunAlgorithm
//
// Purpose:
//
// Programmer:  Dave Pugmire
// Creation:    March 21, 2012
//
// ****************************************************************************

void
avtPODICAlgorithm::PreRunAlgorithm()
{
    picsFilter->InitializeLocators();
}

// ****************************************************************************
// Method:  avtPODICAlgorithm::RunAlgorithm
//
// Purpose: Run algorithm.
//
// Programmer:  Dave Pugmire
// Creation:    March 21, 2012
//
// Modifications:
//
//   Hank Childs, Wed Mar 28 08:36:34 PDT 2012
//   Add support for terminated particle status.
//
// ****************************************************************************

void
avtPODICAlgorithm::RunAlgorithm()
{
    debug1<<"avtPODICAlgorithm::RunAlgorithm() activeICs: "<<activeICs.size()<<" inactiveICs: "<<inactiveICs.size()<<endl;
    
    domIntegrateSteps.resize(numDomains, 0);
    int timer = visitTimer->StartTimer();
    
    if (useTimers) runSW.t += picsFilter->InitialIOTime;
    if (useTimers) runSW.start();

    if (useTimers) commSW.start();
    bool done = false; //HandleCommunication();
    if (useTimers) commSW.stop();
    int round = 0;
    ICLOG<<"POD_Algo: BEGIN"<<endl;
    debug1<<"RunAlgorithm"<<endl;
    while (!done)
    {
        //if (PAR_Rank() == 0) cout<<"******************************* ROUND "<<round<<endl;
        int cnt = 0;
        ICLOG<<"BEG "<<round<<": active:"<<activeICs.size()<<" term: "<<terminatedICs.size()<<" inactive: "<<inactiveICs.size()<<endl;
        while (cnt < maxCount && !activeICs.empty())
        {
            avtIntegralCurve *ic = activeICs.front();
            activeICs.pop_front();
            avtVector p0 = ic->CurrentLocation();
            int d = ic->blockList.front().domain;

            if (useAdvTimer) advSW.start();
            int nIters = AdvectParticle(ic);
            if (useAdvTimer) advSW.stop();
            /* This is a lot slower...... delays sync points...
            do
            {
                AdvectParticle(ic);
            }
            while (ic->status.Integrateable() &&
                   DomainLoaded(ic->blockList.front()));
            */
            domIntegrateSteps[d] += nIters;
            //ic->trk<<p0<<" "<<d<<" --"<<nIters<<"-->"<<ic->blockList<<" "<<ic->CurrentLocation()<<endl;
            
            if (ic->status.EncounteredSpatialBoundary())
            {
                if (!ic->blockList.empty() && DomainLoaded(ic->blockList.front()))
                    activeICs.push_back(ic);
                else
                    inactiveICs.push_back(ic);
            }
            else
            {
                numTerminated++;
                terminatedICs.push_back(ic);
            }
            
            cnt++;
        }
	debug1<<"RunAlgorithm DONE"<<endl;
	
        ICLOG<<"END "<<round<<": active:"<<activeICs.size()<<" term: "<<terminatedICs.size()<<" inactive: "<<inactiveICs.size()<<endl;
        if (useTimers) commSW.start();
        done = HandleCommunication();
        if (useTimers) commSW.stop();
        round++;
        ICLOG<<endl;
    }
    
    if (useTimers) runSW.stop();
    ICLOG<<"POD_Algo: END"<<endl<<endl;

    if (useTimers)
    {
        stringstream sstr;
        sstr<<"maxCnt: "<<maxCount<<" ";
        DumpInfo("", sstr.str());

        Barrier();
        if (abortWhenDone)
            MPI_Abort(VISIT_MPI_COMM, -1);
    }

    TotalTime.value += visitTimer->StopTimer(timer, "Execute");
}

// ****************************************************************************
// Method:  avtPODICAlgorithm::HandleCommunication
//
// Purpose: Process communication.
//
// Programmer:  Dave Pugmire
// Creation:    March 21, 2012
//
// Modifications:
//
//   Dave Pugmire, Fri Mar  8 15:49:14 EST 2013
//   Bug fix. Ensure that the same IC isn't sent to the same rank. Also, when
//   an IC is received, set the domain from the particle point.
//
// ****************************************************************************

bool
avtPODICAlgorithm::DoSyncCommunication()
{
    int numICs = inactiveICs.size() + activeICs.size();
    
    //See if we're done.
    if (useTimers) syncWaitSW.start();
    SumIntAcrossAllProcessors(numICs);
    if (useTimers) syncWaitSW.stop();
    MsgCnt.value++;

    //debug1<<"avtPODICAlgorithm::HandleCommunication() numICs= "<<numICs<<endl;
    ICLOG<<" Num ICS= "<<numICs<<endl;
    if (numICs == 0)
        return true;

    //Tell everyone how many ICs are coming their way.
    int *icCounts = new int[nProcs], *allCounts = new int[nProcs];
    for (int i = 0; i < nProcs; i++)
        icCounts[i] = 0;
    
    list<avtIntegralCurve*>::iterator s;
    map<int, vector<avtIntegralCurve *> > sendICs;
    map<int, vector<avtIntegralCurve *> >::iterator it;
    list<avtIntegralCurve*> tmp;
    set<int> receivers;
    for (s = inactiveICs.begin(); s != inactiveICs.end(); s++)
    {
        int domRank = DomainToRank((*s)->blockList.front());
        icCounts[domRank]++;
        receivers.insert(domRank);            
        //Add to sending map.
        it = sendICs.find(domRank);
        if (it == sendICs.end())
        {
            vector<avtIntegralCurve *> v;
            v.push_back(*s);
            sendICs[domRank] = v;
        }
        else
            it->second.push_back(*s);
    }
    if (!inactiveICs.empty()) ICLOG<<" Sending: "<<inactiveICs.size()<<" to "<<receivers<<endl;
    inactiveICs.clear();
    
    SumIntArrayAcrossAllProcessors(icCounts, allCounts, nProcs);
    bool anyToSend = false;
    for (int i = 0; i < nProcs && !anyToSend; i++)
        anyToSend = (allCounts[i] > 0);
    
    int incomingCnt = allCounts[rank];
    
    //Send out my ICs.
    for (it = sendICs.begin(); it != sendICs.end(); it++)
        SendICs(it->first, it->second);

    //Wait till I get all my ICs.
    set<int> senders;
    int totIncoming = incomingCnt;
    while (incomingCnt > 0)
    {
        list<ICCommData> ics;
        list<ICCommData>::iterator s;

        RecvAny(NULL, &ics, NULL, true);
        for (s = ics.begin(); s != ics.end(); s++)
        {
            avtIntegralCurve *ic = (*s).ic;

            //See if I have this block.
            BlockIDType blk;
            list<BlockIDType> tmp;
            bool blockFound = false;
            while (!ic->blockList.empty())
            {
                blk = ic->blockList.front();
                ic->blockList.pop_front();
                if (DomainLoaded(blk))
                {
                    if (picsFilter->ICInBlock(ic, blk))
                    {
                        ic->status.ClearSpatialBoundary();
                        ic->blockList.clear();
                        ic->blockList.push_back(blk);
                        blockFound = true;
                        activeICs.push_back(ic);
                        senders.insert((*s).rank);
                        break;
                    }
                }
                else
                    tmp.push_back(blk);
            }

            //IC Not in my blocks.  Terminate if blockList empty, or send to
            //block owner of next block in list.
            if (!blockFound)
            {
                ic->blockList = tmp;
                if (ic->blockList.empty())
                    terminatedICs.push_back(ic);
                else
                    inactiveICs.push_back(ic);
            }
        }
        
        incomingCnt -= ics.size();
        CheckPendingSendRequests();
    }

    if (totIncoming > 0) ICLOG<<" Received: "<<totIncoming<<" from "<<senders<<endl;    
    CheckPendingSendRequests(); 
    delete [] icCounts;
    delete [] allCounts;
    
    return false;
}

bool
avtPODICAlgorithm::DoAsyncCommunication()
{
    //cout<<rank<<" HandleCommunication("<<numTerm<<") totalICs= "<<totalNumICs<<endl;
    CheckPendingSendRequests();
    
    //Send out ICs.
    if (!inactiveICs.empty())
    {
        CommICs(inactiveICs);
        inactiveICs.clear();
    }

    CommTerm();

    //Now, see if anything is coming my way.
    vector<MsgCommData> msgs;
    list<ICCommData> ics;
    bool blockAndWait = activeICs.empty() && (totalNumICs > 0);
    
    if (blockAndWait) ICLOG<<" BlockAndWait. #ICs= "<<totalNumICs<<endl;
    if (useTimers) syncWaitSW.start();
    RecvAny(&msgs, &ics, NULL, blockAndWait);
    if (useTimers) syncWaitSW.stop();

    //Process the messages.
    for (int i = 0; i < msgs.size(); i++)
    {
        int fromRank = msgs[i].rank;
        vector<int> &msg = msgs[i].message;
        int msgType = msg[0];
        if (msgType == avtPODICAlgorithm::TERMINATE)
        {
            totalNumICs -= msg[1];
            ICLOG<<" Recv Term: "<<msg[1]<<" from "<<fromRank<<endl;
        }
    }
    if (!msgs.empty()) ICLOG<<" NumICs= "<<totalNumICs<<endl;

    list<avtIntegralCurve *> notMine;
    list<ICCommData>::iterator s;
    set<int> senders;
    int totIncoming = 0;
    for (s = ics.begin(); s != ics.end(); s++)
    {
        avtIntegralCurve *ic = (*s).ic;
        //See if I have this block.
        BlockIDType blk;
        list<BlockIDType> tmp;
        bool blockFound = false;
        while (!ic->blockList.empty())
        {
            blk = ic->blockList.front();
            ic->blockList.pop_front();
            if (DomainLoaded(blk))
            {
                if (picsFilter->ICInBlock(ic, blk))
                {
                    ic->status.ClearSpatialBoundary();
                    ic->blockList.clear();
                    ic->blockList.push_back(blk);
                    blockFound = true;
                    activeICs.push_back(ic);
                    senders.insert((*s).rank);
                    totIncoming++;
                    break;
                }
            }
            else
                tmp.push_back(blk);
        }

        //IC Not in my blocks.  Terminate if blockList empty, or send to
        //block owner of next block in list.
        if (!blockFound)
        {
            ic->blockList = tmp;
            if (ic->blockList.empty())
            {
                terminatedICs.push_back(ic);
                numTerminated++;
            }
            else
                notMine.push_back(ic);
        }
    }
    if (totIncoming > 0) ICLOG<<" Received: "<<totIncoming<<" from "<<senders<<endl;
    
    if (!notMine.empty())
        CommICs(notMine);

    CommTerm();

    //cout<<rank<<" totalNumICs= "<<totalNumICs<<endl;
    CheckPendingSendRequests();
    if (totalNumICs < 0)
        EXCEPTION1(VisItException, "Error: Number of ICs is negative. Bug in communication");
    
    return (totalNumICs == 0);
}

void
avtPODICAlgorithm::CommICs(list<avtIntegralCurve *> &l)
{
    //cout<<rank<<": sending "<<l.size()<<endl;

    list<avtIntegralCurve*>::iterator s;
    map<int, vector<avtIntegralCurve *> > sendICs;
    map<int, vector<avtIntegralCurve *> >::iterator it;
    set<int> receivers;
    for (s = l.begin(); s != l.end(); s++)
    {
        int domRank = DomainToRank((*s)->blockList.front());
        receivers.insert(domRank);
        if (domRank == rank)
            activeICs.push_back(*s);
        else
        {
            //Add to sending map.
            it = sendICs.find(domRank);
            if (it == sendICs.end())
            {
                vector<avtIntegralCurve *> v;
                v.push_back(*s);
                sendICs[domRank] = v;
            }
            else
                it->second.push_back(*s);
        }
    }

    if (!inactiveICs.empty()) ICLOG<<" Sending: "<<inactiveICs.size()<<" to "<<receivers<<endl;
    
    //Send out my ICs.
    for (it = sendICs.begin(); it != sendICs.end(); it++)
        SendICs(it->first, it->second);
}

void
avtPODICAlgorithm::CommTerm()
{
    if (numTerminated == 0)
        return;
    
    totalNumICs -= numTerminated;

    //Tell everyone else the good news.
    vector<int> msg(2);
    msg[0] = avtPODICAlgorithm::TERMINATE;
    msg[1] = numTerminated;
    for (int i = 0; i < nProcs; i++)
        if (i != rank)
            SendMsg(i, msg);

    ICLOG<<" Comm Terminates: "<<numTerminated<<endl;
    numTermMsgs++;
    numTerminated = 0;
    //cout<<rank<<" CommTerm("<<numTerm<<") total= "<<totalNumICs<<endl;
}

#endif
