/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangeWedge.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkLagrangeWedge.h"

#include "vtkCellData.h"
#include "vtkDoubleArray.h"
#include "vtkWedge.h"
#include "vtkIdList.h"
#include "vtkLagrangeCurve.h"
#include "vtkLagrangeInterpolation.h"
#include "vtkLagrangeQuadrilateral.h"
#include "vtkLagrangeTriangle.h"
#include "vtkLine.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkPointData.h"
#include "vtkPoints.h"
#include "vtkTriangle.h"
#include "vtkVector.h"
#include "vtkVectorOperators.h"

vtkStandardNewMacro(vtkLagrangeWedge);

vtkLagrangeWedge::vtkLagrangeWedge()
{
  this->Approx = NULL;
  this->Order[0] = this->Order[1] = this->Order[2] = 1;
  this->Order[3] = 6;
  this->Points->SetNumberOfPoints(6);
  this->PointIds->SetNumberOfIds(6);
  for (int i = 0; i < 6; i++)
    {
    this->Points->SetPoint(i, 0.0, 0.0, 0.0);
    this->PointIds->SetId(i,-1);
    }
}

vtkLagrangeWedge::~vtkLagrangeWedge()
{
}

void vtkLagrangeWedge::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "Order: " << this->GetOrder(3) << "\n";
  if (this->PointParametricCoordinates)
    {
    os
      << indent << "PointParametricCoordinates: "
      << this->PointParametricCoordinates->GetNumberOfPoints()
      << " entries\n";
    }
  os << indent << "Approx: " << this->Approx << "\n";
}

vtkCell* vtkLagrangeWedge::GetEdge(int edgeId)
{
  vtkLagrangeCurve* result = this->BdyEdge.GetPointer();
  const int* order = this->GetOrder();
  int oi = vtkLagrangeInterpolation::GetVaryingParameterOfWedgeEdge(edgeId);
  vtkIdType npts = order[oi >= 0 ? oi : 0] + 1;
  std::vector<vtkIdType> econn(npts);
  result->Points->SetNumberOfPoints(npts);
  vtkVector2i eidx = vtkLagrangeInterpolation::GetPointIndicesBoundingWedgeEdge(edgeId);
  int sn = 0;
  result->Points->SetPoint(sn,
    this->Points->GetPoint(
      this->PointIds->GetId(eidx[0])));
  econn[sn++] = eidx[0];
  result->Points->SetPoint(sn,
    this->Points->GetPoint(
      this->PointIds->GetId(eidx[1])));
  econn[sn++] = eidx[1];
  // Now add edge-interior points in axis order:
  int offset = 6;
  if (oi == 2)
    { // Edge is in t-direction.
    offset += 3 * (order[0] - 1 + order[1] - 1); // Skip edges in r-s plane.
    offset += (edgeId - 6) * (order[2] - 1); // Skip any previous t-axis edges.
    }
  else
    {
    // Edge is in r-s plane. Since we require order[0] == order[1], the offset is simple.
    offset += edgeId * (order[0] - 1);
    }
  for (int jj = 0; jj < order[oi >= 0 ? oi : 0] - 1; ++jj)
    {
    result->Points->SetPoint(sn,
      this->Points->GetPoint(
        this->PointIds->GetId(offset + jj)));
    econn[sn++] = offset + jj;
    }
  // FIXME: Should write to result->Points/PointIds directly?!
  result->vtkCell::Initialize(npts, &econn[0], result->Points);
  return result;
}

vtkCell* vtkLagrangeWedge::GetFace(int faceId)
{
  vtkCell* resultCell = NULL;
  if (faceId < 0 || faceId >= 5)
    {
    return resultCell;
    }

  // Do we need to flip the face to get an outward-pointing normal?
  bool flipFace = vtkLagrangeInterpolation::GetPointIndicesBoundingWedgeFace(faceId)[7] == 1 ? true : false;

  // Handle triangular and quadrilateral faces separately:
  const int* order = this->GetOrder();
  const int* corners = vtkLagrangeInterpolation::GetPointIndicesBoundingWedgeFace(faceId);
  int npts;

  int rsOrder = order[0]; // assert(order[0] == order[1]) for triangular faces
  int tOrder = order[2];

  if (faceId < 2)
    { // Triangular faces
    vtkLagrangeTriangle* result = this->BdyTri.GetPointer();
    resultCell = result;
    npts = (rsOrder + 1) * (rsOrder + 2) / 2;
    result->Points->SetNumberOfPoints(npts);
    result->PointIds->SetNumberOfIds(npts);

    // Add vertex DOFs to result
    int sn = 0;
    if (!flipFace)
      {
      for (int ii = 0; ii < 3; ++ii, ++sn)
        {
        result->Points->SetPoint(sn, this->Points->GetPoint(corners[ii]));
        result->PointIds->SetId(sn, this->PointIds->GetId(corners[ii]));
        }
      }
    else
      {
      for (int ii = 0; ii < 3; ++ii, ++sn)
        {
        result->Points->SetPoint((4 - sn) % 3, this->Points->GetPoint(corners[ii]));
        result->PointIds->SetId((4 - sn) % 3, this->PointIds->GetId(corners[ii]));
        }
      }

    // Add edge DOFs to result
    int offset;
    const int* faceEdges = vtkLagrangeInterpolation::GetEdgeIndicesBoundingWedgeFace(faceId);
    for (int ii = 0; ii < 3; ++ii)
      {
      if (!flipFace)
        {
        offset = 6 /* corner nodes */ + faceEdges[ii] * (rsOrder - 1);
        for (int jj = 0; jj < rsOrder - 1; ++jj, ++sn)
          {
          result->Points->SetPoint(sn, this->Points->GetPoint(offset + jj));
          result->PointIds->SetId(sn, this->PointIds->GetId(offset + jj));
          }
        }
      else
        {
        // Flip both the edge position among edges (ii => (4 - ii) % 4)
        // and the edge's node order (jj => order[pp] - jj - 1).

        offset = 6 /* corner nodes */ + faceEdges[(6 - ii) % 3] * (rsOrder - 1);
        for (int jj = 0; jj < rsOrder - 1; ++jj, ++sn)
          {
          result->Points->SetPoint(sn, this->Points->GetPoint(offset + rsOrder - jj - 2));
          result->PointIds->SetId(sn, this->PointIds->GetId(offset + rsOrder - jj - 2));
          }
        }
      }

    // Now add face DOF
    offset = 6 /* corners */ +  6 * (rsOrder - 1) /* xy-plane edges */ +  3 * (tOrder - 1) /* z-plane edges */;
    int nfdof = (rsOrder - 2) * (rsOrder - 1) / 2;
    if (faceId > 0)
      { // skip DOF for first triangular face
      offset += nfdof;
      }
    if (!flipFace)
      {
      for (int ii = 0; ii < nfdof; ++ii, ++sn)
        {
        result->Points->SetPoint(sn, this->Points->GetPoint(offset + ii));
        result->PointIds->SetId(sn, this->PointIds->GetId(offset + ii));
        }
      }
    else
      {
      int delta = rsOrder - 1;
      for (int jj = 0; jj < (rsOrder - 1); ++jj)
        {
        for (int ii = delta - 1; ii >= 0; --ii, ++sn)
          {
          result->Points->SetPoint(sn, this->Points->GetPoint(offset + ii + jj * delta));
          result->PointIds->SetId(sn, this->PointIds->GetId(offset + ii + jj * delta));
          }
        }
      }
    }
  else
    { // Quadrilateral faces
    vtkLagrangeQuadrilateral* result = this->BdyQuad.GetPointer();
    resultCell = result;
    npts = (rsOrder + 1) * (tOrder + 1);
    result->GetPoints()->SetNumberOfPoints(npts);
    result->GetPointIds()->SetNumberOfIds(npts);

    // Add vertex DOFs to result
    int sn = 0;
    if (!flipFace)
      {
      for (int ii = 0; ii < 4; ++ii, ++sn)
        {
        result->Points->SetPoint(sn, this->Points->GetPoint(corners[ii]));
        result->PointIds->SetId(sn, this->PointIds->GetId(corners[ii]));
        }
      }
    else
      {
      for (int ii = 0; ii < 4; ++ii, ++sn)
        {
        result->Points->SetPoint((5 - sn) % 4, this->Points->GetPoint(corners[ii]));
        result->PointIds->SetId((5 - sn) % 4, this->PointIds->GetId(corners[ii]));
        }
      }

    // Add edge DOFs to result
    int offset;
    const int* faceEdges = vtkLagrangeInterpolation::GetEdgeIndicesBoundingWedgeFace(faceId);
    for (int ii = 0; ii < 4; ++ii)
      {
      offset = 6;
      // If the face is being flipped, flip both the edge position
      // among edges (ii => (4 - ii) % 4) here *and*
      // the edge's node order (jj => order[pp] - jj - 1) below.
      int edgii = flipFace ? faceEdges[(4 - ii) % 4] : faceEdges[ii];
      int eOrder;
      if (edgii < 6)
        {
        offset += edgii * (rsOrder - 1);
        eOrder = rsOrder;
        }
      else
        {
        offset += 3 * 2 * (rsOrder - 1);
        offset += (edgii - 6) * (tOrder - 1);
        eOrder = tOrder;
        }
      if (!flipFace)
        {
        for (int jj = 0; jj < eOrder - 1; ++jj, ++sn)
          {
          result->Points->SetPoint(sn, this->Points->GetPoint(offset + jj));
          result->PointIds->SetId(sn, this->PointIds->GetId(offset + jj));
          }
        }
      else
        {
        for (int jj = 0; jj < eOrder - 1; ++jj, ++sn)
          {
          result->Points->SetPoint(sn, this->Points->GetPoint(offset + eOrder - jj - 2));
          result->PointIds->SetId(sn, this->PointIds->GetId(offset + eOrder - jj - 2));
          }
        }
      }

    // Now add face DOF
    // Skip corner and edge DOF:
    offset = 6 + 6 * (rsOrder - 1) + 3 * (tOrder - 1);

    // Skip DOF for other faces of wedge before this one:
    offset += 2 * (rsOrder - 2) * (rsOrder - 1) / 2; // Triangles
    int nfdof = (rsOrder - 1) * (tOrder - 1);
    int nQuadFace = faceId > 1 ? (faceId - 2) : 0;
    offset += nQuadFace * nfdof; // Quads

    if (!flipFace)
      {
      for (int ii = 0; ii < nfdof; ++ii, ++sn)
        {
        result->Points->SetPoint(sn, this->Points->GetPoint(offset + ii));
        result->PointIds->SetId(sn, this->PointIds->GetId(offset + ii));
        }
      }
    else
      {
      int delta = rsOrder - 1;
      for (int jj = 0; jj < tOrder - 1; ++jj)
        {
        for (int ii = delta - 1; ii >= 0; --ii, ++sn)
          {
          result->Points->SetPoint(sn, this->Points->GetPoint(offset + ii + jj * delta));
          result->PointIds->SetId(sn, this->PointIds->GetId(offset + ii + jj * delta));
          }
        }
      }
    }
  /*
  vtkIdType np2 = resultCell->Points->GetNumberOfPoints();
  std::cout << "Wedge face " << faceId << " has " << np2 << " points  ";
  for (vtkIdType xx = 0; xx < np2; ++xx)
    {
    vtkVector3d pp;
    resultCell->Points->GetPoint(xx, pp.GetData());
    std::cout << " " << resultCell->PointIds->GetId(xx) << pp;
    }
  std::cout << "\n";
  */
  return resultCell;
}

void vtkLagrangeWedge::Initialize()
{
}

int vtkLagrangeWedge::CellBoundary(
  int subId, double pcoords[3], vtkIdList* pts)
{
  // TODO: Fill me in!
  return 0;
}

int vtkLagrangeWedge::EvaluatePosition(
  double* x,
  double* closestPoint,
  int& subId,
  double pcoords[3],
  double& minDist2,
  double* weights)
{
  int result = 0;

  int dummySubId;
  double linearWeights[8];
  double tmpDist2;
  vtkVector3d params;
  vtkVector3d tmpClosestPt;

  minDist2 = VTK_DOUBLE_MAX;
  vtkIdType nhex = vtkLagrangeInterpolation::NumberOfIntervals<3>(this->GetOrder());
  for (int subCell = 0; subCell < nhex; ++subCell)
    {
    vtkWedge* approx = this->GetApproximateWedge(subCell, NULL, NULL);
    int stat = approx->EvaluatePosition(x, tmpClosestPt.GetData(), dummySubId, params.GetData(), tmpDist2, linearWeights);
    if (stat != -1 && tmpDist2 < minDist2)
      {
      result = stat;
      subId = subCell;
      minDist2 = tmpDist2;
      for (int ii = 0; ii < 3; ++ii)
        {
        pcoords[ii] = params[ii]; // We will translate the winning parameter values later.
        if (closestPoint)
          {
          closestPoint[ii] = tmpClosestPt[ii];
          }
        }
      }
    }

  if (result != -1)
    {
    this->TransformApproxToCellParams(subId, pcoords);
    if (closestPoint)
      {
      this->EvaluateLocation(dummySubId, pcoords, closestPoint, weights);
      }
    else
      {
      this->InterpolateFunctions(pcoords, weights);
      }
    }

  return result;
}

void vtkLagrangeWedge::EvaluateLocation(
  int& subId,
  double pcoords[3],
  double x[3], double* weights)
{
  subId = 0; // TODO: Should this be -1?
  this->InterpolateFunctions(pcoords, weights);

  double p[3];
  x[0] = x[1] = x[2] = 0.;
  vtkIdType nPoints = this->GetPoints()->GetNumberOfPoints();
  for (vtkIdType idx = 0; idx < nPoints; ++idx)
    {
    this->Points->GetPoint(idx, p);
    for (vtkIdType jdx = 0; jdx < 3; ++jdx)
      {
      x[jdx] += p[jdx] * weights[idx];
      }
    }
}

void vtkLagrangeWedge::Contour(
  double value,
  vtkDataArray* cellScalars,
  vtkIncrementalPointLocator* locator,
  vtkCellArray* verts,
  vtkCellArray* lines,
  vtkCellArray* polys,
  vtkPointData* inPd,
  vtkPointData* outPd,
  vtkCellData* inCd,
  vtkIdType cellId,
  vtkCellData* outCd)
{
  //std::cout << "Contour " << cellId << " with " << inPd->GetNumberOfTuples() << " tuples\n";
  this->PrepareApproxData(inPd, inCd, cellId, cellScalars); // writes to this->{CellScalars, ApproxPD, ApproxCD}
  vtkIdType nhex = vtkLagrangeInterpolation::NumberOfIntervals<3>(this->GetOrder());
  for (int i = 0; i < nhex; ++i)
    {
    vtkWedge* approx = this->GetApproximateWedge(i, this->CellScalars.GetPointer(), this->Scalars.GetPointer());
    approx->Contour(
      value, this->Scalars.GetPointer(), locator,
      verts, lines, polys, this->ApproxPD, outPd, this->ApproxCD, cellId, outCd);
    }
}

void vtkLagrangeWedge::Clip(
  double value,
  vtkDataArray* cellScalars,
  vtkIncrementalPointLocator* locator,
  vtkCellArray* polys,
  vtkPointData* inPd,
  vtkPointData* outPd,
  vtkCellData* inCd,
  vtkIdType cellId,
  vtkCellData* outCd,
  int insideOut)
{
  this->PrepareApproxData(inPd, inCd, cellId, cellScalars); // writes to this->{CellScalars, ApproxPD, ApproxCD}
  vtkIdType nhex = vtkLagrangeInterpolation::NumberOfIntervals<3>(this->GetOrder());
  for (int i = 0; i < nhex; ++i)
    {
    vtkWedge* approx = this->GetApproximateWedge(i, this->CellScalars.GetPointer(), this->Scalars.GetPointer());
    approx->Clip(
      value, this->Scalars.GetPointer(), locator,
      polys, this->ApproxPD, outPd, this->ApproxCD, cellId,
      outCd, insideOut);
    }
}

int vtkLagrangeWedge::IntersectWithLine(
  double* p1,
  double* p2,
  double tol,
  double& t,
  double* x,
  double* pcoords,
  int& subId)
{
  double tFirst = VTK_DOUBLE_MAX;
  bool intersection = false;
  vtkVector3d tmpX;
  vtkVector3d tmpP;
  int tmpId;
  this->GetOrder(); // Ensure Order is up to date.
  for (int ff = 0; ff < this->GetNumberOfFaces(); ++ff)
    {
    vtkCell* bdy = this->GetFace(ff);
    if (bdy->IntersectWithLine(p1, p2, tol, t, tmpX.GetData(), tmpP.GetData(), tmpId))
      {
      intersection = true;
      if (t < tFirst)
        {
        tFirst = t;
        for (int ii = 0; ii < 3; ++ii)
          {
          x[ii] = tmpX[ii];
          pcoords[ii] = tmpP[ii]; // Translate this after we're sure it's the closest hit.
          subId = ff;
          }
        }
      }
    }
  if (intersection)
    {
    this->TransformFaceToCellParams(subId, pcoords);
    }
  return intersection ? 1 : 0;
}

int vtkLagrangeWedge::Triangulate(
  int vtkNotUsed(index),
  vtkIdList* ptIds,
  vtkPoints* pts)
{
  ptIds->Reset();
  pts->Reset();

  vtkIdType nhex = vtkLagrangeInterpolation::NumberOfIntervals<3>(this->GetOrder());
  vtkVector3i ijk;
  for (int i = 0; i < nhex; ++i)
    {
    vtkWedge* approx = this->GetApproximateWedge(i);
    this->SubCellCoordinatesFromId(ijk, i);
    if (approx->Triangulate(
        (ijk[0] + ijk[1] + ijk[2]) % 2,
        this->TmpIds.GetPointer(),
        this->TmpPts.GetPointer()))
      {
      // Sigh. Triangulate methods all reset their points/ids
      // so we must copy them to our output.
      vtkIdType np = this->TmpPts->GetNumberOfPoints();
      vtkIdType ni = this->TmpIds->GetNumberOfIds();
      vtkIdType offset = pts->GetNumberOfPoints();
      for (vtkIdType ii = 0; ii < np; ++ii)
        {
        pts->InsertNextPoint(this->TmpPts->GetPoint(ii));
        }
      for (vtkIdType ii = 0; ii < ni; ++ii)
        {
        ptIds->InsertNextId(this->TmpIds->GetId(ii) + offset);
        }
      }
    }
  return 1;
}

void vtkLagrangeWedge::Derivatives(
  int vtkNotUsed(subId),
  double vtkNotUsed(pcoords)[3],
  double* vtkNotUsed(values),
  int vtkNotUsed(dim),
  double* vtkNotUsed(derivs))
{
  // TODO: Fill me in?
  return;
}

double* vtkLagrangeWedge::GetParametricCoords()
{
  if (!this->PointParametricCoordinates)
    {
    this->PointParametricCoordinates = vtkSmartPointer<vtkPoints>::New();
    this->PointParametricCoordinates->SetDataTypeToDouble();
    }

  // Ensure Order is up-to-date and check that current point size matches:
  if (static_cast<int>(this->PointParametricCoordinates->GetNumberOfPoints()) != this->GetOrder(3))
    {
    this->PointParametricCoordinates->Initialize();
    vtkLagrangeInterpolation::AppendWedgeCollocationPoints(
      this->PointParametricCoordinates, this->Order);
    }

  return
    vtkDoubleArray::SafeDownCast(
      this->PointParametricCoordinates->GetData())->GetPointer(0);
}

double vtkLagrangeWedge::GetParametricDistance(double pcoords[3])
{
  double pDist, pDistMax = 0.0;

  for (int ii = 0; ii < 3; ++ii)
    {
    pDist =
      (pcoords[ii] < 0. ? -pcoords[ii] :
       (pcoords[ii] > 1. ? pcoords[ii] - 1. :
        0.));
    if (pDist > pDistMax)
      {
      pDistMax = pDist;
      }
    }

  return pDistMax;
}

const int* vtkLagrangeWedge::GetOrder()
{
  // FIXME: The interpolation routines can handle different order along each axis
  //   but we cannot infer the order from the number of points in that case.
  //   This method currrently assumes hexahedra are of the same order on each axis.
  //   We populate the Order array for use with the interpolation class.
  vtkIdType npts = this->Points->GetNumberOfPoints();
  if (this->Order[3] != npts)
    {
    double n = static_cast<double>(npts);
    static const double third(1./3.);
    static const double ninth(1./9.);
    static const double twentyseventh(1./27.);
    double term =
      pow(third * sqrt(third) * sqrt((27.0 * n - 2.0) * n) + n - twentyseventh, third);
    double order = term + ninth / term - 4 * third;

    double proximity = fabs(order - round(order));
    if (proximity > 1e-12)
      {
      vtkErrorMacro(
        "Incorrect number of points "
        <<  npts << " for wedge; nearest order "
        << order << " not close enough to integer. Was within "
        << proximity << " but expected 1e-12 or smaller."
      );
      order = -1.;
      }

    int iorder = static_cast<int>(round(order));
    for (int i = 0; i < 3; ++i)
      {
      this->Order[i] = iorder;
      }
    this->Order[3] = static_cast<int>(npts);
    this->CellScalars->SetNumberOfTuples(npts);
    }
  return this->Order;
}

void vtkLagrangeWedge::InterpolateFunctions(
  double pcoords[3], double* weights)
{
  vtkLagrangeInterpolation::TensorShapeFunctions<3>(this->GetOrder(), pcoords, weights);
}

void vtkLagrangeWedge::InterpolateDerivs(
  double pcoords[3], double* derivs)
{
  vtkLagrangeInterpolation::TensorShapeDerivatives<3>(this->GetOrder(), pcoords, derivs);
}

/// Return a linear hexahedron used to approximate a region of the nonlinear hex.
vtkWedge* vtkLagrangeWedge::GetApprox()
{
  if (!this->Approx)
    {
    this->Approx = vtkSmartPointer<vtkWedge>::New();
    this->ApproxPD = vtkSmartPointer<vtkPointData>::New();
    this->ApproxCD = vtkSmartPointer<vtkCellData>::New();
    }
  return this->Approx.GetPointer();
}

/**\brief Prepare point data for use by linear approximating-elements.
  *
  * This copies the point data for the current cell into a new point-data
  * object so that the point ids and scalar ids can match.
  */
void vtkLagrangeWedge::PrepareApproxData(vtkPointData* pd, vtkCellData* cd, vtkIdType cellId, vtkDataArray* cellScalars)
{
  this->GetApprox(); // Ensure this->Approx{PD,CD} are non-NULL.
  this->GetOrder(); // Ensure the order has been updated to match this element.
  vtkIdType npts = this->Order[3];
  vtkIdType nele = this->Order[0] * this->Order[1] * this->Order[2];
  this->ApproxPD->Initialize();
  this->ApproxCD->Initialize();
  this->ApproxPD->CopyAllOn();
  this->ApproxCD->CopyAllOn();
  this->ApproxPD->CopyAllocate(pd, npts);
  this->ApproxCD->CopyAllocate(cd, nele);
  for (int pp = 0; pp < npts; ++pp)
    {
    this->ApproxPD->CopyData(pd, this->PointIds->GetId(pp), pp);
    this->CellScalars->SetValue(pp, cellScalars->GetTuple1(pp));
    }
  for (int ee = 0; ee < nele; ++ee)
    {
    this->ApproxCD->CopyData(cd, cellId, ee);
    }
}

/**\brief Populate the linear hex returned by GetApprox() with point-data from one voxel-like intervals of this cell.
  *
  * Ensure that you have called GetOrder() before calling this method
  * so that this->Order is up to date. This method does no checking
  * before using it to map connectivity-array offsets.
  */
vtkWedge* vtkLagrangeWedge::GetApproximateWedge(
  int subId, vtkDataArray* scalarsIn, vtkDataArray* scalarsOut)
{
  vtkWedge* approx = this->GetApprox();
  bool doScalars = (scalarsIn && scalarsOut);
  if (doScalars)
    {
    scalarsOut->SetNumberOfTuples(8);
    }
  int i, j, k;
  this->SubCellCoordinatesFromId(i, j, k, subId);
  // Get the point coordinates (and optionally scalars) for each of the 8 corners
  // in the approximating hexahedron spanned by (i, i+1) x (j, j+1) x (k, k+1):
  //vtkIdType aconn[8]; // = {0, 1, 2, 3, 4, 5, 6, 7};
  //std::cout << "Wedgeproximate " << subId << "\n";
  for (int ic = 0; ic < 8; ++ic)
    {
    int corner = this->PointIndexFromIJK(
      i + (((ic + 1) / 2) % 2 ? 1 : 0),
      j + ((ic / 2) % 2 ? 1 : 0),
      k + (ic / 4 ? 1 : 0));
    vtkVector3d cp;
    this->Points->GetPoint(corner, cp.GetData());
    //std::cout << "    corner " << ic << " @ " << corner << ": " << cp << "\n";
    //aconn[ic] = this->PointIds->GetId(corner);
    // aconn[ic] = corner;
    approx->PointIds->SetId(ic, corner);
    approx->Points->SetPoint(ic, cp.GetData()); // this->Points->GetPoint(corner));
    if (doScalars)
      {
      //std::cout << "    corner " << ic << " @ " << corner << ": " << scalarsIn->GetTuple(corner)[0] << "\n";
      scalarsOut->SetTuple(ic,
        scalarsIn->GetTuple(
          corner));
      }
    }
  //approx->Initialize(8, &aconn[0], this->Points);
  //approx->Initialize(8, &aconn[0], this->ApproxPts);
  return approx;
}

/// A convenience method; see the overloaded variant for more information.
bool vtkLagrangeWedge::SubCellCoordinatesFromId(vtkVector3i& ijk, int subId)
{
  return this->SubCellCoordinatesFromId(ijk[0], ijk[1], ijk[2], subId);
}

/**\brief Given an integer specifying an approximating linear hex, compute its IJK coordinate-position in this cell.
 *
 * The \a subId specifies the lower-, left-, front-most vertex of the approximating hex.
 * This sets the ijk coordinates of that point.
 *
 * You must have called this->GetOrder() **before** invoking this method so that the order will be up to date.
 */
bool vtkLagrangeWedge::SubCellCoordinatesFromId(int& i, int& j, int& k, int subId)
{
  if (subId < 0)
    {
    return false;
    }

  int layerSize = this->Order[0] * this->Order[1];
  i = subId % this->Order[0];
  j = (subId / this->Order[0]) % this->Order[1];
  k = subId / layerSize;
  return true; // TODO: detect more invalid subId values
}

/**\brief Given (i,j,k) coordinates within the Lagrange hex, return an offset into the local connectivity (PointIds) array.
  *
  * Ensure that you have called GetOrder() before calling this method
  * so that this->Order is up to date. This method does no checking
  * before using it to map connectivity-array offsets.
  */
int vtkLagrangeWedge::PointIndexFromIJK(int i, int j, int k)
{
  bool ibdy = (i == 0 || i == this->Order[0]);
  bool jbdy = (j == 0 || j == this->Order[1]);
  bool kbdy = (k == 0 || k == this->Order[2]);
  // How many boundaries do we lie on at once?
  int nbdy = (ibdy ? 1 : 0) + (jbdy ? 1 : 0) + (kbdy ? 1 : 0);

  if (nbdy == 3) // Vertex DOF
    { // ijk is a corner node. Return the proper index (somewhere in [0,7]):
    return (i ? (j ? 2 : 1) : (j ? 3 : 0)) + (k ? 4 : 0);
    }

  int offset = 8;
  if (nbdy == 2) // Edge DOF
    {
    if (!ibdy)
      { // On i axis
      return (i - 1) +
        (j ? this->Order[0] - 1 + this->Order[1] - 1 : 0) +
        (k ? 2 * (this->Order[0] - 1 + this->Order[1] - 1) : 0) +
        offset;
      }
    if (!jbdy)
      { // On j axis
      return (j - 1) +
        (i ? this->Order[0] - 1 : 2 * (this->Order[0] - 1) + this->Order[1] - 1) +
        (k ? 2 * (this->Order[0] - 1 + this->Order[1] - 1) : 0) +
        offset;
      }
    // !kbdy, On k axis
    offset += 4 * (this->Order[0] - 1) + 4 * (this->Order[1] - 1);
    return (k - 1) + (this->Order[2] - 1) * (i ? (j ? 3 : 1) : (j ? 2 : 0)) + offset;
    }

  offset += 4 * (this->Order[0] - 1 + this->Order[1] - 1 + this->Order[2] - 1);
  if (nbdy == 1) // Face DOF
    {
    if (ibdy) // On i-normal face
      {
      return (j - 1) + ((this->Order[1] - 1) * (k - 1)) + (i ? (this->Order[1] - 1) * (this->Order[2] - 1) : 0) + offset;
      }
    offset += 2 * (this->Order[1] - 1) * (this->Order[2] - 1);
    if (jbdy) // On j-normal face
      {
      return (i - 1) + ((this->Order[0] - 1) * (k - 1)) + (j ? (this->Order[2] - 1) * (this->Order[0] - 1) : 0) + offset;
      }
    offset += 2 * (this->Order[2] - 1) * (this->Order[0] - 1);
    // kbdy, On k-normal face
    return (i - 1) + ((this->Order[0] - 1) * (j - 1)) + (k ? (this->Order[0] - 1) * (this->Order[1] - 1) : 0) + offset;
    }

  // nbdy == 0: Body DOF
  offset += 2 * (
    (this->Order[1] - 1) * (this->Order[2] - 1) +
    (this->Order[2] - 1) * (this->Order[0] - 1) +
    (this->Order[0] - 1) * (this->Order[1] - 1));
  return offset +
    (i - 1) + (this->Order[0] - 1) * (
      (j - 1) + (this->Order[1] - 1) * (
        (k - 1)));
}

/**\brief Given the index, \a subCell, of a linear approximating-hex, translate pcoords from that hex into this nonlinear hex.
  *
  * You must call this->GetOrder() **before** invoking this method as it assumes
  * the order is up to date.
  */
bool vtkLagrangeWedge::TransformApproxToCellParams(int subCell, double* pcoords)
{
  vtkVector3i ijk;
  if (!this->SubCellCoordinatesFromId(ijk, subCell))
    {
    return false;
    }
  for (int pp = 0; pp < 3; ++pp)
    {
    pcoords[pp] = (pcoords[pp] + ijk[pp]) / this->Order[pp];
    }
  return true;
}

/**\brief Given the index, \a subCell, of a linear approximating-hex, translate pcoords from that hex into this nonlinear hex.
  *
  * You must call this->GetOrder() **before** invoking this method as it assumes
  * the order is up to date.
  */
bool vtkLagrangeWedge::TransformFaceToCellParams(int bdyFace, double* pcoords)
{
  if (bdyFace < 0 || bdyFace >= 6)
    {
    return false;
    }

  vtkVector2i faceParams = vtkLagrangeInterpolation::GetVaryingParametersOfWedgeFace(bdyFace);
  vtkVector3d tmp(pcoords);
  int pp;
  for (pp = 0; pp < 2; ++pp)
    {
    pcoords[faceParams[pp]] = tmp[pp];
    }
  if (bdyFace % 2 == 1)
    {
    // Flip first parametric axis of "positive" faces to compensate for GetFace,
    // which flips odd faces to obtain inward-pointing normals for each boundary.
    pcoords[faceParams[0]] = 1. - pcoords[faceParams[0]];
    }
  pp = vtkLagrangeInterpolation::GetFixedParameterOfWedgeFace(bdyFace);
  pcoords[pp] = (bdyFace % 2 == 0 ? 0.0 : 1.0);
  return true;
}
