/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangeTriangle.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkLagrangeTriangle - cell represents an n-th order isoparametric triangle
// .SECTION Description
// vtkLagrangeTriangle is a concrete implementation of vtkNonLinearCell to
// represent a two-dimensional, n-th order, isoparametric triangle. The order
// of The cell is determined by the number of points associated with the cell:
// a canonical order n triangle has (n+1)*(n+2)/2 points, defined in the
// following manner: point ids (0 to 2) are the three vertices; the next 3*(n-1)
// point ids represent the respective edge points in between vertices 0 and 1,
// 1 and 2 and 2 and 0; and the remaining points describe the face points that
// comprise the order (n-3) triangle that remains by masking the original
// triangle's vertices and edge points. The interpolation is the standard
// finite element, n-th order isoparametric shape function, adapted from P.
// Silvester, "High-Order Polynomial Triangular Finite Elements for Potential
// Problems". Int. J. Engng Sci. Vol. 7, pp. 849-861. Pergamon Press, 1969.

// .SECTION See Also
// vtkLagrangeTetra

#ifndef vtkLagrangeTriangle_h
#define vtkLagrangeTriangle_h

#include "vtkCommonDataModelModule.h" // For export macro
#include "vtkNonLinearCell.h"

#define VTK_LAGRANGE_TRIANGLE_MAX_ORDER 6

#define MAX_POINTS ((VTK_LAGRANGE_TRIANGLE_MAX_ORDER + 1) *     \
                    (VTK_LAGRANGE_TRIANGLE_MAX_ORDER + 2)/2)

#define MAX_SUBTRIANGLES (VTK_LAGRANGE_TRIANGLE_MAX_ORDER *     \
                          VTK_LAGRANGE_TRIANGLE_MAX_ORDER)

class vtkDoubleArray;
class vtkLagrangeCurve;
class vtkTriangle;

class VTKCOMMONDATAMODEL_EXPORT vtkLagrangeTriangle : public vtkNonLinearCell
{
public:
  static vtkLagrangeTriangle *New();
  vtkTypeMacro(vtkLagrangeTriangle,vtkNonLinearCell);
  void PrintSelf(ostream& os, vtkIndent indent);

  int GetCellType() { return VTK_LAGRANGE_TRIANGLE; }
  int GetCellDimension() { return 2; }
  int RequiresInitialization() { return 1; }
  int GetNumberOfEdges() { return 3; }
  int GetNumberOfFaces() { return 0; }
  vtkCell *GetEdge(int edgeId);
  vtkCell *GetFace(int) { return 0; }

  void Initialize();

  static int MaximumOrder() { return VTK_LAGRANGE_TRIANGLE_MAX_ORDER; }
  static int MaximumNumberOfPoints()
  {
    return ((vtkLagrangeTriangle::MaximumOrder() + 1) *
            (vtkLagrangeTriangle::MaximumOrder() + 2)/2);
  }

  int CellBoundary(int subId, double pcoords[3], vtkIdList *pts);
  int EvaluatePosition(double x[3], double* closestPoint,
                       int& subId, double pcoords[3],
                       double& dist2, double *weights);
  void EvaluateLocation(int& subId, double pcoords[3], double x[3],
                        double *weights);
  void Contour(double value, vtkDataArray *cellScalars,
               vtkIncrementalPointLocator *locator, vtkCellArray *verts,
               vtkCellArray *lines, vtkCellArray *polys,
               vtkPointData *inPd, vtkPointData *outPd,
               vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd);
  void Clip(double value, vtkDataArray *cellScalars,
            vtkIncrementalPointLocator *locator, vtkCellArray *polys,
            vtkPointData *inPd, vtkPointData *outPd,
            vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd,
            int insideOut);
  int IntersectWithLine(double p1[3], double p2[3], double tol, double& t,
                        double x[3], double pcoords[3], int& subId);
  int Triangulate(int index, vtkIdList *ptIds, vtkPoints *pts);
  void JacobianInverse(double pcoords[3], double** inverse, double* derivs);
  void Derivatives(int subId, double pcoords[3], double *values,
                   int dim, double *derivs);
  double* GetParametricCoords();
  static void ComputeParametricCoords(double*,vtkIdType);

  int GetParametricCenter(double pcoords[3]);
  double GetParametricDistance(double pcoords[3]);

  void InterpolateFunctions(double pcoords[3], double* weights);
  void InterpolateDerivs(double pcoords[3], double* derivs);

  vtkIdType GetOrder() const { return this->Order; }
  vtkIdType ComputeOrder();

  void ToBarycentricIndex(vtkIdType index, vtkIdType* bindex);
  vtkIdType ToIndex(const vtkIdType* bindex);

  static void BarycentricIndex(vtkIdType index, vtkIdType* bindex,
                               vtkIdType order);
  static vtkIdType Index(const vtkIdType* bindex, vtkIdType order);

  static double eta(vtkIdType n, vtkIdType chi, double sigma);
  static double d_eta(vtkIdType n, vtkIdType chi,double sigma);

protected:
  vtkLagrangeTriangle();
  ~vtkLagrangeTriangle();

  vtkIdType GetNumberOfSubtriangles() const {return this->NumberOfSubtriangles;}
  vtkIdType ComputeNumberOfSubtriangles();

  // Description:
  // Given the index of the subtriangle, compute the barycentric indices of
  // the subtriangle's vertices.
  void SubtriangleBarycentricPointIndices(vtkIdType cellIndex,
                                          vtkIdType (&pointBIndices)[3][3]);

  vtkLagrangeCurve *Edge;
  vtkTriangle *Face;
  vtkDoubleArray *Scalars; //used to avoid New/Delete in contouring/clipping
  vtkIdType Order;
  vtkIdType NumberOfSubtriangles;

  // The usual convention for parametric coordinates is to return a pointer to
  // a static array defining the point positions. Because this cell is dynamic,
  // this array must be created at runtime to reflect the proper order of the
  // cell.
  double* ParametricCoordinates;

  vtkIdType EdgeIds[VTK_LAGRANGE_TRIANGLE_MAX_ORDER + 1];
  vtkIdType BarycentricIndexMap[3*MAX_POINTS];
  vtkIdType IndexMap[(VTK_LAGRANGE_TRIANGLE_MAX_ORDER + 1) *
                     (VTK_LAGRANGE_TRIANGLE_MAX_ORDER + 1)];
  vtkIdType SubtriangleIndexMap[9*MAX_SUBTRIANGLES];

private:
  vtkLagrangeTriangle(const vtkLagrangeTriangle&);  // Not implemented.
  void operator=(const vtkLagrangeTriangle&);  // Not implemented.
};

#undef MAX_POINTS
#undef MAX_SUBTRIANGLES

#endif
