/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangeTetra.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkLagrangeTetra - cell represents an n-th order isoparametric tetrahedron
// .SECTION Description
// vtkLagrangeTetra is a concrete implementation of vtkNonLinearCell to
// represent a three-dimensional, n-th order, isoparametric tetrahedron. The
// order of The cell is determined by the number of points associated with the
// cell: a canonical order n tetrahedron has (n+1)*(n+2)*(n+3)/6 points, defined
// in the following manner: point ids (0 to 3) are the four vertices; the next
// 6*(n-1) point ids represent the respective edge points in between vertices
// (0-1), (1,2), (2,0), (0,3), (1,3), (2,3); the next 4*(n-2)*(n-1) points
// describe the face points that for faces (0-1-3), (1-2-3), (2-0-3) and
// (0-2-1); the remaining point ids comprise the order (n-4) tetrahedron
// constructed by masking the outer vertex, edge and face points. The
// interpolation is the standard finite element, n-th order isoparametric shape
// function, adapted from P. Silvester, "High-Order Polynomial Triangular
// Finite Elements for Potential Problems". Int. J. Engng Sci. Vol. 7, pp.
// 849-861. Pergamon Press, 1969.

// .SECTION See Also
// vtkLagrangeTriangle

#ifndef vtkLagrangeTetra_h
#define vtkLagrangeTetra_h

#include "vtkCommonDataModelModule.h" // For export macro
#include "vtkNonLinearCell.h"

#define VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER 6

#define MAX_POINTS ((VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 1) *  \
                    (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 2) *  \
                    (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 3)/6)

#define MAX_SUBTETRAHEDRA (((VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER - 2) *  \
                            (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER - 1) *  \
                            (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER)) +     \
                           4*((VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER - 1) * \
                              (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER) *    \
                              (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 1)) + \
                           ((VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER) *      \
                            (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 1) *  \
                            (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 2))/6)

class vtkTetra;
class vtkLagrangeCurve;
class vtkLagrangeTriangle;
class vtkDoubleArray;

class VTKCOMMONDATAMODEL_EXPORT vtkLagrangeTetra : public vtkNonLinearCell
{
public:
  static vtkLagrangeTetra *New();
  vtkTypeMacro(vtkLagrangeTetra,vtkNonLinearCell);
  void PrintSelf(ostream& os, vtkIndent indent);

  int GetCellType() { return VTK_LAGRANGE_TETRAHEDRON; }
  int GetCellDimension() { return 3; }
  int RequiresInitialization() { return 1; }
  int GetNumberOfEdges() { return 6; }
  int GetNumberOfFaces() { return 4; }
  vtkCell *GetEdge(int edgeId);
  vtkCell *GetFace(int faceId);

  void Initialize();

  static int MaximumOrder() { return VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER; }
  static int MaximumNumberOfPoints()
  {
    return ((vtkLagrangeTetra::MaximumOrder() + 1) *
            (vtkLagrangeTetra::MaximumOrder() + 2)/2);
  }
  int CellBoundary(int subId, double pcoords[3], vtkIdList *pts);
  int EvaluatePosition(double x[3], double* closestPoint,
                       int& subId, double pcoords[3],
                       double& dist2, double *weights);
  void EvaluateLocation(int& subId, double pcoords[3], double x[3],
                        double *weights);
  void Contour(double value, vtkDataArray *cellScalars,
               vtkIncrementalPointLocator *locator, vtkCellArray *verts,
               vtkCellArray *lines, vtkCellArray *polys,
               vtkPointData *inPd, vtkPointData *outPd,
               vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd);
  void Clip(double value, vtkDataArray *cellScalars,
            vtkIncrementalPointLocator *locator, vtkCellArray *polys,
            vtkPointData *inPd, vtkPointData *outPd,
            vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd,
            int insideOut);
  int IntersectWithLine(double p1[3], double p2[3], double tol, double& t,
                        double x[3], double pcoords[3], int& subId);
  int Triangulate(int index, vtkIdList *ptIds, vtkPoints *pts);
  void JacobianInverse(double pcoords[3], double** inverse, double* derivs);
  void Derivatives(int subId, double pcoords[3], double *values,
                   int dim, double *derivs);
  double* GetParametricCoords();

  int GetParametricCenter(double pcoords[3]);
  double GetParametricDistance(double pcoords[3]);

  void InterpolateFunctions(double pcoords[3], double* weights);
  void InterpolateDerivs(double pcoords[3], double* derivs);

  vtkIdType GetOrder() const { return this->Order; }
  vtkIdType ComputeOrder();

  void ToBarycentricIndex(vtkIdType index, vtkIdType* bindex);
  vtkIdType ToIndex(const vtkIdType* bindex);

  static void BarycentricIndex(vtkIdType index, vtkIdType* bindex,
                                      vtkIdType order);
  static vtkIdType Index(const vtkIdType* bindex, vtkIdType order);

protected:
  vtkLagrangeTetra();
  ~vtkLagrangeTetra();

  vtkIdType GetNumberOfSubtetras() const { return this->NumberOfSubtetras; }
  vtkIdType ComputeNumberOfSubtetras();

  // Description:
  // Given the index of the subtriangle, compute the barycentric indices of
  // the subtriangle's vertices.
  void SubtetraBarycentricPointIndices(vtkIdType cellIndex,
                                       vtkIdType (&pointBIndices)[4][4]);
  void TetraFromOctahedron(vtkIdType cellIndex,
                           const vtkIdType (&octBIndices)[6][4],
                           vtkIdType (&tetraBIndices)[4][4]);

  vtkLagrangeCurve *Edge;
  vtkLagrangeTriangle *Face;
  vtkTetra *Tetra;
  vtkDoubleArray *Scalars; //used to avoid New/Delete in contouring/clipping
  vtkIdType Order;
  vtkIdType NumberOfSubtetras;
  double* ParametricCoordinates;

  vtkIdType EdgeIds[VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 1];
  vtkIdType BarycentricIndexMap[4*MAX_POINTS];
  vtkIdType IndexMap[(VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 1) *
                     (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 1) *
                     (VTK_LAGRANGE_TETRAHEDRON_MAX_ORDER + 1)];
  vtkIdType SubtetraIndexMap[16*MAX_SUBTETRAHEDRA];

private:
  vtkLagrangeTetra(const vtkLagrangeTetra&);  // Not implemented.
  void operator=(const vtkLagrangeTetra&);  // Not implemented.
};

#undef MAX_POINTS
#undef MAX_SUBTETRAHEDRA

#endif
