#include "vtkLagrangeInterpolation.h"

#include "vtkLagrangeTriangle.h"
#include "vtkMath.h"
#include "vtkNew.h"
#include "vtkPoints.h"
#include "vtkVector.h"
#include "vtkDoubleArray.h"
#include "vtkVectorOperators.h"
#include "vtkObjectFactory.h"

vtkStandardNewMacro(vtkLagrangeInterpolation);

// -----------------------------------------------------------------------------
static const double hexCorner[8][3] = {
    {  0.,  0.,  0. },
    { +1.,  0.,  0. },
    { +1., +1.,  0. },
    {  0., +1.,  0. },
    {  0.,  0., +1. },
    { +1.,  0., +1. },
    { +1., +1., +1. },
    {  0., +1., +1. }
};

// Edges and faces are always oriented along quad/hexahedron axes,
// not any "cell-local" direction (i.e., faces do not all
// have inward-pointing normals).
static const int hexEdgeCorners[12][5] = {
  // e0 e1    varying-  fixed- parametric coordinate(s)
    { 0, 1,   0,        1, 2 },
    { 1, 2,   1,        0, 2 },
    { 3, 2,   0,        1, 2 },
    { 0, 3,   1,        0, 2 },
    { 4, 5,   0,        1, 2 },
    { 5, 6,   1,        0, 2 },
    { 7, 6,   0,        1, 2 },
    { 4, 7,   1,        0, 2 },
    { 0, 4,   2,        0, 1 },
    { 1, 5,   2,        0, 1 },
    { 3, 7,   2,        0, 1 },
    { 2, 6,   2,        0, 1 }
};

static const int hexFaceCorners[6][7] = {
  // c0 c1 c2 c3    varying- fixed-parametric coordinate(s)
    { 0, 3, 7, 4,   1, 2,    0 },
    { 1, 2, 6, 5,   1, 2,    0 },
    { 0, 1, 5, 4,   0, 2,    1 },
    { 3, 2, 6, 7,   0, 2,    1 },
    { 0, 1, 2, 3,   0, 1,    2 },
    { 4, 5, 6, 7,   0, 1,    2 },
};

static const int hexFaceEdges[6][4] = {
   // e0  e1  e2  e3
    {  3, 10,  7,  8 },
    {  1, 11,  5,  9 },
    {  0,  9,  4,  8 },
    {  2, 11,  6, 10 },
    {  0,  1,  2,  3 },
    {  4,  5,  6,  7 },
};
// -----------------------------------------------------------------------------
static const double wedgeCorner[6][3] = {
    {  0.,  0.,  0. },
    { +1.,  0.,  0. },
    {  0., +1.,  0. },
    {  0.,  0., +1. },
    { +1.,  0., +1. },
    {  0., +1., +1. }
};

// Edges and faces are always oriented along quad/hexahedron axes,
// not any "cell-local" direction (i.e., faces do not all
// have inward-pointing normals).
static const int wedgeEdgeCorners[9][5] = {
  // e0 e1    varying-  fixed- parametric coordinate(s)
    { 0, 1,   0,        1, 2 },
    { 1, 2,  -1,       -1, 2 },
    { 2, 0,   1,        0, 2 },

    { 3, 4,   0,        1, 2 },
    { 4, 5,  -1,       -1, 2 },
    { 5, 3,   1,        0, 2 },

    { 0, 3,   2,        0, 1 },
    { 1, 4,   2,        0, 1 },
    { 2, 5,   2,        0, 1 }
};

static const int wedgeFaceCorners[5][8] = {
   // c0  c1  c2  c3   varying-  fixed-param. coordinate(s)  orientation (0 is negative, 1 is positive)
    { 0,  1,  2, -1,    0,  1,     2,                         0 },
    { 3,  4,  5, -1,    0,  1,     2,                         1 },

    { 0,  1,  4,  3,    0,  2,     1,                         1 },
    { 1,  2,  5,  4,   -1, -1,    -1,                         1 },
    { 0,  2,  5,  3,    1,  2,     0,                         0 },
};

static const int wedgeFaceEdges[5][5] = {
   // e0  e1  e2  e3    orientation (<- 1 when implied normal points in, not out)
    {  0,  1,  2, -1,   0 },
    {  3,  4,  5, -1,   1 },

    {  0,  7,  3,  6,   0 },
    {  1,  8,  4,  7,   0 },
    {  2,  8,  5,  6,   0 },
};
// -----------------------------------------------------------------------------

vtkLagrangeInterpolation::vtkLagrangeInterpolation()
{
  int maxOrder[3] = {
    MaxDegree,
    MaxDegree,
    MaxDegree
  };
  vtkLagrangeInterpolation::PrepareForOrder(maxOrder);
}

vtkLagrangeInterpolation::~vtkLagrangeInterpolation()
{
}

void vtkLagrangeInterpolation::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

/// Evaluate 1-D shape functions for the given \a order at the given \a pcoord (in [0,1]).
void vtkLagrangeInterpolation::EvaluateShapeFunctions(int order, double pcoord, double* shape)
{
  int j, k;
  double v = order * pcoord;
  for (j = 0; j <= order; ++j)
    {
    shape[j] = 1.;
    for (k = 0; k <= order; ++k)
      {
      if (j != k)
        { // FIXME: (j - k) could be a register decremented inside the k loop:
        // and even better: normalization 1./(j - k) could be pre-computed and stored
        // somehow for each order that is actually used, removing division operations.
        shape[j] *= (v - k) / (j - k);
        }
      }
    }
}

/// Evaluate 1-D shape functions and their derivatives for the given \a order at the given \a pcoord (in [0,1]).
void vtkLagrangeInterpolation::EvaluateShapeAndGradient(int order, double pcoord, double* shape, double* deriv)
{
  int j, k, q;
  double dtmp;
  double v = order * pcoord;
  for (j = 0; j <= order; ++j)
    {
    //std::cout << "ShapeDeriv j = " << j << "  v = " << v << "\n";
    shape[j] = 1.;
    deriv[j] = 0.;
    for (k = 0; k <= order; ++k)
      {
      if (j != k)
        { // FIXME: (j - k) could be a register decremented inside the k loop:
        // and even better: normalization could be pre-computed and stored
        // somehow for each order that is actually used.
        shape[j] *= (v - k) / (j - k);

        // Now compute the derivative of shape[j]; we use the differentiation
        // rule d/dx(a * b) = a * d/dx(b) + b * d/dx(a) instead of faster
        // methods because it keeps the truncation error low(er):
        dtmp = 1.;
        //std::cout << "           k = " << k;
        for (q = 0; q <= order; ++q)
          {
          if (q == j)
            {
            continue;
            }
          dtmp *= (q == k ? 1. : (v - q)) / (j - q);
          //std::cout << "  q " << q << " dtmp *= " << ((q == k ? 1. : (v - q)) / (j - q));
          }
        //std::cout << "        dtmp = " << dtmp << "\n";
        deriv[j] += dtmp;
        }
      }
    }
}

/// Quadrilateral shape function computation
void vtkLagrangeInterpolation::Tensor2ShapeFunctions(const int order[2], const double pcoords[3], double* shape)
{
  // FIXME: Eventually needs to be varying length.
  double ll[2][vtkLagrangeInterpolation::MaxDegree + 1];
  int i, j;

  for (i = 0; i < 2; ++i)
    {
    vtkLagrangeInterpolation::EvaluateShapeFunctions(order[i], pcoords[i], ll[i]);
    }

  int sn = 0;

  // Corners
  shape[sn++] = ll[0][0]        * ll[1][0];
  shape[sn++] = ll[0][order[0]] * ll[1][0];
  shape[sn++] = ll[0][order[0]] * ll[1][order[1]];
  shape[sn++] = ll[0][0]        * ll[1][order[1]];

  int sn1 = sn + order[0] + order[1] - 2;
  for (i = 1; i < order[0]; ++i)
    {
    //cout << sn << ", " << sn1 << "\n";
    shape[sn++ ] = ll[0][i] * ll[1][0];               // Edge 0-1
    shape[sn1++] = ll[0][i] * ll[1][order[1]];        // Edge 2-3
    }

  for (i = 1; i < order[1]; ++i)
    {
    //cout << sn << ", " << sn1 << "\n";
    shape[sn++ ] = ll[0][order[0]] * ll[1][i];        // Edge 1-2
    shape[sn1++] = ll[0][0] * ll[1][i];               // Edge 3-0
    }
  sn = sn1; // Advance to the end of all edge DOFs.

  for (i = 1; i < order[1]; ++i)
    {
    for (j = 1; j < order[0]; ++j)
      {
      //cout << sn << "\n";
      shape[sn++ ] = ll[0][j] * ll[1][i];        // Face 0-1-2-3
      }
    }
}

// Quadrilateral shape-function derivatives
void vtkLagrangeInterpolation::Tensor2ShapeDerivatives(const int order[2], const double pcoords[2], double* deriv)
{
  // FIXME: Eventually needs to be varying length.
  double ll[2][vtkLagrangeInterpolation::MaxDegree + 1];
  double dd[2][vtkLagrangeInterpolation::MaxDegree + 1];
  int i, j;

  for (i = 0; i < 2; ++i)
    {
    vtkLagrangeInterpolation::EvaluateShapeAndGradient(
      order[i], pcoords[i], ll[i], dd[i]);
    }

  int sn = 0;

  // Corners
  deriv[sn++] = dd[0][0]        * ll[1][0];
  deriv[sn++] = ll[0][0]        * dd[1][0];

  deriv[sn++] = dd[0][order[0]] * ll[1][0];
  deriv[sn++] = ll[0][order[0]] * dd[1][0];

  deriv[sn++] = dd[0][order[0]] * ll[1][order[1]];
  deriv[sn++] = ll[0][order[0]] * dd[1][order[1]];

  deriv[sn++] = dd[0][0]        * ll[1][order[1]];
  deriv[sn++] = ll[0][0]        * dd[1][order[1]];

  int sn1 = sn + 2 * (order[0] + order[1] - 2);
  for (i = 1; i < order[0]; ++i)
    {
    //cout << sn << ", " << sn1 << "\n";
    deriv[sn++ ] = dd[0][i] * ll[1][0];               // Edge 0-1
    deriv[sn++ ] = ll[0][i] * dd[1][0];               // Edge 0-1

    deriv[sn1++] = dd[0][i] * ll[1][order[1]];        // Edge 2-3
    deriv[sn1++] = ll[0][i] * dd[1][order[1]];        // Edge 2-3
    }

  for (i = 1; i < order[1]; ++i)
    {
    //cout << sn << ", " << sn1 << "\n";
    deriv[sn++ ] = dd[0][order[0]] * ll[1][i];        // Edge 1-2
    deriv[sn++ ] = ll[0][order[0]] * dd[1][i];        // Edge 1-2

    deriv[sn1++] = dd[0][0] * ll[1][i];               // Edge 3-0
    deriv[sn1++] = ll[0][0] * dd[1][i];               // Edge 3-0
    }
  sn = sn1;
  for (i = 1; i < order[1]; ++i)
    {
    for (j = 1; j < order[0]; ++j)
      {
      //cout << sn << "\n";
      deriv[sn++ ] = dd[0][j] * ll[1][i];        // Face 0-1-2-3
      deriv[sn++ ] = ll[0][j] * dd[1][i];        // Face 0-1-2-3
      }
    }
}

/// Hexahedral shape function computation
void vtkLagrangeInterpolation::Tensor3ShapeFunctions(const int order[3], const double pcoords[3], double* shape)
{
  // FIXME: Eventually needs to be varying length.
  double ll[3][vtkLagrangeInterpolation::MaxDegree + 1];
  int i,j,k;

  for (i = 0; i < 3; ++i)
    {
    vtkLagrangeInterpolation::EvaluateShapeFunctions(order[i], pcoords[i], ll[i]);
    }

  int sn = 0;

  // Corners
  shape[sn++] = ll[0][0]        * ll[1][0]        * ll[2][0];
  shape[sn++] = ll[0][order[0]] * ll[1][0]        * ll[2][0];
  shape[sn++] = ll[0][order[0]] * ll[1][order[1]] * ll[2][0];
  shape[sn++] = ll[0][0]        * ll[1][order[1]] * ll[2][0];
  shape[sn++] = ll[0][0]        * ll[1][0]        * ll[2][order[2]];
  shape[sn++] = ll[0][order[0]] * ll[1][0]        * ll[2][order[2]];
  shape[sn++] = ll[0][order[0]] * ll[1][order[1]] * ll[2][order[2]];
  shape[sn++] = ll[0][0]        * ll[1][order[1]] * ll[2][order[2]];

  int sn1, sn2, sn3;
  sn1 = order[0] + order[1] - 2;
  sn2 = sn1 * 2;
  sn3 = sn + sn1 + sn2;
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[0]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    shape[sn++ ] = ll[0][i] * ll[1][0] * ll[2][0];               // Edge 0-1
    shape[sn1++] = ll[0][i] * ll[1][order[1]] * ll[2][0];        // Edge 2-3
    shape[sn2++] = ll[0][i] * ll[1][0] * ll[2][order[2]];        // Edge 4-5
    shape[sn3++] = ll[0][i] * ll[1][order[1]] * ll[2][order[2]]; // Edge 6-7
    }

  for (i = 1; i < order[1]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    shape[sn++ ] = ll[0][order[0]] * ll[1][i] * ll[2][0];        // Edge 1-2
    shape[sn1++] = ll[0][0] * ll[1][i] * ll[2][0];               // Edge 3-0
    shape[sn2++] = ll[0][order[0]] * ll[1][i] * ll[2][order[2]]; // Edge 5-6
    shape[sn3++] = ll[0][0] * ll[1][i] * ll[2][order[2]];        // Edge 7-4
    }
  sn = sn3;
  sn1 = order[2] - 1;
  sn2 = sn1 * 2;
  sn3 = sn + sn1 + sn2;
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[2]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    shape[sn++ ] = ll[0][0] * ll[1][0] * ll[2][i];                // Edge 0-4
    shape[sn1++] = ll[0][order[0]] * ll[1][0] * ll[2][i];         // Edge 1-5
    // Kitware insists on swapping edges 10 and 11 as follows:
    shape[sn3++] = ll[0][order[0]] * ll[1][order[1]] * ll[2][i];  // Edge 2-6
    shape[sn2++] = ll[0][0] * ll[1][order[1]] * ll[2][i];         // Edge 3-7
    }

  sn = sn3;
  sn1 = (order[1] - 1)*(order[2] - 1);
  sn2 = sn1 * 2;
  sn3 = sn + sn2 + (order[2] - 1)*(order[0] - 1);
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[2]; ++i)
    {
    for (j = 1; j < order[1]; ++j)
      {
      //cout << sn << ", " << sn1 << "\n";
      shape[sn++ ] = ll[0][0] * ll[1][j] * ll[2][i];        // Face 0-4-7-3
      shape[sn1++] = ll[0][order[0]] * ll[1][j] * ll[2][i]; // Face 1-2-6-5
      }
    for (j = 1; j < order[0]; ++j)
      {
      //cout << sn2 << ", " << sn3 << "\n";
      shape[sn2++] = ll[0][j] * ll[1][0] * ll[2][i];        // Face 0-1-5-4
      shape[sn3++] = ll[0][j] * ll[1][order[1]] * ll[2][i]; // Face 2-3-7-6
      }
    }
  sn = sn3;
  sn1 = sn + (order[0] - 1)*(order[1] - 1);
  for (i = 1; i < order[1]; ++i)
    {
    for (j = 1; j < order[0]; ++j)
      {
      //cout << sn << ", " << sn1 << "\n";
      shape[sn++ ] = ll[0][j] * ll[1][i] * ll[2][0];        // Face 0-1-2-3
      shape[sn1++] = ll[0][j] * ll[1][i] * ll[2][order[2]]; // Face 4-7-6-5
      }
    }
  sn = sn1;
  for (k = 1; k < order[2]; ++k)
    {
    for (j = 1; j < order[1]; ++j)
      {
      for (i = 1; i < order[0]; ++i)
        {
        //cout << sn << "\n";
        shape[sn++] = ll[0][i] * ll[1][j] * ll[2][k]; // Body
        }
      }
    }
}

void vtkLagrangeInterpolation::Tensor3ShapeDerivatives(const int order[2], const double pcoords[3], double* deriv)
{
  // FIXME: Eventually needs to be varying length.
  double ll[3][vtkLagrangeInterpolation::MaxDegree + 1];
  double dd[3][vtkLagrangeInterpolation::MaxDegree + 1];
  int i, j, k;

  for (i = 0; i < 3; ++i)
    {
    vtkLagrangeInterpolation::EvaluateShapeAndGradient(
      order[i], pcoords[i], ll[i], dd[i]);
    }

  int sn = 0;

  // Corners
  deriv[sn++] = dd[0][0]        * ll[1][0]        * ll[2][0];
  deriv[sn++] = ll[0][0]        * dd[1][0]        * ll[2][0];
  deriv[sn++] = ll[0][0]        * ll[1][0]        * dd[2][0];

  deriv[sn++] = dd[0][order[0]] * ll[1][0]        * ll[2][0];
  deriv[sn++] = ll[0][order[0]] * dd[1][0]        * ll[2][0];
  deriv[sn++] = ll[0][order[0]] * ll[1][0]        * dd[2][0];

  deriv[sn++] = dd[0][order[0]] * ll[1][order[1]] * ll[2][0];
  deriv[sn++] = ll[0][order[0]] * dd[1][order[1]] * ll[2][0];
  deriv[sn++] = ll[0][order[0]] * ll[1][order[1]] * dd[2][0];

  deriv[sn++] = dd[0][0]        * ll[1][order[1]] * ll[2][0];
  deriv[sn++] = ll[0][0]        * dd[1][order[1]] * ll[2][0];
  deriv[sn++] = ll[0][0]        * ll[1][order[1]] * dd[2][0];

  deriv[sn++] = dd[0][0]        * ll[1][0]        * ll[2][order[2]];
  deriv[sn++] = ll[0][0]        * dd[1][0]        * ll[2][order[2]];
  deriv[sn++] = ll[0][0]        * ll[1][0]        * dd[2][order[2]];

  deriv[sn++] = dd[0][order[0]] * ll[1][0]        * ll[2][order[2]];
  deriv[sn++] = ll[0][order[0]] * dd[1][0]        * ll[2][order[2]];
  deriv[sn++] = ll[0][order[0]] * ll[1][0]        * dd[2][order[2]];

  deriv[sn++] = dd[0][order[0]] * ll[1][order[1]] * ll[2][order[2]];
  deriv[sn++] = ll[0][order[0]] * dd[1][order[1]] * ll[2][order[2]];
  deriv[sn++] = ll[0][order[0]] * ll[1][order[1]] * dd[2][order[2]];

  deriv[sn++] = dd[0][0]        * ll[1][order[1]] * ll[2][order[2]];
  deriv[sn++] = ll[0][0]        * dd[1][order[1]] * ll[2][order[2]];
  deriv[sn++] = ll[0][0]        * ll[1][order[1]] * dd[2][order[2]];


  int sn1, sn2, sn3;
  sn1 = 3 * (order[0] + order[1] - 2);
  sn2 = sn1 * 2;
  sn3 = sn + sn1 + sn2;
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[0]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    deriv[sn++ ] = dd[0][i] * ll[1][0] * ll[2][0];               // Edge 0-1
    deriv[sn++ ] = ll[0][i] * dd[1][0] * ll[2][0];               // Edge 0-1
    deriv[sn++ ] = ll[0][i] * ll[1][0] * dd[2][0];               // Edge 0-1

    deriv[sn1++] = dd[0][i] * ll[1][order[1]] * ll[2][0];        // Edge 2-3
    deriv[sn1++] = ll[0][i] * dd[1][order[1]] * ll[2][0];        // Edge 2-3
    deriv[sn1++] = ll[0][i] * ll[1][order[1]] * dd[2][0];        // Edge 2-3

    deriv[sn2++] = dd[0][i] * ll[1][0] * ll[2][order[2]];        // Edge 4-5
    deriv[sn2++] = ll[0][i] * dd[1][0] * ll[2][order[2]];        // Edge 4-5
    deriv[sn2++] = ll[0][i] * ll[1][0] * dd[2][order[2]];        // Edge 4-5

    deriv[sn3++] = dd[0][i] * ll[1][order[1]] * ll[2][order[2]]; // Edge 6-7
    deriv[sn3++] = ll[0][i] * dd[1][order[1]] * ll[2][order[2]]; // Edge 6-7
    deriv[sn3++] = ll[0][i] * ll[1][order[1]] * dd[2][order[2]]; // Edge 6-7
    }

  for (i = 1; i < order[1]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    deriv[sn++ ] = dd[0][order[0]] * ll[1][i] * ll[2][0];        // Edge 1-2
    deriv[sn++ ] = ll[0][order[0]] * dd[1][i] * ll[2][0];        // Edge 1-2
    deriv[sn++ ] = ll[0][order[0]] * ll[1][i] * dd[2][0];        // Edge 1-2

    deriv[sn1++] = dd[0][0] * ll[1][i] * ll[2][0];               // Edge 3-0
    deriv[sn1++] = ll[0][0] * dd[1][i] * ll[2][0];               // Edge 3-0
    deriv[sn1++] = ll[0][0] * ll[1][i] * dd[2][0];               // Edge 3-0

    deriv[sn2++] = dd[0][order[0]] * ll[1][i] * ll[2][order[2]]; // Edge 5-6
    deriv[sn2++] = ll[0][order[0]] * dd[1][i] * ll[2][order[2]]; // Edge 5-6
    deriv[sn2++] = ll[0][order[0]] * ll[1][i] * dd[2][order[2]]; // Edge 5-6

    deriv[sn3++] = dd[0][0] * ll[1][i] * ll[2][order[2]];        // Edge 7-4
    deriv[sn3++] = ll[0][0] * dd[1][i] * ll[2][order[2]];        // Edge 7-4
    deriv[sn3++] = ll[0][0] * ll[1][i] * dd[2][order[2]];        // Edge 7-4

    }
  sn = sn3;
  sn1 = 3 * (order[2] - 1);
  sn2 = sn1 * 2;
  sn3 = sn + sn1 + sn2;
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[2]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    deriv[sn++ ] = dd[0][0] * ll[1][0] * ll[2][i];                // Edge 0-4
    deriv[sn++ ] = ll[0][0] * dd[1][0] * ll[2][i];                // Edge 0-4
    deriv[sn++ ] = ll[0][0] * ll[1][0] * dd[2][i];                // Edge 0-4

    deriv[sn1++] = dd[0][order[0]] * ll[1][0] * ll[2][i];         // Edge 1-5
    deriv[sn1++] = ll[0][order[0]] * dd[1][0] * ll[2][i];         // Edge 1-5
    deriv[sn1++] = ll[0][order[0]] * ll[1][0] * dd[2][i];         // Edge 1-5

    // Kitware insists on swapping edges 10 and 11 as follows:
    deriv[sn3++] = dd[0][order[0]] * ll[1][order[1]] * ll[2][i];  // Edge 2-6
    deriv[sn3++] = ll[0][order[0]] * dd[1][order[1]] * ll[2][i];  // Edge 2-6
    deriv[sn3++] = ll[0][order[0]] * ll[1][order[1]] * dd[2][i];  // Edge 2-6

    deriv[sn2++] = dd[0][0] * ll[1][order[1]] * ll[2][i];         // Edge 3-7
    deriv[sn2++] = ll[0][0] * dd[1][order[1]] * ll[2][i];         // Edge 3-7
    deriv[sn2++] = ll[0][0] * ll[1][order[1]] * dd[2][i];         // Edge 3-7
    }

  sn = sn3;
  sn1 = 3 * (order[1] - 1) * (order[2] - 1);
  sn2 = sn1 * 2;
  sn3 = sn + sn2 + 3 * (order[2] - 1)*(order[0] - 1);
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[2]; ++i)
    {
    for (j = 1; j < order[1]; ++j)
      {
      //cout << sn << ", " << sn1 << "\n";
      deriv[sn++ ] = dd[0][0] * ll[1][j] * ll[2][i];        // Face 0-4-7-3
      deriv[sn++ ] = ll[0][0] * dd[1][j] * ll[2][i];        // Face 0-4-7-3
      deriv[sn++ ] = ll[0][0] * ll[1][j] * dd[2][i];        // Face 0-4-7-3

      deriv[sn1++] = dd[0][order[0]] * ll[1][j] * ll[2][i]; // Face 1-2-6-5
      deriv[sn1++] = ll[0][order[0]] * dd[1][j] * ll[2][i]; // Face 1-2-6-5
      deriv[sn1++] = ll[0][order[0]] * ll[1][j] * dd[2][i]; // Face 1-2-6-5
      }
    for (j = 1; j < order[0]; ++j)
      {
      //cout << sn2 << ", " << sn3 << "\n";
      deriv[sn2++] = dd[0][j] * ll[1][0] * ll[2][i];        // Face 0-1-5-4
      deriv[sn2++] = ll[0][j] * dd[1][0] * ll[2][i];        // Face 0-1-5-4
      deriv[sn2++] = ll[0][j] * ll[1][0] * dd[2][i];        // Face 0-1-5-4

      deriv[sn3++] = dd[0][j] * ll[1][order[1]] * ll[2][i]; // Face 2-3-7-6
      deriv[sn3++] = ll[0][j] * dd[1][order[1]] * ll[2][i]; // Face 2-3-7-6
      deriv[sn3++] = ll[0][j] * ll[1][order[1]] * dd[2][i]; // Face 2-3-7-6
      }
    }
  sn = sn3;
  sn1 = sn + 3 * (order[0] - 1) * (order[1] - 1);
  for (i = 1; i < order[1]; ++i)
    {
    for (j = 1; j < order[0]; ++j)
      {
      //cout << sn << ", " << sn1 << "\n";
      deriv[sn++ ] = dd[0][j] * ll[1][i] * ll[2][0];        // Face 0-1-2-3
      deriv[sn++ ] = ll[0][j] * dd[1][i] * ll[2][0];        // Face 0-1-2-3
      deriv[sn++ ] = ll[0][j] * ll[1][i] * dd[2][0];        // Face 0-1-2-3

      deriv[sn1++] = dd[0][j] * ll[1][i] * ll[2][order[2]]; // Face 4-7-6-5
      deriv[sn1++] = ll[0][j] * dd[1][i] * ll[2][order[2]]; // Face 4-7-6-5
      deriv[sn1++] = ll[0][j] * ll[1][i] * dd[2][order[2]]; // Face 4-7-6-5
      }
    }
  sn = sn1;
  for (k = 1; k < order[2]; ++k)
    {
    for (j = 1; j < order[1]; ++j)
      {
      for (i = 1; i < order[0]; ++i)
        {
        //cout << sn << "\n";
        deriv[sn++] = dd[0][i] * ll[1][j] * ll[2][k]; // Body
        deriv[sn++] = ll[0][i] * dd[1][j] * ll[2][k]; // Body
        deriv[sn++] = ll[0][i] * ll[1][j] * dd[2][k]; // Body
        }
      }
    }
}

/// Wedge shape function computation
void vtkLagrangeInterpolation::WedgeShapeFunctions(const int order[3], const double pcoords[3], double* shape)
{
  static vtkNew<vtkLagrangeTriangle> tri;

  if (order[0] != order[1])
    {
    vtkGenericWarningMacro(
      "Orders 0 and 1 (parametric coordinates of triangle, " << order[0] << " and " << order[1] << ") must match.");
    return;
    }

  int rsOrder = order[0];
  int tOrder = order[2];
  if (
    rsOrder > vtkLagrangeInterpolation::MaxDegree ||
    tOrder > vtkLagrangeInterpolation::MaxDegree)
    {
    vtkGenericWarningMacro(
      "vtkLagrangeInterpolation::MaxDegree exceeded by "
      << order[0] << ", " << order[1] << ", " << order[2]);
    return;
    }

  // FIXME: Eventually needs to be varying length.
  double ll[vtkLagrangeInterpolation::MaxDegree + 1];
  double tt[(vtkLagrangeInterpolation::MaxDegree + 1) * (vtkLagrangeInterpolation::MaxDegree + 2) / 2];
  vtkLagrangeInterpolation::EvaluateShapeFunctions(tOrder, pcoords[2], ll);
  vtkVector3d triP(pcoords);
  triP[2] = 0;
  tri->GetPoints()->SetNumberOfPoints((rsOrder + 1) * (rsOrder + 2) / 2);
  tri->Initialize();
  tri->InterpolateFunctions(triP.GetData(), tt);

  // I. Corner nodes
  int sn = 0;
  int sn1 = sn + 3;
  int sn2;
  int delta;
  for (int i = 0; i < 3; ++i, ++sn, ++sn1)
    {
    shape[sn] = ll[0] * tt[i]; // "Bottom" face
    shape[sn1] = ll[tOrder] * tt[i]; // "Top" face
    }
  sn = sn1; // Advance past top face nodes

  // II. Edge nodes
  //     A. Triangle edges
  delta = 3 * (rsOrder - 1);
  sn1 = sn + delta;
  int tOff = 3; // Offset into tt: skip past corners to edge-DOFs.
  for (int i = 0; i < delta; ++i, ++sn, ++sn1, ++tOff)
    {
    shape[sn]  = ll[0]       * tt[tOff];
    shape[sn1] = ll[tOrder]  * tt[tOff];
    }
  //     B. Vertical (quadrilateral) edges
  sn = sn1;
  for (int i = 1; i < tOrder; ++i)
    {
    for (int j = 0; j < 3; ++j, ++sn)
      {
      shape[sn] = ll[i] * tt[j];
      }
    }

  // III. Face nodes
  //      A. Triangular faces
  sn1 = sn + (rsOrder - 2) * (rsOrder - 1) / 2;
  for (int i = 1; i < rsOrder; ++i)
    {
    for (int j = 1; j <= i; ++j, ++sn, ++tOff)
      {
      shape[sn]  = ll[0]      * tt[tOff];
      shape[sn1] = ll[tOrder] * tt[tOff];
      }
    }
  sn = sn1;
  //      B. Quadrilateral faces
  delta = (rsOrder - 1) * (tOrder - 1);
  sn1 = sn + delta;
  sn2 = sn + 2 * delta;
  for (int i = 1; i < tOrder; ++i)
    {
    tOff = 3; // Start at edge DOFs
    int tOff1 = tOff + (rsOrder - 1);
    int tOff2 = tOff + (rsOrder - 1) * 2;
    for (int j = 1; j < rsOrder; ++j, ++sn, ++sn1, ++sn2, ++tOff, ++tOff1, ++tOff2)
      {
      shape[sn]  = ll[j] * tt[tOff];
      shape[sn1] = ll[j] * tt[tOff1];
      shape[sn2] = ll[j] * tt[tOff2];
      }
    }
  sn = sn2;

  // IV. Volume DOFs
  for (int k = 1; k < tOrder; ++k)
    {
    tOff = 3 + 3 * (rsOrder - 1); // Skip past corners and edges
    for (int j = 1; j < rsOrder; ++j)
      {
      for (int i = 1; i <= j; ++i, ++tOff)
        {
        shape[sn] = ll[k] * tt[tOff];
        }
      }
    }
}

/// Wedge shape-function derivative evaluation
void vtkLagrangeInterpolation::WedgeShapeDerivatives(const int order[2], const double pcoords[3], double* deriv)
{
  // FIXME: Eventually needs to be varying length.
  /*
  double ll[3][vtkLagrangeInterpolation::MaxDegree + 1];
  double dd[3][vtkLagrangeInterpolation::MaxDegree + 1];
  int i, j, k;

  for (i = 0; i < 3; ++i)
    {
    vtkLagrangeInterpolation::EvaluateShapeAndGradient(
      order[i], pcoords[i], ll[i], dd[i]);
    }

  int sn = 0;

  // Corners
  deriv[sn++] = dd[0][0]        * ll[1][0]        * ll[2][0];
  deriv[sn++] = ll[0][0]        * dd[1][0]        * ll[2][0];
  deriv[sn++] = ll[0][0]        * ll[1][0]        * dd[2][0];

  deriv[sn++] = dd[0][order[0]] * ll[1][0]        * ll[2][0];
  deriv[sn++] = ll[0][order[0]] * dd[1][0]        * ll[2][0];
  deriv[sn++] = ll[0][order[0]] * ll[1][0]        * dd[2][0];

  deriv[sn++] = dd[0][order[0]] * ll[1][order[1]] * ll[2][0];
  deriv[sn++] = ll[0][order[0]] * dd[1][order[1]] * ll[2][0];
  deriv[sn++] = ll[0][order[0]] * ll[1][order[1]] * dd[2][0];

  deriv[sn++] = dd[0][0]        * ll[1][order[1]] * ll[2][0];
  deriv[sn++] = ll[0][0]        * dd[1][order[1]] * ll[2][0];
  deriv[sn++] = ll[0][0]        * ll[1][order[1]] * dd[2][0];

  deriv[sn++] = dd[0][0]        * ll[1][0]        * ll[2][order[2]];
  deriv[sn++] = ll[0][0]        * dd[1][0]        * ll[2][order[2]];
  deriv[sn++] = ll[0][0]        * ll[1][0]        * dd[2][order[2]];

  deriv[sn++] = dd[0][order[0]] * ll[1][0]        * ll[2][order[2]];
  deriv[sn++] = ll[0][order[0]] * dd[1][0]        * ll[2][order[2]];
  deriv[sn++] = ll[0][order[0]] * ll[1][0]        * dd[2][order[2]];

  deriv[sn++] = dd[0][order[0]] * ll[1][order[1]] * ll[2][order[2]];
  deriv[sn++] = ll[0][order[0]] * dd[1][order[1]] * ll[2][order[2]];
  deriv[sn++] = ll[0][order[0]] * ll[1][order[1]] * dd[2][order[2]];

  deriv[sn++] = dd[0][0]        * ll[1][order[1]] * ll[2][order[2]];
  deriv[sn++] = ll[0][0]        * dd[1][order[1]] * ll[2][order[2]];
  deriv[sn++] = ll[0][0]        * ll[1][order[1]] * dd[2][order[2]];


  int sn1, sn2, sn3;
  sn1 = 3 * (order[0] + order[1] - 2);
  sn2 = sn1 * 2;
  sn3 = sn + sn1 + sn2;
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[0]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    deriv[sn++ ] = dd[0][i] * ll[1][0] * ll[2][0];               // Edge 0-1
    deriv[sn++ ] = ll[0][i] * dd[1][0] * ll[2][0];               // Edge 0-1
    deriv[sn++ ] = ll[0][i] * ll[1][0] * dd[2][0];               // Edge 0-1

    deriv[sn1++] = dd[0][i] * ll[1][order[1]] * ll[2][0];        // Edge 2-3
    deriv[sn1++] = ll[0][i] * dd[1][order[1]] * ll[2][0];        // Edge 2-3
    deriv[sn1++] = ll[0][i] * ll[1][order[1]] * dd[2][0];        // Edge 2-3

    deriv[sn2++] = dd[0][i] * ll[1][0] * ll[2][order[2]];        // Edge 4-5
    deriv[sn2++] = ll[0][i] * dd[1][0] * ll[2][order[2]];        // Edge 4-5
    deriv[sn2++] = ll[0][i] * ll[1][0] * dd[2][order[2]];        // Edge 4-5

    deriv[sn3++] = dd[0][i] * ll[1][order[1]] * ll[2][order[2]]; // Edge 6-7
    deriv[sn3++] = ll[0][i] * dd[1][order[1]] * ll[2][order[2]]; // Edge 6-7
    deriv[sn3++] = ll[0][i] * ll[1][order[1]] * dd[2][order[2]]; // Edge 6-7
    }

  for (i = 1; i < order[1]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    deriv[sn++ ] = dd[0][order[0]] * ll[1][i] * ll[2][0];        // Edge 1-2
    deriv[sn++ ] = ll[0][order[0]] * dd[1][i] * ll[2][0];        // Edge 1-2
    deriv[sn++ ] = ll[0][order[0]] * ll[1][i] * dd[2][0];        // Edge 1-2

    deriv[sn1++] = dd[0][0] * ll[1][i] * ll[2][0];               // Edge 3-0
    deriv[sn1++] = ll[0][0] * dd[1][i] * ll[2][0];               // Edge 3-0
    deriv[sn1++] = ll[0][0] * ll[1][i] * dd[2][0];               // Edge 3-0

    deriv[sn2++] = dd[0][order[0]] * ll[1][i] * ll[2][order[2]]; // Edge 5-6
    deriv[sn2++] = ll[0][order[0]] * dd[1][i] * ll[2][order[2]]; // Edge 5-6
    deriv[sn2++] = ll[0][order[0]] * ll[1][i] * dd[2][order[2]]; // Edge 5-6

    deriv[sn3++] = dd[0][0] * ll[1][i] * ll[2][order[2]];        // Edge 7-4
    deriv[sn3++] = ll[0][0] * dd[1][i] * ll[2][order[2]];        // Edge 7-4
    deriv[sn3++] = ll[0][0] * ll[1][i] * dd[2][order[2]];        // Edge 7-4

    }
  sn = sn3;
  sn1 = 3 * (order[2] - 1);
  sn2 = sn1 * 2;
  sn3 = sn + sn1 + sn2;
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[2]; ++i)
    {
    //cout << sn << ", " << sn1 << ", " << sn2 << ", " << sn3 << "\n";
    deriv[sn++ ] = dd[0][0] * ll[1][0] * ll[2][i];                // Edge 0-4
    deriv[sn++ ] = ll[0][0] * dd[1][0] * ll[2][i];                // Edge 0-4
    deriv[sn++ ] = ll[0][0] * ll[1][0] * dd[2][i];                // Edge 0-4

    deriv[sn1++] = dd[0][order[0]] * ll[1][0] * ll[2][i];         // Edge 1-5
    deriv[sn1++] = ll[0][order[0]] * dd[1][0] * ll[2][i];         // Edge 1-5
    deriv[sn1++] = ll[0][order[0]] * ll[1][0] * dd[2][i];         // Edge 1-5

    // Kitware insists on swapping edges 10 and 11 as follows:
    deriv[sn3++] = dd[0][order[0]] * ll[1][order[1]] * ll[2][i];  // Edge 2-6
    deriv[sn3++] = ll[0][order[0]] * dd[1][order[1]] * ll[2][i];  // Edge 2-6
    deriv[sn3++] = ll[0][order[0]] * ll[1][order[1]] * dd[2][i];  // Edge 2-6

    deriv[sn2++] = dd[0][0] * ll[1][order[1]] * ll[2][i];         // Edge 3-7
    deriv[sn2++] = ll[0][0] * dd[1][order[1]] * ll[2][i];         // Edge 3-7
    deriv[sn2++] = ll[0][0] * ll[1][order[1]] * dd[2][i];         // Edge 3-7
    }

  sn = sn3;
  sn1 = 3 * (order[1] - 1) * (order[2] - 1);
  sn2 = sn1 * 2;
  sn3 = sn + sn2 + 3 * (order[2] - 1)*(order[0] - 1);
  sn1 += sn;
  sn2 += sn;
  for (i = 1; i < order[2]; ++i)
    {
    for (j = 1; j < order[1]; ++j)
      {
      //cout << sn << ", " << sn1 << "\n";
      deriv[sn++ ] = dd[0][0] * ll[1][j] * ll[2][i];        // Face 0-4-7-3
      deriv[sn++ ] = ll[0][0] * dd[1][j] * ll[2][i];        // Face 0-4-7-3
      deriv[sn++ ] = ll[0][0] * ll[1][j] * dd[2][i];        // Face 0-4-7-3

      deriv[sn1++] = dd[0][order[0]] * ll[1][j] * ll[2][i]; // Face 1-2-6-5
      deriv[sn1++] = ll[0][order[0]] * dd[1][j] * ll[2][i]; // Face 1-2-6-5
      deriv[sn1++] = ll[0][order[0]] * ll[1][j] * dd[2][i]; // Face 1-2-6-5
      }
    for (j = 1; j < order[0]; ++j)
      {
      //cout << sn2 << ", " << sn3 << "\n";
      deriv[sn2++] = dd[0][j] * ll[1][0] * ll[2][i];        // Face 0-1-5-4
      deriv[sn2++] = ll[0][j] * dd[1][0] * ll[2][i];        // Face 0-1-5-4
      deriv[sn2++] = ll[0][j] * ll[1][0] * dd[2][i];        // Face 0-1-5-4

      deriv[sn3++] = dd[0][j] * ll[1][order[1]] * ll[2][i]; // Face 2-3-7-6
      deriv[sn3++] = ll[0][j] * dd[1][order[1]] * ll[2][i]; // Face 2-3-7-6
      deriv[sn3++] = ll[0][j] * ll[1][order[1]] * dd[2][i]; // Face 2-3-7-6
      }
    }
  sn = sn3;
  sn1 = sn + 3 * (order[0] - 1) * (order[1] - 1);
  for (i = 1; i < order[1]; ++i)
    {
    for (j = 1; j < order[0]; ++j)
      {
      //cout << sn << ", " << sn1 << "\n";
      deriv[sn++ ] = dd[0][j] * ll[1][i] * ll[2][0];        // Face 0-1-2-3
      deriv[sn++ ] = ll[0][j] * dd[1][i] * ll[2][0];        // Face 0-1-2-3
      deriv[sn++ ] = ll[0][j] * ll[1][i] * dd[2][0];        // Face 0-1-2-3

      deriv[sn1++] = dd[0][j] * ll[1][i] * ll[2][order[2]]; // Face 4-7-6-5
      deriv[sn1++] = ll[0][j] * dd[1][i] * ll[2][order[2]]; // Face 4-7-6-5
      deriv[sn1++] = ll[0][j] * ll[1][i] * dd[2][order[2]]; // Face 4-7-6-5
      }
    }
  sn = sn1;
  for (k = 1; k < order[2]; ++k)
    {
    for (j = 1; j < order[1]; ++j)
      {
      for (i = 1; i < order[0]; ++i)
        {
        //cout << sn << "\n";
        deriv[sn++] = dd[0][i] * ll[1][j] * ll[2][k]; // Body
        deriv[sn++] = ll[0][i] * dd[1][j] * ll[2][k]; // Body
        deriv[sn++] = ll[0][i] * ll[1][j] * dd[2][k]; // Body
        }
      }
    }
  */
}

template<>
void vtkLagrangeInterpolation::TensorShapeFunctions<3>(const int order[3], const double* pcoords, double* shape)
{
  vtkLagrangeInterpolation::Tensor3ShapeFunctions(order, pcoords, shape);
}

template<>
void vtkLagrangeInterpolation::TensorShapeFunctions<2>(const int order[2], const double* pcoords, double* shape)
{
  vtkLagrangeInterpolation::Tensor2ShapeFunctions(order, pcoords, shape);
}

template<>
void vtkLagrangeInterpolation::TensorShapeFunctions<1>(const int order[1], const double* pcoords, double* shape)
{
  vtkLagrangeInterpolation::EvaluateShapeFunctions(order[0], pcoords[0], shape);
}

template<>
void vtkLagrangeInterpolation::TensorShapeDerivatives<3>(const int order[3], const double* pcoords, double* deriv)
{
  vtkLagrangeInterpolation::Tensor3ShapeDerivatives(order, pcoords, deriv);
}

template<>
void vtkLagrangeInterpolation::TensorShapeDerivatives<2>(const int order[2], const double* pcoords, double* deriv)
{
  vtkLagrangeInterpolation::Tensor2ShapeDerivatives(order, pcoords, deriv);
}

template<>
void vtkLagrangeInterpolation::TensorShapeDerivatives<1>(const int order[1], const double* pcoords, double* deriv)
{
  std::vector<double> dummy(order[0] + 1);
  vtkLagrangeInterpolation::EvaluateShapeAndGradient(order[0], pcoords[0], &dummy[0], deriv);
}

vtkVector3d vtkLagrangeInterpolation::GetParametricHexCoordinates(int vertexId)
{
  return vtkVector3d(hexCorner[vertexId]);
}

vtkVector2i vtkLagrangeInterpolation::GetPointIndicesBoundingHexEdge(int edgeId)
{
  return vtkVector2i(hexEdgeCorners[edgeId][0], hexEdgeCorners[edgeId][1]);
}

int vtkLagrangeInterpolation::GetVaryingParameterOfHexEdge(int edgeId)
{
  return hexEdgeCorners[edgeId][2];
}

vtkVector2i vtkLagrangeInterpolation::GetFixedParametersOfHexEdge(int edgeId)
{
  return vtkVector2i(hexEdgeCorners[edgeId][3], hexEdgeCorners[edgeId][4]);
}

const int* vtkLagrangeInterpolation::GetPointIndicesBoundingHexFace(int faceId)
{
  return hexFaceCorners[faceId];
}

const int* vtkLagrangeInterpolation::GetEdgeIndicesBoundingHexFace(int faceId)
{
  return hexFaceEdges[faceId];
}

vtkVector2i vtkLagrangeInterpolation::GetVaryingParametersOfHexFace(int faceId)
{
  return vtkVector2i(hexFaceCorners[faceId][4], hexFaceCorners[faceId][5]);
}

int vtkLagrangeInterpolation::GetFixedParameterOfHexFace(int faceId)
{
  return hexFaceCorners[faceId][6];
}

vtkVector3d vtkLagrangeInterpolation::GetParametricWedgeCoordinates(int vertexId)
{
  return vtkVector3d(wedgeCorner[vertexId]);
}

vtkVector2i vtkLagrangeInterpolation::GetPointIndicesBoundingWedgeEdge(int edgeId)
{
  return vtkVector2i(wedgeEdgeCorners[edgeId][0], wedgeEdgeCorners[edgeId][1]);
}

int vtkLagrangeInterpolation::GetVaryingParameterOfWedgeEdge(int edgeId)
{
  return wedgeEdgeCorners[edgeId][2];
}

vtkVector2i vtkLagrangeInterpolation::GetFixedParametersOfWedgeEdge(int edgeId)
{
  return vtkVector2i(wedgeEdgeCorners[edgeId][3], wedgeEdgeCorners[edgeId][4]);
}

const int* vtkLagrangeInterpolation::GetPointIndicesBoundingWedgeFace(int faceId)
{
  return wedgeFaceCorners[faceId];
}

/// Return 4 edge ids bounding face (with -1 as last id for triangles) plus a face orientation as the 5th number.
const int* vtkLagrangeInterpolation::GetEdgeIndicesBoundingWedgeFace(int faceId)
{
  return wedgeFaceEdges[faceId];
}

vtkVector2i vtkLagrangeInterpolation::GetVaryingParametersOfWedgeFace(int faceId)
{
  return vtkVector2i(wedgeFaceCorners[faceId][4], wedgeFaceCorners[faceId][5]);
}

int vtkLagrangeInterpolation::GetFixedParameterOfWedgeFace(int faceId)
{
  return wedgeFaceCorners[faceId][6];
}

void vtkLagrangeInterpolation::AppendCurveCollocationPoints(vtkSmartPointer<vtkPoints>& pts, const int order[1])
{
  if (!pts)
    {
    pts = vtkSmartPointer<vtkPoints>::New();
    }

  vtkIdType existing = pts->GetNumberOfPoints();
  vtkIdType np = order[0] + 1;
  pts->SetNumberOfPoints(existing + np);
  vtkVector3d e0( 0., 0., 0.);
  vtkVector3d e1(+1., 0., 0.);

  // Insert corner points
  vtkIdType sn = existing;
  pts->SetPoint(sn++, e0.GetData());
  pts->SetPoint(sn++, e1.GetData());

  // Insert edge points
  for (int ii = 1; ii < order[0]; ++ii)
    {
    pts->SetPoint(sn++,
      ii / static_cast<double>(order[0]), 0.0, 0.0 // Force quad to z = 0 plane
    );
    }
}

void vtkLagrangeInterpolation::AppendQuadrilateralCollocationPoints(vtkSmartPointer<vtkPoints>& pts, const int order[2])
{
  if (!pts)
    {
    pts = vtkSmartPointer<vtkPoints>::New();
    }

  vtkIdType existing = pts->GetNumberOfPoints();
  vtkIdType np = (order[0] + 1) * (order[1] + 1);
  pts->SetNumberOfPoints(existing + np);
  // Insert corner points
  vtkIdType sn = existing;
  for (int ii = 0; ii < 4; ++ii)
    {
    vtkVector3d cc(hexCorner[ii]);
    cc[2] = 0.0; // Force quad to z = 0 plane
    pts->SetPoint(sn++, cc.GetData());
    }

  // Insert edge points
  for (int ii = 0; ii < 4; ++ii)
    {
    vtkVector3d e0(hexCorner[hexEdgeCorners[ii][0]]);
    vtkVector3d e1(hexCorner[hexEdgeCorners[ii][1]]);
    for (int jj = 1; jj < order[hexEdgeCorners[ii][2]]; ++jj)
      {
      double rr = jj / static_cast<double>(order[hexEdgeCorners[ii][2]]);
      vtkVector3d vv = ((1. - rr) * e0 + rr * e1);
      vv[2] = 0.0; // Force quad to z = 0 plane
      pts->SetPoint(sn++, vv.GetData());
      }
    }

  // Insert face points
  for (int jj = 1; jj < order[1]; ++jj)
    {
    for (int ii = 1; ii < order[0]; ++ii)
      {
      pts->SetPoint(sn++,
        ii / static_cast<double>(order[0]),
        jj / static_cast<double>(order[1]),
        0.0 // Force quad to z = 0 plane
        );
      }
    }
}

void vtkLagrangeInterpolation::AppendHexahedronCollocationPoints(vtkSmartPointer<vtkPoints>& pts, const int order[3])
{
  if (!pts)
    {
    pts = vtkSmartPointer<vtkPoints>::New();
    }

  vtkIdType existing = pts->GetNumberOfPoints();
  vtkIdType np = (order[0] + 1) * (order[1] + 1) * (order[2] + 1);
  pts->SetNumberOfPoints(existing + np);
  // Insert corner points
  vtkIdType sn = existing;
  for (int ii = 0; ii < 8; ++ii)
    {
    pts->SetPoint(sn++, hexCorner[ii]);
    }

  // Insert edge points
  for (unsigned ii = 0; ii < sizeof(hexEdgeCorners) / sizeof(hexEdgeCorners[0]); ++ii)
    {
    vtkVector3d e0(hexCorner[hexEdgeCorners[ii][0]]);
    vtkVector3d e1(hexCorner[hexEdgeCorners[ii][1]]);
    for (int jj = 1; jj < order[hexEdgeCorners[ii][2]]; ++jj)
      {
      double rr = jj / static_cast<double>(order[hexEdgeCorners[ii][2]]);
      vtkVector3d vv = ((1. - rr) * e0 + rr * e1);
      pts->SetPoint(sn++, vv.GetData());
      }
    }

  // Insert face points
  for (unsigned kk = 0; kk < sizeof(hexFaceCorners) / sizeof(hexFaceCorners[0]); ++kk)
    {
    vtkVector3d f0(hexCorner[hexFaceCorners[kk][0]]);
    vtkVector3d f1(hexCorner[hexFaceCorners[kk][1]]);
    vtkVector3d f2(hexCorner[hexFaceCorners[kk][2]]);
    vtkVector3d f3(hexCorner[hexFaceCorners[kk][3]]);
    for (int jj = 1; jj < order[hexFaceCorners[kk][5]]; ++jj)
      {
      double ss = jj / static_cast<double>(order[hexFaceCorners[kk][5]]);
      for (int ii = 1; ii < order[hexFaceCorners[kk][4]]; ++ii)
        {
        double rr = ii / static_cast<double>(order[hexFaceCorners[kk][4]]);
        vtkVector3d vv =
          (1. - ss) * ((1. - rr) * f0 + rr * f1) +
          ss        * ((1. - rr) * f3 + rr * f2);
        pts->SetPoint(sn++, vv.GetData());
        }
      }
    }

  // Insert body points
  for (int kk = 1; kk < order[2]; ++kk)
    {
    for (int jj = 1; jj < order[1]; ++jj)
      {
      for (int ii = 1; ii < order[0]; ++ii)
        {
        pts->SetPoint(sn++,
          ii / static_cast<double>(order[0]),
          jj / static_cast<double>(order[1]),
          kk / static_cast<double>(order[2]));
        }
      }
    }
}

void vtkLagrangeInterpolation::AppendWedgeCollocationPoints(vtkSmartPointer<vtkPoints>& pts, const int order[3])
{
  if (!pts)
    {
    pts = vtkSmartPointer<vtkPoints>::New();
    }

  vtkIdType existing = pts->GetNumberOfPoints();
  vtkIdType np = (order[0] + 1) * (order[1] + 2) * (order[2] + 1) / 2; // NB: assert(order[0] == order[1])
  pts->SetNumberOfPoints(existing + np);
  // Insert corner points
  vtkIdType sn = existing;
  for (int ii = 0; ii < 6; ++ii)
    {
    pts->SetPoint(sn++, wedgeCorner[ii]);
    }

  int rsOrder = order[0]; // assert(order[0] == order[1])
  int tOrder = order[2];

  // Insert edge points
  for (unsigned ii = 0; ii < sizeof(wedgeEdgeCorners) / sizeof(wedgeEdgeCorners[0]); ++ii)
    {
    vtkVector3d e0(wedgeCorner[wedgeEdgeCorners[ii][0]]);
    vtkVector3d e1(wedgeCorner[wedgeEdgeCorners[ii][1]]);
    int varyingParam = wedgeEdgeCorners[ii][2];
    int edgeOrder = varyingParam >= 0 ? order[varyingParam] : rsOrder;
    for (unsigned jj = 1; jj < edgeOrder; ++jj)
      {
      double rr = jj / static_cast<double>(edgeOrder);
      vtkVector3d vv = ((1. - rr) * e0 + rr * e1);
      pts->SetPoint(sn++, vv.GetData());
      }
    }

  // Insert face points
  unsigned kk;
  for (kk = 0; kk < 2; ++kk)
    { // Triangular faces
    vtkVector3d f0(wedgeCorner[wedgeFaceCorners[kk][0]]);
    vtkVector3d f1(wedgeCorner[wedgeFaceCorners[kk][1]]);
    // Note funky f3/f2 numbering here matches quadrilateral/hex code
    // where points are in CCW loop:
    vtkVector3d f3(wedgeCorner[wedgeFaceCorners[kk][2]]);
    vtkVector3d f2 = f0 + (f1 - f0) + (f3 - f0);

    for (int jj = 1; jj < rsOrder; ++jj)
      {
      double ss = jj / static_cast<double>(rsOrder);
      for (int ii = 1; ii < rsOrder - jj; ++ii)
        {
        double rr = ii / static_cast<double>(rsOrder);
        vtkVector3d vv =
          (1. - ss) * ((1. - rr) * f0 + rr * f1) +
          ss        * ((1. - rr) * f3 + rr * f2);
        pts->SetPoint(sn++, vv.GetData());
        }
      }
    }

  for (; kk < sizeof(wedgeFaceCorners) / sizeof(wedgeFaceCorners[0]); ++kk)
    { // Quadrilateral faces
    vtkVector3d f0(wedgeCorner[wedgeFaceCorners[kk][0]]);
    vtkVector3d f1(wedgeCorner[wedgeFaceCorners[kk][1]]);
    vtkVector3d f2(wedgeCorner[wedgeFaceCorners[kk][2]]);
    vtkVector3d f3(wedgeCorner[wedgeFaceCorners[kk][3]]);

    for (int jj = 1; jj < tOrder; ++jj)
      {
      double ss = jj / static_cast<double>(tOrder);
      for (int ii = 1; ii < rsOrder; ++ii)
        {
        double rr = ii / static_cast<double>(rsOrder);
        vtkVector3d vv =
          (1. - ss) * ((1. - rr) * f0 + rr * f1) +
          ss        * ((1. - rr) * f3 + rr * f2);
        pts->SetPoint(sn++, vv.GetData());
        }
      }
    }

  // Insert body points
  for (int kk = 1; kk < tOrder; ++kk)
    {
    for (int jj = 1; jj < rsOrder; ++jj)
      {
      for (int ii = 1; ii < rsOrder - jj; ++ii)
        {
        pts->SetPoint(sn++,
          ii / static_cast<double>(rsOrder),
          jj / static_cast<double>(rsOrder),
          kk / static_cast<double>(tOrder));
        }
      }
    }
}

void vtkLagrangeInterpolation::AppendWedgeCollocationPoints(vtkPoints* pts, int o1, int o2, int o3)
{
  int o[3] = {o1, o2, o3};
  vtkSmartPointer<vtkPoints> spts(pts);
  vtkLagrangeInterpolation::AppendWedgeCollocationPoints(spts, o);
}

void vtkLagrangeInterpolation::PrepareForOrder(const int order[3])
{
  // Ensure some scratch space is allocated for templated evaluation methods.
  std::size_t maxShape = (order[0] + 1) * (order[1] + 1) * (order[2] + 1);
  std::size_t maxDeriv = maxShape * 3;
  if (this->ShapeSpace.size() < maxShape)
    {
    this->ShapeSpace.resize(maxShape);
    }
  if (this->DerivSpace.size() < maxDeriv)
    {
    this->DerivSpace.resize(maxDeriv);
    }
}
