
import os
import os.path
from vtk import *
from vtk.test import Testing

class LagrangeGeometricOperations(Testing.vtkTest):
    def setUp(self):
        self.rw = vtkRenderWindow()
        self.rr = vtkRenderer()
        self.ri = vtkRenderWindowInteractor()
        self.rw.AddRenderer(self.rr)
        self.rw.SetInteractor(self.ri)
        self.ri.Initialize()
        self.rs = self.ri.GetInteractorStyle()
        self.rs.SetCurrentStyleToTrackballCamera()
        self.rs.SetCurrentStyleToMultiTouchCamera()
        self.rr.SetBackground(1,1,1)

        self.rdr = vtkXMLUnstructuredGridReader()
        self.rdr.SetFileName(self.pathToData('Elements.vtu'))
        print 'Reading ', self.rdr.GetFileName()
        self.rdr.Update()

    def addToScene(self, filt):
        ac = vtkActor()
        mp = vtkDataSetMapper()
        mp.SetInputConnection(filt.GetOutputPort())
        ac.SetMapper(mp)
        self.rr.AddActor(ac)

        return (ac, mp)

    def testContour(self):
        ## Contour actor
        con = vtkContourFilter()
        con.SetInputConnection(self.rdr.GetOutputPort())
        con.SetInputArrayToProcess(0,0,0, vtkDataSet.FIELD_ASSOCIATION_POINTS_THEN_CELLS, 'Ellipsoid')
        con.SetComputeNormals(1)
        con.SetComputeScalars(1)
        con.SetComputeGradients(1)
        con.SetNumberOfContours(4)
        con.SetValue(0, 2.5)
        con.SetValue(1, 1.5)
        con.SetValue(2, 0.5)
        con.SetValue(3, 1.05)
        con.Update()

        # Add the contour to the scene:
        a1, m1 = self.addToScene(con)
        clr = vtkColorSeries()
        lkup = vtkLookupTable()
        # Color the contours with a qualitative color scheme:
        clr.SetColorScheme(vtkColorSeries.BREWER_QUALITATIVE_DARK2)
        clr.BuildLookupTable(lkup, vtkColorSeries.CATEGORICAL);
        lkup.SetAnnotation(vtkVariant(0.5), 'Really Low')
        lkup.SetAnnotation(vtkVariant(1.05), 'Somewhat Low')
        lkup.SetAnnotation(vtkVariant(1.5), 'Medium')
        lkup.SetAnnotation(vtkVariant(2.5), 'High')
        m1.SelectColorArray('Ellipsoid')
        m1.SetLookupTable(lkup)

        ## Surface actor
        dss = vtkDataSetSurfaceFilter()
        dss.SetInputConnection(self.rdr.GetOutputPort())
        dss.SetNonlinearSubdivisionLevel(3)

        nrm = vtkPolyDataNormals()
        nrm.SetInputConnection(dss.GetOutputPort())

        a2, m2 = self.addToScene(nrm)
        a2.GetProperty().SetOpacity(0.2)

        self.ri.Initialize()
        cam = self.rr.GetActiveCamera()
        cam.SetPosition(12.9377265875, 6.5914481094, 7.54647854482)
        cam.SetFocalPoint(4.38052401617, 0.925973308028, 1.91021697659)
        cam.SetViewUp(-0.491867406412, -0.115590747077, 0.862963054655)

        ## Other nice viewpoints:
        # cam.SetPosition(-1.53194314907, -6.07277748432, 19.283152654)
        # cam.SetFocalPoint(4.0, 2.25, 2.25)
        # cam.SetViewUp(0.605781341771, 0.619386648223, 0.499388772365)
        #
        # cam.SetPosition(10.5925480421, -3.08988382244, 9.2072891403)
        # cam.SetFocalPoint(4.0, 2.25, 2.25)
        # cam.SetViewUp(-0.384040517561, 0.519961374525, 0.762989547683)

        self.rw.SetSize(512, 512)
        self.rw.Render()
        image = 'LagrangeGeometricOperations-Contour.png'
        events = self.prepareTestImage(self.ri, filename=os.path.join('/tmp', image))
        #vtk.test.Testing.compareImage(self.rw, self.pathToValidatedOutput(image))
        #vtk.test.Testing.interact()

        ## Write the contours out
        # wri = vtkXMLPolyDataWriter()
        # wri.SetInputConnection(con.GetOutputPort())
        # wri.SetFileName('/tmp/contours.vtp')
        # wri.Write()
        ## Write extracted surface out

    def testBoundaryExtraction(self):
        ugg = vtkUnstructuredGridGeometryFilter()
        ugg.SetInputConnection(self.rdr.GetOutputPort())

        wri = vtkXMLUnstructuredGridWriter()
        wri.SetInputConnection(ugg.GetOutputPort())
        wri.SetDataModeToAscii()
        wri.SetFileName('/tmp/surface.vtu')
        wri.Write()

    def testClip(self):

        # Color the cells with a qualitative color scheme:
        clr = vtkColorSeries()
        lkup = vtkLookupTable()
        clr.SetColorScheme(vtkColorSeries.BREWER_QUALITATIVE_DARK2)
        clr.BuildLookupTable(lkup, vtkColorSeries.CATEGORICAL);
        lkup.SetAnnotation(vtkVariant(0), 'First cell')
        lkup.SetAnnotation(vtkVariant(1), 'Second cell')

        ## Clip
        pln = vtkPlane()
        pln.SetOrigin(4, 2, 2)
        pln.SetNormal(-0.28735, -0.67728, 0.67728)
        clp = vtkClipDataSet()
        clp.SetInputConnection(self.rdr.GetOutputPort())
        clp.SetClipFunction(pln)
        # clp.InsideOutOn()
        #clp.GenerateClipScalarsOn()
        clp.Update()

        # Add the clipped data to the scene:
        a1, m1 = self.addToScene(clp)
        m1.SetScalarModeToUseCellFieldData()
        m1.SelectColorArray('SrcCellNum')
        m1.SetLookupTable(lkup)

        ## Surface actor
        dss = vtkDataSetSurfaceFilter()
        dss.SetInputConnection(self.rdr.GetOutputPort())
        dss.SetNonlinearSubdivisionLevel(3)

        nrm = vtkPolyDataNormals()
        nrm.SetInputConnection(dss.GetOutputPort())

        a2, m2 = self.addToScene(nrm)
        a2.GetProperty().SetOpacity(0.2)
        m2.SetScalarModeToUseCellFieldData()
        m2.SelectColorArray('SrcCellNum')
        m2.SetLookupTable(lkup)

        self.ri.Initialize()
        cam = self.rr.GetActiveCamera()
        cam.SetPosition(16.0784261776, 11.8079343039, -6.69074553411)
        cam.SetFocalPoint(4.54685488135, 1.74152986486, 2.38091647662)
        cam.SetViewUp(-0.523934540522, 0.81705750638, 0.240644194852)
        self.rw.SetSize(512, 512)
        self.rw.Render()
        image = 'LagrangeGeometricOperations-Clip.png'
        events = self.prepareTestImage(self.ri, filename=os.path.join('/tmp', image))
        #vtk.test.Testing.compareImage(self.rw, self.pathToValidatedOutput(image))
        #vtk.test.Testing.interact()

        #ri.Start()

    def testCut(self):

        # Color the cells with a qualitative color scheme:
        clr = vtkColorSeries()
        lkup = vtkLookupTable()
        clr.SetColorScheme(vtkColorSeries.BREWER_QUALITATIVE_DARK2)
        clr.BuildLookupTable(lkup, vtkColorSeries.CATEGORICAL);
        lkup.SetAnnotation(vtkVariant(0), 'First cell')
        lkup.SetAnnotation(vtkVariant(1), 'Second cell')

        ## Cuts
        pln = vtkPlane()
        pln.SetOrigin(4, 2, 2)
        pln.SetNormal(-0.28735, -0.67728, 0.67728)
        cut = vtkCutter()
        cut.SetInputConnection(self.rdr.GetOutputPort())
        cut.SetCutFunction(pln)
        cut.Update()

        # Add the cut to the scene:
        a1, m1 = self.addToScene(cut)
        m1.SetScalarModeToUseCellFieldData()
        m1.SelectColorArray('SrcCellNum')
        m1.SetLookupTable(lkup)

        ## Surface actor
        dss = vtkDataSetSurfaceFilter()
        dss.SetInputConnection(self.rdr.GetOutputPort())
        dss.SetNonlinearSubdivisionLevel(3)

        nrm = vtkPolyDataNormals()
        nrm.SetInputConnection(dss.GetOutputPort())

        a2, m2 = self.addToScene(nrm)
        a2.GetProperty().SetOpacity(0.2)
        # m2.SetScalarModeToUseCellFieldData()
        # m2.SelectColorArray('SrcCellNum')
        # m2.SetLookupTable(lkup)

        self.ri.Initialize()
        cam = self.rr.GetActiveCamera()
        cam.SetPosition(16.0784261776, 11.8079343039, -6.69074553411)
        cam.SetFocalPoint(4.54685488135, 1.74152986486, 2.38091647662)
        cam.SetViewUp(-0.523934540522, 0.81705750638, 0.240644194852)
        self.rw.SetSize(512, 512)
        self.rw.Render()
        image = 'LagrangeGeometricOperations-Cut.png'
        events = self.prepareTestImage(self.ri, filename=os.path.join('/tmp', image))
        #vtk.test.Testing.compareImage(self.rw, self.pathToValidatedOutput(image))
        #vtk.test.Testing.interact()

    def testIntersectWithLine(self):
        import numpy as np
        rn = vtkMinimalStandardRandomSequence()
        ## Choose some random lines and intersect them with our cells
        def rnums(N, vmin, vmax):
            result = []
            delta = vmax - vmin
            for i in range(N):
                result.append(rn.GetValue() * delta + vmin)
                rn.Next()
            return result
        # p1 = zip(rnums(10, -4,  8), rnums(10, -4, 4), rnums(10, -4, 4))
        # p2 = zip(rnums(10,  0, 12), rnums(10,  0, 8), rnums(10,  0, 8))
        p1 = [ \
            (-4, 2, 2), (2, -4, 2), (2, 2, -4), (0.125, 0.125, 4.125), (8.125, 0.125, 4.125), \
            (0.125, 0.125, 0.125), (7.875, 3.875, 3.875), \
            ]
        p2 = [ \
            (12, 2, 2), (2,  8, 2), (2, 2,  8), (3.45,  0.125, 4.125), (3.65,  0.125, 4.125), \
            (4.8, 4.3, 4.3), (3.3, -0.5, -0.5),
            ]
        lca = vtkCellArray()
        lpt = vtkPoints()
        nli = len(p1)
        [lpt.InsertNextPoint(x) for x in p1]
        [lpt.InsertNextPoint(x) for x in p2]
        [lca.InsertNextCell(2, [i, nli + i]) for i in range(nli)]
        lpd = vtkPolyData()
        lpd.SetPoints(lpt)
        lpd.SetLines(lca)
        tub = vtkTubeFilter()
        tub.SetInputDataObject(0, lpd)
        tub.SetVaryRadiusToVaryRadiusOff()
        tub.SetRadius(0.025)
        tub.SetCapping(1)
        tub.SetNumberOfSides(16)
        al, ml = self.addToScene(tub)
        al.GetProperty().SetColor(0.5, 0.5, 0.5)

        ug = self.rdr.GetOutputDataObject(0)
        tt = vtk.mutable(0)
        subId = vtk.mutable(-1)
        xx = [0,0,0]
        rr = [0,0,0]
        ipt = vtkPoints()
        ica = vtkCellArray()
        for cidx in range(ug.GetNumberOfCells()):
            cell = ug.GetCell(cidx)
            order = [cell.GetOrder(i) for i in range(cell.GetCellDimension())]
            npts = reduce(lambda x, y: x * (y + 1), order)
            weights = np.zeros((npts,1));
            print 'Cell {:1}'.format(cidx)
            for pp in range(len(p1)):
                print '  Line {p1} -- {p2}'.format(p1=p1[pp], p2=p2[pp])
                result = cell.IntersectWithLine(p1[pp], p2[pp], 1e-8, tt, xx, rr, subId)
                print '  Hit: {hit}  @{t} posn {posn} param {rr} subId {subId}'.format( \
                    hit=result, t=tt.get(), posn=xx, rr=rr, subId=subId.get())
                if result:
                    pid = [ipt.InsertNextPoint(xx)]
                    ica.InsertNextCell(1, pid)

        ipd = vtkPolyData()
        ipd.SetPoints(ipt)
        ipd.SetVerts(ica)
        print '{N} vertices to glyph'.format(N=ipd.GetNumberOfCells())
        gly = vtkGlyph3D()
        ssc = vtkSphereSource()
        gly.SetSourceConnection(ssc.GetOutputPort())
        gly.SetInputDataObject(0, ipd)
        gly.SetScaleFactor(0.15)
        ai, mi = self.addToScene(gly)
        ai.GetProperty().SetColor(0.8, 0.3, 0.3)

        ## Surface actor
        dss = vtkDataSetSurfaceFilter()
        dss.SetInputConnection(self.rdr.GetOutputPort())
        dss.SetNonlinearSubdivisionLevel(3)

        nrm = vtkPolyDataNormals()
        nrm.SetInputConnection(dss.GetOutputPort())

        a2, m2 = self.addToScene(nrm)
        a2.GetProperty().SetOpacity(0.2)

        ## Render test scene
        self.ri.Initialize()
        self.rw.SetSize(512, 512)
        cam = self.rr.GetActiveCamera()
        cam.SetPosition(4.14824823557, -15.3201939164, 7.48529277914)
        cam.SetFocalPoint(4.0392921746, 2.25197875899, 1.59174422348)
        cam.SetViewUp(-0.00880943634729, 0.317921564576, 0.948076090095)

        self.rw.Render()
        image = 'LagrangeGeometricOperations-Stab.png'
        events = self.prepareTestImage(self.ri, filename=os.path.join('/tmp', image))

if __name__ == "__main__":
    Testing.main([(LagrangeGeometricOperations, 'test')])
